"""
Functions to analyze Python scripts to find classes usable as Icy plugins.

Copyright 2013 Institut Pasteur.
 
This file is part of ICY.
 
ICY is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.
 
ICY is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
 
You should have received a copy of the GNU General Public License
along with ICY. If not, see <http://www.gnu.org/licenses/>.

Created on Tue Nov 14

@author: tlecomte
Inspired from https://github.com/jimbaker/jython-storm-starter/blob/master/clamp/__init__.py (for serialization and package name),
and from the MiniClampMaker.java test in Jython (for Java annotations).
"""

import site
import os
import sys
import inspect
import importlib

from icy.plugin.abstract_ import Plugin

from plugins.tlecomte.pythonExtractor import PythonLibraries

class ClassProperty:
    """
    Bring together the various properties associated with a class
    that tell if that class can be used as a plugin in Icy.
    """
    def __init__(self, klass, moduleName, dirName):
        
        self.klass = klass
        self.moduleName = moduleName
        self.dirName = dirName
        
        self.isPlugin = False
        self.isAnnotated = False
        self.packageName = ""
        self.instance = None

        try:
            self.isPlugin = issubclass(klass, Plugin)
        except TypeError:
            # klass is not a class
            return
        
        if not self.isPlugin:
            return
                       
        try:
            self.instance = klass()
        except TypeError:
            # klass cannot be instantiated
            return

        self.isAnnotated = hasPythonLibrariesAnnotation(self.instance)

        self.packageName = getPackageName(self.instance)


    def print_report(self):
        if self.isValid():
            if self.isAnnotated:
                print "%s is a valid Plugin class and has a \"PythonLibraries\" annotation." %(self.klass)
            else:
                print "%s is a valid Plugin class but does not have a \"PythonLibraries\" annotation." %(self.klass)           
        else:
            print "%s is not a valid Plugin class." %(self.klass)
            
            if not self.isPlugin:
                print "%s is not a subclass of icy.plugin.abstract_.Plugin." %(self.klass)
            elif self.instance is None:
                print "%s cannot be instantiated." %(self.klass)           
            elif not isValidPackageName(self.packageName):
                print "%s does not a proper package name starting with \"plugins.\"" %(self.klass)
        

    def isValid(self):
        """
        Tells if this class can be used as an Icy plugin.
        """
        return (self.isPlugin
                and self.instance is not None
                and isValidPackageName(self.packageName))


    def __repr__(self):
        return self.getQualifiedName()


    def getQualifiedName(self):
        """
        Returns the fullly-qualified name of this class.
        
        For example: plugins.tlecomte.<modulename>.<classname>
        """
        return ".".join((self.packageName, self.moduleName, self.klass.__name__))


def analyze_module_classes(mod, moduleName, dirName):
    """
    Build a list of classes contained in a module.
    The module name and the directory name are needed to declare the ClassProperty elements of the list
    """
    classes = [] # list of ClassProperty
    
    for attr in dir(mod):
        klass = getattr(mod, attr)
        thisClass = ClassProperty(klass, moduleName, dirName)
        classes.append(thisClass)
               
    return classes


def hasPythonLibrariesAnnotation(instance):
    """
    Tests if an object is annotated with 'plugins.tlecomte.pythonExtractor.PythonLibraries'
    """
    try:
        isPythonLibraries =  PythonLibraries in instance.__clamp_class_annotations__
    except AttributeError:
        return False
    else:
        if not isPythonLibraries:
            return False
        else:
            return True


def isValidPackageName(packageName):
    """
    Tests if a package name can be understood by Icy as the package of a plugin.
    
    This means that the package name starts with '.plugins'
    """
    splitPackage = packageName.split('.')
    
    if splitPackage[0] <> "plugins":
        return False
    else:
        return True


def getPackageName(instance):
    """
    Returns the package name for the given object.
    
    The first parts of the package name are defined in a class definition by a line like:
        __proxymaker__ = PackageProxy("plugins.tlecomte")
    and it is completed by the module name and the class name.   
    """
    try:
        package = instance.__proxymaker__.package
    except AttributeError:
        return ""
    else:
        return package


# a valid script must contain a class:
# - deriving from icy.plugin.abstract_.Plugin
# - annotated with plugins.tlecomte.pythonExtractor.PythonLibraries (so that Icy will properly extract the Python scripts)
# - assigned to a package name beginning with "plugins."

def analyze_script(fileName):
    """
    For a given file, look at all the attributes that it contains,
    and build a list of classes from them.
    """
    dirName, baseName = os.path.split(fileName)

    if dirName not in sys.path:
       sys.path.insert(0, dirName)
    
    moduleInfo = inspect.getmoduleinfo(fileName)   
      
    mod = importlib.import_module(moduleInfo.name)
    
    classes = analyze_module_classes(mod, moduleInfo.name, dirName)
       
    return classes


def analyse_folder(folderName):
    """
    Look at all the files in a given folder and build a list of classes that they contain.
    """
    folderClasses = []
    
    for f in os.listdir(folderName):
        root, ext = os.path.splitext(f)
        if ext == ".py":
            scriptClasses = analyze_script(os.path.join(folderName, f))
            folderClasses += scriptClasses

    return folderClasses


def validateClasses(classes):
    """
    Walk the class list, select the classes that define valid Plugins.
    
    These classes:
     - derive from icy.plugin.abstract_.Plugin,
     - are not abstract (i.e. can be instantiated),
     - are assigned to a package name beginning with "plugins.".
    """
    
    validClasses = []
    
    # make a list of the valid classes
    for classProperty in classes:
        if classProperty.isValid():
            validClasses.append(classProperty)

    return validClasses


def countAnnotations(classes):
    """
    Walk the class list, count the number of classes that have
    the plugins.tlecomte.pythonExtractor.PythonLibraries annotation,
    and return that number.
    """
    
    count = 0
    
    for classProperty in classes:
        if classProperty.isAnnotated:
            count += 1

    return count
