package plugins.fab.trackgenerator;

/***
 * generates images of Gaussian-like objects, either symmetrical if sigmaX=sigmaY, 
 * or microtubule-like (elongated commet-like) objects, if sigmaX > sigmaY 
 */

import java.util.ArrayList;

import flanagan.analysis.Stat;
import flanagan.math.PsRandom;

public class ImageCreatorMicrotubuleLikeObjects {

	private ArrayList<Particle> allParticles;
	private ArrayList<VolumeImage> volumeImageList;
	private double[][][][] floatImage;
	private int width;
	private int height;
	private int depth;
	private int sequenceLength;
	private double pixelSize;
	private double sliceSpacing; 
	private int psfWidth;
	private int psfDepth;
	private double psfMax;
	private PsRandom ran;
	private double bg = 10;  // background level
	private double alphaSNR; // coefficient for making the desired SNR 
	private double sigmaX, sigmaY, sigmaZ;
	
	ImageCreatorMicrotubuleLikeObjects(ArrayList<Particle> allParticles, int width, int height, int depth, int sequenceLength, double sigmaX, double sigmaY, double sigmaZ, PsRandom ran, double pixelSize, double sliceSpacing, double SNR)
	{
		this.width = width;
		this.height = height;
		this.depth = depth;
		this.sequenceLength = sequenceLength;
		this.ran = ran;
		this.pixelSize = pixelSize;
		this.sliceSpacing = sliceSpacing;
	
		this.sigmaX = sigmaX;
		this.sigmaY = sigmaY;
		this.sigmaZ = sigmaZ;
		
//		Particle temp = allParticles.get(0);
//		temp.detectionArrayList.clear();
//		double dt = pixelSize / sliceSpacing;
//		temp.detectionArrayList.add(new TGDetection(100, 100, 1, 0));
//		temp.detectionArrayList.add(new TGDetection(110, 110, 2, 1));
//		temp.detectionArrayList.add(new TGDetection(120, 120, 2.5, 2));
//		temp.detectionArrayList.add(new TGDetection(130, 130, 3, 3));
//		temp.detectionArrayList.add(new TGDetection(140, 140, 5, 4));
//		allParticles.clear();
//		allParticles.add(temp);

		this.allParticles = allParticles;
		
		psfWidth = (int)Math.round(7 * sigmaX);
		psfWidth = (psfWidth % 2 == 0) ? psfWidth + 1 : psfWidth; 
		psfDepth = (int)Math.round(7 * sigmaZ);
		psfDepth = (psfDepth % 2 == 0) ? psfDepth + 1 : psfDepth;
		
		// the max value for PSF (for normalization and SNR computation)
		psfMax = 1; 
		alphaSNR = (Math.pow(SNR, 2) + SNR * Math.sqrt(Math.pow(SNR, 2) + 4 * bg)) / 2.0;
		
		// double valued image for placing the PSF and adding the noise
		floatImage = new double[sequenceLength][depth][height][width];  

		// final 8bit image for the output
		volumeImageList = new ArrayList<VolumeImage>();
		
		//RANDOMIZE BACKGROUND LEVEL FOR THE TEST DATA, SO PEOPLE CANNOT FIX THAT VALUE FROM THE VERY BEGINNING
		bg += 10 * ran.nextDouble();
		
	}
	
	public ArrayList<VolumeImage> getImage()
	{
		double Vx, Vy, Vz;
		
		for(Particle p : allParticles)
		{
			for (int j = 0; j < p.detectionArrayList.size(); j++) 
			{
				TGDetection detection =  p.detectionArrayList.get(j);
					
				int x = (int)Math.round(detection.x);
				int y = (int)Math.round(detection.y);
				int z = (int)Math.round(detection.z);
				
				if (j == 0){
					Vx = p.detectionArrayList.get(1).x - detection.x; 
					Vy = p.detectionArrayList.get(1).y - detection.y;
					Vz = p.detectionArrayList.get(1).z - detection.z;
				} else if (j == p.detectionArrayList.size() - 1) {
					Vx = detection.x - p.detectionArrayList.get(j - 1).x; 
					Vy = detection.y - p.detectionArrayList.get(j - 1).y;
					Vz = detection.z - p.detectionArrayList.get(j - 1).z;
				} else {
					Vx = (p.detectionArrayList.get(j + 1).x - p.detectionArrayList.get(j - 1).x) * 0.5; 
					Vy = (p.detectionArrayList.get(j + 1).y - p.detectionArrayList.get(j - 1).y) * 0.5;
					Vz = (p.detectionArrayList.get(j + 1).z - p.detectionArrayList.get(j - 1).z) * 0.5;
				}
		
				double alpha = Math.atan2(Vy, Vx);
				
				double dx = detection.x - x;
				double dy = detection.y - y;
				double dz = detection.z - z;

				for (int zz = 0; zz < depth; zz++) 
				{
					if (x >= 0 && x < width && y >= 0 && y < height) 
					{
						for (int jj = 0; jj < psfWidth; jj++) 
						{
							for (int ii = 0; ii < psfWidth; ii++) 
							{
								int xi = - psfWidth / 2 + ii;
								int yi = - psfWidth / 2 + jj; 
								if (xi + x >= 0 && xi + x < width && yi + y>= 0 && yi + y< height) 
								{
									// rotate coordinates 
									double xx =  (xi - dx) * Math.cos(alpha) + (yi - dy) * Math.sin(alpha);
									double yy = -(xi - dx) * Math.sin(alpha) + (yi - dy) * Math.cos(alpha);
																		
									floatImage[detection.t][zz][yi + y][xi + x] += (xx < 0) ? 
											Math.exp(-(z - zz + dz) * (z - zz + dz) / (2 * sigmaZ * sigmaZ)) * 
											Math.exp(-(xx * xx) / (2 * sigmaX * sigmaX) 
											-(yy * yy) / (2 * sigmaY * sigmaY)) / psfMax * alphaSNR : 
											Math.exp(-(z - zz + dz) * (z - zz + dz) / (2 * sigmaZ * sigmaZ)) * 
											Math.exp(-(xx * xx) / (2 * sigmaY * sigmaY) 
											-(yy * yy) / (2 * sigmaY * sigmaY)) / psfMax * alphaSNR;
								}			
							}
						}						
					}
				}
			}			
		}
		
		// add background and apply noise
		for (int tt = 0; tt < sequenceLength; tt++) {
			VolumeImage volumeImage = new VolumeImage();
			volumeImage.imageData = new byte [depth][width * height];
			for (int zz = 0; zz < depth; zz++) {
				for (int jj = 0; jj < height; jj++) {
					for (int ii = 0; ii < width; ii++) {
						floatImage[tt][zz][jj][ii] += bg; 
						
						//add noise
						floatImage[tt][zz][jj][ii] = PoissonNoiseGenerator.nextPoissonian(floatImage[tt][zz][jj][ii]);

						// convert the original float image to byte, taking into account pixel SATURATION
						volumeImage.imageData[zz][ii + jj * width] = (floatImage[tt][zz][jj][ii] > 255) ? 
								(byte)255 : (byte)floatImage[tt][zz][jj][ii];
					}
				}
			}
			volumeImageList.add(volumeImage);
		}
		return volumeImageList;
	}   
}
