package plugins.tlecomte.contourPlot;

/*
 * General purpose contour tracer for quadrilateral meshes.
 * Handles single level contours, or region between a pair of levels.
 * 
 * This contouring code is a java port of the Matplotlib contour
 * code, which itself comes from gctrn.c, part of the GIST package.
 * 
 * To the best of my knowledge, the license of this code is
 * compatible with GPL, since it is used by Matplotlib and Chaco.   
 */

/* Note that all arrays in these routines are Fortran-style,
 * in the sense that the "i" index varies fastest; the dimensions
 * of the corresponding C array are z[jmax][imax] in the notation
 * used here. We can identify i and j with the x and y dimensions,
 * respectively.
 */

/* What is a contour?
 *
 * Given a quadrilateral mesh (x,y), and values of a z at the points
 * of that mesh, we seek a set of polylines connecting points at a
 * particular value of z. Each point on such a contour curve lies
 * on an edge of the mesh, at a point linearly interpolated to the
 * contour level z0 between the given values of z at the endpoints
 * of the edge.
 *
 * Identifying these points is easy. Figuring out how to connect them
 * into a curve -- or possibly a set of disjoint curves -- is difficult.
 * Each disjoint curve may be either a closed circuit, or it may begin
 * and end on a mesh boundary.
 *
 * One of the problems with a quadrilateral mesh is that when the z
 * values at one pair of diagonally opposite points lie below z0, and
 * the values at the other diagonal pair of the same zone lie above z0,
 * all four edges of the zone are cut, and there is an ambiguity in
 * how we should connect the points. I call this a saddle zone.
 * The problem is that two disjoint curves cut through a saddle zone
 * (I reject the alternative of connecting the opposite points to make
 * a single self-intersecting curve, since those make ugly contour plots
 * -- I've tried it). The solution is to determine the z value of the
 * centre of the zone, which is the mean of the z values of the four
 * corner points. If the centre z is higher than the contour level of
 * interest and you are moving aint the line with higher values on the
 * left, turn right to leave the saddle zone. If the centre z is lower
 * than the contour level turn left. Whether the centre z is higher
 * than the 1 or 2 contour levels is stored in the saddle array so that
 * it does not need to be recalculated in subsequent passes.
 *
 * Another complicating factor is that there may be logical holes in
 * the mesh -- zones which do not exist. We want our contours to stop
 * if they hit the edge of such a zone, just as if they'd hit the edge
 * of the whole mesh. The input region array addresses this issue.
 *
 * Yet another complication: We may want a list of closed polygons which
 * outline the region between two contour levels z0 and z1. These may
 * include sections of the mesh boundary (including edges of logical
 * holes defined by the region array), in addition to sections of the
 * contour curves at one or both levels. This introduces a huge
 * topological problem -- if one of the closed contours (possibly
 * including an interior logical hole in the mesh, but not any part of
 * the boundary of the whole mesh) encloses a region which is not
 * between z0 and z1, that curve must be connected by a slit (or "branch
 * cut") to the enclosing curve, so that the list of disjoint polygons
 * we return is each simply connected.
 *
 * Okay, one final stunning difficulty: For the two level case, no
 * individual polygon should have more than a few thousand sides, since
 * huge filled polygons place an inordinate load on rendering software,
 * which needs an amount of scratch space proportional to the number
 * of sides it needs to fill. So in the two level case, we want to
 * chunk the mesh into rectangular pieces of no more than, say, 30x30
 * zones, which keeps each returned polygon to less than a few thousand
 * sides (the worst case is very very bad -- you can easily write down
 * a function and two level values which produce a polygon that cuts
 * every edge of the mesh twice).
 */

/*
 * Here is the numbering scheme for points, edges, and zones in
 * the mesh -- note that each ij corresponds to one point, one zone,
 * one i-edge (i=constant edge) and one j-edge (j=constant edge):
 *
 * (ij-1)-------(ij)-------(ij)
 * | |
 * | |
 * | |
 * (ij-1) (ij) (ij)
 * | |
 * | |
 * | |
 * (ij-iX-1)----(ij-iX)----(ij-iX)
 *
 * At each point, the function value is either 0, 1, or 2, depending
 * on whether it is below z0, between z0 and z1, or above z1.
 * Each zone either exists (1) or not (0).
 * From these three bits of data, all of the curve connectivity follows.
 *
 * The tracing algorithm is naturally edge-based: Either you are at a
 * point where a level cuts an edge, ready to step across a zone to
 * another edge, or you are drawing the edge itself, if it happens to
 * be a boundary with at least one section between z0 and z1.
 *
 * In either case, the edge is a directed edge -- either the zone
 * you are advancing into is to its left or right, or you are actually
 * drawing it. I always trace curves keeping the region between z0 and
 * z1 to the left of the curve. If I'm tracing a boundary, I'm always
 * moving CCW (counter clockwise) around the zone that exists. And if
 * I'm about to cross a zone, I'll make the direction of the edge I'm
 * sitting on be such that the zone I'm crossing is to its left.
 *
 * I start tracing each curve near its lower left corner (mesh oriented
 * as above), which is the first point I encounter scanning through the
 * mesh in order. When I figure the 012 z values and zonal existence,
 * I also mark the potential starting points: Each edge may harbor a
 * potential starting point corresponding to either direction, so there
 * are four start possibilities at each ij point. Only the following
 * possibilities need to be marked as potential starting edges:
 *
 * +-+-+-+
 * | | | |
 * A-0-C-+ One or both levels cut E and have z=1 above them, and
 * | EZ| | 0A is cut and either 0C is cut or CD is cut.
 * +-B-D-+ Or, one or both levels cut E and E is a boundary edge.
 * | | | | (and Z exists)
 * +-+-+-+
 *
 * +-+-+-+
 * | | | |
 * +-A-0-C One or both levels cut E and have z=1 below them, and
 * | |ZE | 0A is cut and either 0C is cut or CD is cut.
 * +-+-B-D Or, one or both levels cut E and E is a boundary edge.
 * | | | | (and Z exists)
 * +-+-+-+
 *
 * +-+-+-+
 * | | | |
 * +-+-+-+ E is a boundary edge, Z exists, at some point on E
 * | |Z| | lies between the levels.
 * +-+E+-+
 * | | | |
 * +-+-+-+
 *
 * +-+-+-+
 * | | | |
 * +-+E+-+ E is a boundary edge, Z exists, at some point on E
 * | |Z| | lies between the levels.
 * +-+-+-+
 * | | | |
 * +-+-+-+
 *
 * During the first tracing pass, the start mark is erased whenever
 * any non-starting edge is encountered, reducing the number of points
 * that need to be considered for the second pass. The first pass
 * makes the basic connectivity decisions. It figures out how many
 * disjoint curves there will be, and identifies slits for the two level
 * case or open contours for the single level case, and removes all but
 * the actual start markers. A second tracing pass can perform the
 * actual final trace.
 */

/* ------------------------------------------------------------------------ */

/* here is the minimum structure required to tell where we are in the
 * mesh sized data array */
class ContourTracer
{
	int edge; /* ij of current edge */
	int left; /* +-1 or +-imax as the zone is to right, left, below,
	 * or above the edge */
	int imax; /* imax for the mesh */
	int jmax; /* jmax for the mesh */
	int n; /* number of points marked on this curve so far */
	int count; /* count of start markers visited */
	double[] zlevel = new double[2]; /* contour levels, zlevel[1]<=zlevel[0] signals single level case */
	
	/* information to decide on correct contour direction in saddle zones
	 * is stored in a mesh sized array. Only those entries corresponding
	 * to saddle zones have nonzero values in this array. */
	int[] saddle; /* saddle zone information for the mesh */
	
	char[] reg; /* region array for the mesh (was int) */
	
	/* the data about edges, zones, and points -- boundary or not, exists
	 * or not, z value 0, 1, or 2 -- is kept in a mesh sized data array */
	int[] data; /* added by EF */
	
	int edge0, left0; /* starting site on this curve for closure */
	int level0; /* starting level for closure */
	int edge00; /* site needing START_ROW mark */

	/* making the actual marks requires a bunch of other stuff */
	double[] x , y , z; /* mesh coordinates and function values */
	double[] xcp = null, ycp = null; /* output contour points */
	int[] kcp = null; /* kind of contour point */
	
	/* the Cdata array consists of the following bits:
	 * Z_VALUE (2 bits) 0, 1, or 2 function value at point
	 * ZONE_EX 1 zone exists, 0 zone doesn't exist
	 * I_BNDY this i-edge (i=constant edge) is a mesh boundary
	 * J_BNDY this j-edge (i=constant edge) is a mesh boundary
	 * I0_START this i-edge is a start point into zone to left
	 * I1_START this i-edge is a start point into zone to right
	 * J0_START this j-edge is a start point into zone below
	 * J1_START this j-edge is a start point into zone above
	 * START_ROW next start point is in current row (accelerates 2nd pass)
	 * SLIT_UP marks this i-edge as the beginning of a slit upstroke
	 * SLIT_DN marks this i-edge as the beginning of a slit downstroke
	 * OPEN_END marks an i-edge start point whose other endpoint is
	 * on a boundary for the single level case
	 * ALL_DONE marks final start point
	 * SLIT_DN_VISITED this slit downstroke hasn't/has been visited in pass 2
	 */
	static final int Z_VALUE         = (1<<0) + (1<<1);
	static final int ZONE_EX         = (1<<2);
	static final int I_BNDY          = (1<<3);
	static final int J_BNDY          = (1<<4);
	static final int I0_START        = (1<<5);
	static final int I1_START        = (1<<6);
	static final int J0_START        = (1<<7);
	static final int J1_START        = (1<<8);
	static final int START_ROW       = (1<<9);
	static final int SLIT_UP         = (1<<10);
	static final int SLIT_DN         = (1<<11);
	static final int OPEN_END        = (1<<12);
	static final int ALL_DONE        = (1<<13);
	static final int SLIT_DN_VISITED = (1<<14);

	static final int ANY_START = (I0_START|I1_START|J0_START|J1_START);
	
	/* some helpful macros to find points relative to a given directed
	 * edge -- points are designated 0, 1, 2, 3 CCW around zone with 0 and
	 * 1 the endpoints of the current edge */
	static int FORWARD(int left_, int ix) {
		return (left_>0?(left_>1?1:-ix):(left_<-1?-1:ix));
	}
	
	static int POINT0(int edge_, int fwd) {
		return (edge_-(fwd>0?fwd:0));
	}
	
	static int POINT1(int edge_, int fwd) {
		return (edge_+(fwd<0?fwd:0));
	}
	
	static boolean IS_JEDGE(int edge_, int left_) {
		return (left_>0 ? (left_>1 ? true : false) : (left_ < -1 ? true : false));
	}
	
	static int START_MARK(int left_) {
		return (left_>0?(left_>1?J1_START:I1_START):(left_<-1?J0_START:I0_START)); 
	}

	static final int kind_zone = 0;
	static final int kind_edge1 = 1;
	static final int kind_edge2 = 2;
	static final int kind_slit_up = 3;
	static final int kind_slit_down = 4;
	static final int kind_start_slit = 16;

	/* Saddle zone array consists of the following bits:
	 * SADDLE_SET whether zone's saddle data has been set.
	 * SADDLE_GT0 whether z of centre of zone is higher than site.level[0].
	 * SADDLE_GT1 whether z of centre of zone is higher than site.level[1].
	 */
	static final int SADDLE_SET = (1<<0);
	static final int SADDLE_GT0 = (1<<1);
	static final int SADDLE_GT1 = (1<<2);

	ContourTracer(int iMax, int jMax, double[] x, double[] y, double[] z, int[] mask)
	{
		int ijmax = iMax * jMax;
		int nreg = iMax * jMax + iMax + 1;

		imax = iMax;
		jmax = jMax;
		data = new int[nreg];
		saddle = new int[ijmax];
		reg = null;
		if (mask != null)
		{
			reg = new char[nreg];
			mask_zones(iMax, jMax, mask, reg);
		}
		/* data will be initialized in data_init*/
		this.x = x;
		this.y = y;
		this.z = z;
	}
	
	/* reg should have the same dimensions as data, which
	 * has an extra iMax + 1 points relative to Z.
	 * It differs from mask in being the opposite (True
	 * where a region exists, versus the mask, which is True
	 * where a data point is bad), and in that it marks
	 * zones, not points. All four zones sharing a bad
	 * point must be marked as not existing.
	 */
	void
	mask_zones (int iMax, int jMax, int[] mask, char[] reg2)
	{
		int i, j, ij;
		int nreg = iMax * jMax + iMax + 1;

		for (ij = iMax+1; ij < iMax*jMax; ij++)
		{
			reg2[ij] = 1;
		}

		ij = 0;
		for (j = 0; j < jMax; j++)
		{
			for (i = 0; i < iMax; i++, ij++)
			{
				if (i == 0 || j == 0) reg2[ij] = 0;
				if (mask[ij] != 0)
				{
					reg2[ij] = 0;
					reg2[ij + 1] = 0;
					reg2[ij + iMax] = 0;
					reg2[ij + iMax + 1] = 0;
				}
			}
		}
		for (; ij < nreg; ij++)
		{
			reg2[ij] = 0;
		}
	}
	
	void print()
	{
		int nd = (int) (imax * (jmax + 1) + 1);
		System.out.printf("zlevels: %8.2lg %8.2lg\n", zlevel[0], zlevel[1]);
		System.out.printf("edge %ld, left %ld, n %ld, count %ld, edge0 %ld, left0 %ld\n",
				edge, left, n, count,
				edge0, left0);
		System.out.printf(" level0 %d, edge00 %ld\n", level0, edge00);
		System.out.printf("%04x\n", data[nd-1]);
		for (int j = (int) jmax; j >= 0; j--)
		{
			for (int i=0; i < imax; i++)
			{
				int ij = (int) (i + j * imax);
				System.out.printf("%04x ", data[ij]);
			}
			System.out.printf("\n");
		}
		System.out.printf("\n");
	}

	/* ------------------------------------------------------------------------ */

	/* zone_crosser, edge_walker and slit_cutter actually mark points */

	/* ------------------------------------------------------------------------ */

	/* zone_crosser assumes you are sitting at a cut edge about to cross
	 * the current zone. It always marks the initial point, crosses at
	 * least one zone, and marks the final point. On non-boundary i-edges,
	 * it is responsible for removing start markers on the first pass. */
	int
	zone_crosser (int level, boolean pass2) throws Exception
	{
		int edge_ = edge;
		int left_ = left;
		int n_ = n;
		int fwd = FORWARD (left_, imax);
		boolean jedge = IS_JEDGE (edge_, left_);
		int edge0_ = edge0;
		int left0_ = left0;
		boolean level0_ = (level0 == level);
		boolean two_levels = zlevel[1] > zlevel[0];

		int done = 0;
		int n_kind;

		if (level != 0)
			level = 2;

		for (;;)
		{
			n_kind = 0;
			/* set edge endpoints */
			int p0 = POINT0 (edge_, fwd);
			int p1 = POINT1 (edge_, fwd);

			/* always mark cut on current edge */
			if (pass2)
			{
				/* second pass actually computes and stores the point */
				double zcp = (zlevel[level] - z[p0]) / (z[p1] - z[p0]);
				xcp[n_] = zcp * (x[p1] - x[p0]) + x[p0];
				ycp[n_] = zcp * (y[p1] - y[p0]) + y[p0];
				kcp[n_] = kind_zone;
				n_kind = n_;
			}
			if (done==0 && !jedge)
			{
				if (n_!=0)
				{
					/* if this is not the first point on the curve, and we're
					 * not done, and this is an i-edge, check several things */
					if (!two_levels && !pass2 && (data[edge_] & OPEN_END)!=0)
					{
						/* reached an OPEN_END mark, skip the n++ */
						done = 4; /* same return value 4 used below */
						break;
					}

					/* check for curve closure -- if not, erase any start mark */
					if (edge_ == edge0_ && left_ == left0_)
					{
						/* may signal closure on a downstroke */
						if (level0_)
							done = (!pass2 && two_levels && left_ < 0) ? 5 : 3;
					}
					else if (!pass2)
					{
						int start = data[edge_] & (fwd > 0 ? I0_START : I1_START);
						if (start!=0)
						{
							data[edge_] &= ~start;
							count--;
						}
						if (!two_levels)
						{
							start = data[edge_] & (fwd > 0 ? I1_START : I0_START);
							if (start!=0)
							{
								data[edge_] &= ~start;
								count--;
							}
						}
					}
				}
			}
			n_++;
			if (done!=0)
				break;
			if (n_ >= (imax+1)*(jmax+1)) {
				throw new Exception("Bug in n");
			}

			/* cross current zone to another cut edge */
			boolean z0 = (data[p0] & Z_VALUE) != level; /* 1 if fill toward p0 */
			boolean z1 = !z0; /* know level cuts edge */
			boolean z2 = (data[p1 + left_] & Z_VALUE) != level;
			boolean z3 = (data[p0 + left_] & Z_VALUE) != level;
			if (z0 == z2)
			{
				boolean turnRight = true;
				if (z1 == z3)
				{
					/* this is a saddle zone, determine whether to turn left or
					 * right depending on height of centre of zone relative to
					 * contour level. Set saddle[zone] if not already decided. */
					int zone = edge_ + (left_ > 0 ? left_ : 0);
					if ((saddle[zone] & SADDLE_SET)==0)
					{
						saddle[zone] = SADDLE_SET;
						double zcentre = (z[p0] + z[p0+left_] + z[p1] + z[p1+left_])/4.0;
						if (zcentre > zlevel[0])
							saddle[zone] |= (two_levels && zcentre > zlevel[1])	? SADDLE_GT0 | SADDLE_GT1 : SADDLE_GT0;
					}

					turnRight = level == 2 ? (saddle[zone] & SADDLE_GT1)!=0 : (saddle[zone] & SADDLE_GT0)!=0;
					if (z1 ^ (level == 2))
						turnRight = !turnRight;
				}
				
				if (turnRight)
				{
					/* bend forward (right along curve) */
					jedge = !jedge;
					edge_ = p1 + (left_ > 0 ? left_ : 0);
					int tmp = fwd;
					fwd = -left_;
					left_ = tmp;
				} else {
					/* bend backward (left along curve) */
					jedge = !jedge;
					edge_ = p0 + (left_ > 0 ? left_ : 0);
					int tmp = fwd;
					fwd = left_;
					left_ = -tmp;
				}
			}
			else if (z1 == z3)
			{
				/* bend backward (left along curve) */
				jedge = !jedge;
				edge_ = p0 + (left_ > 0 ? left_ : 0);
				int tmp = fwd;
				fwd = left_;
				left_ = -tmp;
			}
			else
			{
				/* straight across to opposite edge */
				edge_ += left_;
			}
			/* after crossing zone, edge/left/fwd is oriented CCW relative to
			 * the next zone, assuming we will step there */

			/* now that we've taken a step, check for the downstroke
			 * of a slit on the second pass (upstroke checked above)
			 * -- taking step first avoids a race condition */
			if (pass2 && two_levels && !jedge)
			{
				if (left_ > 0)
				{
					if ((data[edge_] & SLIT_UP)!=0)
						done = 6;
				}
				else
				{
					if ((data[edge_] & SLIT_DN)!=0)
						done = 5;
				}
			}

			if (done==0)
			{
				/* finally, check if we are on a boundary */
				if ((data[edge_] & (jedge ? J_BNDY : I_BNDY))!=0)
				{
					done = two_levels ? 2 : 4;
					/* flip back into the zone that exists */
					left_ = -left_;
					fwd = -fwd;
					if (!pass2 && (edge_ != edge0_ || left_ != left0_))
					{
						int start = data[edge_] & START_MARK (left_);
						if (start!=0)
						{
							data[edge_] &= ~start;
							count--;
						}
					}
				}
			}
		}

		edge = edge_;
		n = n_;
		left = left_;
		if (done <= 4)
		{
			return done;
		}
		if (pass2 && n_kind!=0)
		{
			kcp[n_kind] += kind_start_slit;
		}
		return slit_cutter ((done - 5 != 0), pass2);
	}

	/* edge_walker assumes that the current edge is being drawn CCW
	 * around the current zone. Since only boundary edges are drawn
	 * and we always walk around with the filled region to the left,
	 * no edge is ever drawn CW. We attempt to advance to the next
	 * edge on this boundary, but if current second endpoint is not
	 * between the two contour levels, we exit back to zone_crosser.
	 * Note that we may wind up marking no points.
	 * -- edge_walker is never called for single level case */
	int
	edge_walker (boolean pass2)
	{
		int edge_ = edge;
		int left_ = left;
		int n_ = n;
		int fwd = FORWARD (left_, imax);
		int p0 = POINT0 (edge_, fwd);
		int p1 = POINT1 (edge_, fwd);
		boolean jedge = IS_JEDGE (edge_, left_);
		int edge0_ = edge0;
		int left0_ = left0;
		boolean level0_ = level0 == 2;
		boolean marked;
		int n_kind = 0;

		int z0, z1;
		boolean heads_up = false;

		for (;;)
		{
			/* mark endpoint 0 only if value is 1 there, and this is a
			 * two level task */
			z0 = data[p0] & Z_VALUE;
			z1 = data[p1] & Z_VALUE;
			marked = false;
			n_kind = 0;
			if (z0 == 1)
			{
				/* mark current boundary point */
				if (pass2)
				{
					xcp[n_] = x[p0];
					ycp[n_] = y[p0];
					kcp[n_] = kind_edge1;
					n_kind = n_;
				}
				marked = true;
			}
			else if (n_==0)
			{
				/* if this is the first point is not between the levels
				 * must do the job of the zone_crosser and mark the first cut here,
				 * so that it will be marked again by zone_crosser as it closes */
				if (pass2)
				{
					double zcp = zlevel[(z0 != 0)?1:0];
					zcp = (zcp - z[p0]) / (z[p1] - z[p0]);
					xcp[n_] = zcp * (x[p1] - x[p0]) + x[p0];
					ycp[n_] = zcp * (y[p1] - y[p0]) + y[p0];
					kcp[n_] = kind_edge2;
					n_kind = n_;
				}
				marked = true;
			}
			if (n_!=0)
			{
				/* check for closure */
				if (level0_ && edge_ == edge0_ && left_ == left0_)
				{
					edge = edge_;
					left = left_;
					n = n_ + (marked?1:0);
					/* if the curve is closing on a hole, need to make a downslit */
					if (fwd < 0 && (data[edge_] & (jedge ? J_BNDY : I_BNDY))==0)
					{
						if (n_kind!=0) kcp[n_kind] += kind_start_slit;
						return slit_cutter (false, pass2);
					}
					if (fwd < 0 && level0_ && left_ < 0)
					{
						/* remove J0_START from this boundary edge as boundary is
						 * included by the upwards slit from contour line below. */
						data[edge_] &= ~J0_START;
						if (n_kind!=0) kcp[n_kind] += kind_start_slit;
						return slit_cutter (false, pass2);
					}
					return 3;
				}
				else if (pass2)
				{
					if (heads_up || (fwd < 0 && (data[edge_] & SLIT_DN)!=0))
					{
						if (!heads_up && (data[edge_] & SLIT_DN_VISITED)==0)
							data[edge_] |= SLIT_DN_VISITED;
						else
						{
							edge = edge_;
							left = left_;
							n = n_ + (marked?1:0);
							if (n_kind!=0) kcp[n_kind] += kind_start_slit;
							return slit_cutter (heads_up, pass2);
						}
					}
				}
				else
				{
					/* if this is not first point, clear start mark for this edge */
					int start = data[edge_] & START_MARK (left_);
					if (start!=0)
					{
						data[edge_] &= ~start;
						count--;
					}
				}
			}
			if (marked)
				n_++;

			/* if next endpoint not between levels, need to exit to zone_crosser */
			if (z1 != 1)
			{
				edge = edge_;
				left = left_;
				n = n_;
				return (z1 != 0)?1:0; /* return level closest to p1 */
			}

			/* step to p1 and find next edge
			 * -- turn left if possible, else straight, else right
			 * -- check for upward slit beginning at same time */
			edge_ = p1 + (left_ > 0 ? left_ : 0);
			if (pass2 && jedge && fwd > 0 && (data[edge_] & SLIT_UP)!=0)
			{
				jedge = !jedge;
				heads_up = true;
			}
			else if ((data[edge_] & (jedge ? I_BNDY : J_BNDY))!=0)
			{
				int tmp = fwd;
				fwd = left_;
				left_ = -tmp;
				jedge = !jedge;
			}
			else
			{
				edge_ = p1 + (fwd > 0 ? fwd : 0);
				if (pass2 && !jedge && fwd > 0 && (data[edge_] & SLIT_UP)!=0)
				{
					heads_up = true;
				}
				else if ((data[edge_] & (jedge ? J_BNDY : I_BNDY))==0)
				{
					edge_ = p1 - (left_ < 0 ? left_ : 0);
					jedge = !jedge;
					{
						int tmp = fwd;
						fwd = -left_;
						left_ = tmp;
					}
				}
			}
			p0 = p1;
			p1 = POINT1 (edge_, fwd);
		}
	}

	/* -- slit_cutter is never called for single level case */
	int
	slit_cutter (boolean up, boolean pass2)
	{
		int n_ = n;

		if (up)
		{
			/* upward stroke of slit proceeds up left side of slit until
			 * it hits a boundary or a point not between the contour levels
			 * -- this never happens on the first pass */
			int p1 = edge;
			int z1;
			for (;;)
			{
				z1 = data[p1] & Z_VALUE;
				if (z1 != 1)
				{
					edge = p1;
					left = -1;
					n = n_;
					return (z1 != 0)?1:0;
				}
				else if ((data[p1] & J_BNDY)!=0)
				{
					/* this is very unusual case of closing on a mesh hole */
					edge = p1;
					left = -imax;
					n = n_;
					return 2;
				}
				xcp[n_] = x[p1];
				ycp[n_] = y[p1];
				kcp[n_] = kind_slit_up;
				n_++;
				p1 += imax;
			}

		}
		else
		{
			/* downward stroke proceeds down right side of slit until it
			 * hits a boundary or point not between the contour levels */
			int p0 = edge;
			int z0;
			/* at beginning of first pass, mark first i-edge with SLIT_DN */
			data[p0] |= SLIT_DN;
			p0 -= imax;
			for (;;)
			{
				z0 = data[p0] & Z_VALUE;
				if (!pass2)
				{
					if (z0 != 1 || (data[p0] & I_BNDY)!=0 || (data[p0 + 1] & J_BNDY)!=0)
					{
						/* at end of first pass, mark final i-edge with SLIT_UP */
						data[p0 + imax] |= SLIT_UP;
						/* one extra count for splicing at outer curve */
						n = n_ + 1;
						return 4; /* return same special value as for OPEN_END */
					}
				}
				else
				{
					if (z0 != 1)
					{
						edge = p0 + imax;
						left = 1;
						n = n_;
						return (z0 != 0)?1:0;
					}
					else if ((data[p0 + 1] & J_BNDY)!=0)
					{
						edge = p0 + 1;
						left = imax;
						n = n_;
						return 2;
					}
					else if ((data[p0] & I_BNDY)!=0)
					{
						edge = p0;
						left = 1;
						n = n_;
						return 2;
					}
				}
				if (pass2)
				{
					xcp[n_] = x[p0];
					ycp[n_] = y[p0];
					kcp[n_] = kind_slit_down;
					n_++;
				}
				else
				{
					/* on first pass need to count for upstroke as well */
					n_ += 2;
				}
				p0 -= imax;
			}
		}
	}

	/* ------------------------------------------------------------------------ */

	/* curve_tracer finds the next starting point, then traces the curve,
	 * returning the number of points on this curve
	 * -- in a two level trace, the return value is negative on the
	 * first pass if the curve closed on a hole
	 * -- in a single level trace, the return value is negative on the
	 * first pass if the curve is an incomplete open curve
	 * -- a return value of 0 indicates no more curves */

	/* this calls the first three to trace the next disjoint curve
	 * -- return value is number of points on this curve, or
	 * 0 if there are no more curves this pass
	 * -(number of points) on first pass if:
	 * this is two level case, and the curve closed on a hole
	 * this is single level case, curve is open, and will start from
	 * a different point on the second pass
	 * -- in both cases, this curve will be combined with another
	 * on the second pass */
	
	int
	curve_tracer (boolean pass2) throws Exception
	{
		int edge0_ = edge0;
		int left0_ = left0;
		int edge00_ = edge00;
		boolean two_levels = zlevel[1] > zlevel[0];

		/* it is possible for a single i-edge to serve as two actual start
		 * points, one to the right and one to the left
		 * -- for the two level case, this happens on the first pass for
		 * a doubly cut edge, or on a chunking boundary
		 * -- for single level case, this is impossible, but a similar
		 * situation involving open curves is handled below
		 * a second two start possibility is when the edge0 zone does not
		 * exist and both the i-edge and j-edge boundaries are cut
		 * yet another possibility is three start points at a junction
		 * of chunk cuts
		 * -- sigh, several other rare possibilities,
		 * allow for general case, just go in order i1, i0, j1, j0 */
		int two_starts;
		/* printf("curve_tracer pass %d\n", pass2); */
		/* print_Csite(site); */
		if (left0_ == 1)
			two_starts = data[edge0_] & (I0_START | J1_START | J0_START);
		else if (left0_ == -1)
			two_starts = data[edge0_] & (J1_START | J0_START);
		else if (left0_ == imax)
			two_starts = data[edge0_] & J0_START;
		else
			two_starts = 0;
		
		if (pass2 || edge0_ == 0)
		{
			/* zip up to row marked on first pass (or by data_init if edge0==0)
			 * -- but not for double start case */
			if (two_starts==0)
			{
				/* final start point marked by ALL_DONE marker */
				boolean first = (edge0_ == 0 && !pass2);
				int e0 = edge0_;
				if ((data[edge0_] & ALL_DONE)!=0)
					return 0;
				while ((data[edge0_] & START_ROW)==0)
					edge0_ += imax;
				if (e0 == edge0_)
					edge0_++; /* two starts handled specially */
				if (first)
					/* if this is the very first start point, we want to remove
					 * the START_ROW marker placed by data_init */
					data[edge0_ - edge0_ % imax] &= ~START_ROW;
			}
		}
		else
		{
			/* first pass ends when all potential start points visited */
			if (count <= 0)
			{
				/* place ALL_DONE marker for second pass */
				data[edge00_] |= ALL_DONE;
				/* reset initial site for second pass */
				edge0 = edge00 = left0 = 0;
				return 0;
			}
			if (two_starts==0)
				edge0_++;
		}

		int level, level0_;
		
		if (two_starts!=0)
		{
			/* trace second curve with this start immediately */
			if (left0_ == 1 && (data[edge0_] & I0_START)!=0)
			{
				left0_ = -1;
				level = (data[edge0_] & I_BNDY)!=0 ? 2 : 0;
			}
			else if ((left0_ == 1 || left0_ == -1) && (data[edge0_] & J1_START)!=0)
			{
				left0_ = imax;
				level = 2;
			}
			else
			{
				left0_ = -imax;
				level = 2;
			}
		}
		else
		{
			/* usual case is to scan for next start marker
			 * -- on second pass, this is at most one row of mesh, but first
			 * pass hits nearly every point of the mesh, since it can't
			 * know in advance which potential start marks removed */
			while ((data[edge0_] & ANY_START)==0)
				edge0_++;

			if ((data[edge0_] & I1_START)!=0)
				left0_ = 1;
			else if ((data[edge0_] & I0_START)!=0)
				left0_ = -1;
			else if ((data[edge0_] & J1_START)!=0)
				left0_ = imax;
			else /*data[edge0]&J0_START */
				left0_ = -imax;

			if ((data[edge0_] & (I1_START | I0_START))!=0)
				level = (data[edge0_] & I_BNDY)!=0 ? 2 : 0;
			else
				level = 2;
		}

		/* this start marker will not be unmarked, but it has been visited */
		if (!pass2)
			count--;

		/* if this curve starts on a non-boundary i-edge, we need to
		 * determine the level */
		if (level==0 && two_levels)
		{
			level = (left0_ > 0 ?
					((data[edge0_ - imax] & Z_VALUE) !=	0) : ((data[edge0_] & Z_VALUE) != 0))?1:0;
		}
			
		/* initialize site for this curve */
		edge = edge0 = edge0_;
		left = left0 = left0_;
		level0 = level0_ = level; /* for open curve detection only */

		/* single level case just uses zone_crosser */
		if (!two_levels)
			level = 0;

		/* to generate the curve, alternate between zone_crosser and
		 * edge_walker until closure or first call to edge_walker in
		 * single level case */
		n = 0;
		for (;;)
		{
			if (level < 2)
				level = zone_crosser (level, pass2);
			else if (level < 3)
				level = edge_walker (pass2);
			else
				break;
		}
		
		int n_ = n;

		/* single level case may have ended at a boundary rather than closing
		 * -- need to recognize this case here in order to place the
		 * OPEN_END mark for zone_crosser, remove this start marker,
		 * and be sure not to make a START_ROW mark for this case
		 * two level case may close with slit_cutter, in which case start
		 * must also be removed and no START_ROW mark made
		 * -- change sign of return n to inform caller */
		boolean mark_row;
		
		if (!pass2 && level > 3 && (two_levels || level0_ == 0))
		{
			if (!two_levels)
				data[edge0_] |= OPEN_END;
			data[edge0_] &= ~(left0_ > 0 ? I1_START : I0_START);
			mark_row = false; /* do not mark START_ROW */
			n_ = -n_;
		}
		else
		{
			if (two_levels)
				mark_row = two_starts==0;
			else
				mark_row = true;
		}

		/* on first pass, must apply START_ROW mark in column above previous
		 * start marker
		 * -- but skip if we just did second of two start case */
		if (!pass2 && mark_row)
		{
			data[edge0_ - (edge0_ - edge00_) % imax] |= START_ROW;
			edge00 = edge0_;
		}

		return n_;
	}

	/* ------------------------------------------------------------------------ */
	/* this initializes the data array for curve_tracer */
	/* ------------------------------------------------------------------------ */

	void
	data_init (int nchunk)
	{
		boolean two_levels = zlevel[1] > zlevel[0];
		int icsize = imax - 1;
		int jcsize = jmax - 1;
		int irem, jrem;

		if (nchunk!=0 && two_levels)
		{
			/* figure out chunk sizes
			 * -- input nchunk is square root of maximum allowed zones per chunk
			 * -- start points for single level case are wrong, so don't try it */
			int inum = (nchunk * nchunk) / (jmax - 1);
			int jnum = (nchunk * nchunk) / (imax - 1);
			if (inum < nchunk)
				inum = nchunk;
			if (jnum < nchunk)
				jnum = nchunk;
			/* ijnum= actual number of chunks,
			 * ijrem= number of those chunks needing one more zone (ijcsize+1) */
			inum = (imax - 2) / inum + 1;
			icsize = (imax - 1) / inum;
			irem = (imax - 1) % inum;
			jnum = (jmax - 2) / jnum + 1;
			jcsize = (jmax - 1) / jnum;
			jrem = (jmax - 1) % jnum;
			/* convert ijrem into value of i or j at which to begin adding an
			 * extra zone */
			irem = (inum - irem) * icsize;
			jrem = (jnum - jrem) * jcsize;
		}
		else
		{
			irem = imax;
			jrem = jmax;
		}

		/* do everything in a single pass through the data array to
		 * minimize cache faulting (z, reg, and data are potentially
		 * very large arrays)
		 * access to the z and reg arrays is strictly sequential,
		 * but we need two rows (+-imax) of the data array at a time */
		if (z[0] > zlevel[0])
			data[0] = (two_levels && z[0] > zlevel[1]) ? 2 : 1;
		else
			data[0] = 0;
		int jchunk = 0;
		boolean started = false;
		int count_ = 0;
		for (int j = 0, ij = 0; j < jmax; j++)
		{
			int ichunk = 0;
			boolean i_was_chunk = false;
			for (int i = 0; i < imax; i++, ij++)
			{
				/* transfer zonal existence from reg to data array
				 * -- get these for next row so we can figure existence of
				 * points and j-edges for this row */
				data[ij + imax + 1] = 0;
				if (reg!=null)
				{
					if (reg[ij + imax + 1] != 0)
						data[ij + imax + 1] = ZONE_EX;			
				}
				else
				{
					if (i < imax - 1 && j < jmax - 1)
						data[ij + imax + 1] = ZONE_EX;
				}

				/* translate z values to 0, 1, 2 flags */
				if (ij < imax) /* first row */
					data[ij + 1] = 0;
				if (ij < imax * jmax - 1 && z[ij + 1] > zlevel[0])
					data[ij + 1] |= (two_levels && z[ij + 1] > zlevel[1]) ? 2 : 1;

				/* apply edge boundary marks */
				boolean ibndy = i == ichunk
						|| (data[ij] & ZONE_EX) != (data[ij + 1] & ZONE_EX);
				boolean jbndy = j == jchunk
						|| (data[ij] & ZONE_EX) != (data[ij + imax] & ZONE_EX);
				if (ibndy)
					data[ij] |= I_BNDY;
				if (jbndy)
					data[ij] |= J_BNDY;

				/* apply i-edge start marks
				 * -- i-edges are only marked when actually cut
				 * -- no mark is necessary if one of the j-edges which share
				 * the lower endpoint is also cut
				 * -- no I0 mark necessary unless filled region below some cut,
				 * no I1 mark necessary unless filled region above some cut */
				if (j!=0)
				{
					int v0 = (data[ij] & Z_VALUE);
					int vb = (data[ij - imax] & Z_VALUE);
					if (v0 != vb)
					{ /* i-edge is cut */
						if (ibndy)
						{
							if ((data[ij] & ZONE_EX)!=0)
							{
								data[ij] |= I0_START;
								count_++;
							}
							if ((data[ij + 1] & ZONE_EX)!=0)
							{
								data[ij] |= I1_START;
								count_++;
							}
						}
						else
						{
							int va = (data[ij - 1] & Z_VALUE);
							int vc = (data[ij + 1] & Z_VALUE);
							int vd = (data[ij - imax + 1] & Z_VALUE);
							if (v0 != 1 && va != v0
									&& (vc != v0 || vd != v0) && (data[ij] & ZONE_EX)!=0)
							{
								data[ij] |= I0_START;
								count_++;
							}
							if (vb != 1 && va == vb
									&& (vc == vb || vd == vb)
									&& (data[ij + 1] & ZONE_EX)!=0)
							{
								data[ij] |= I1_START;
								count_++;
							}
						}
					}
				}

				/* apply j-edge start marks
				 * -- j-edges are only marked when they are boundaries
				 * -- all cut boundary edges marked
				 * -- for two level case, a few uncut edges must be marked
				 */
				if (i!=0 && jbndy)
				{
					int v0 = (data[ij] & Z_VALUE);
					int vb = (data[ij - 1] & Z_VALUE);
					if (v0 != vb)
					{
						if ((data[ij] & ZONE_EX)!=0)
						{
							data[ij] |= J0_START;
							count_++;
						}
						if ((data[ij + imax] & ZONE_EX)!=0)
						{
							data[ij] |= J1_START;
							count_++;
						}
					}
					else if (two_levels && v0 == 1)
					{
						if ((data[ij + imax] & ZONE_EX)!=0)
						{
							if (i_was_chunk || (data[ij + imax - 1] & ZONE_EX)==0)
							{
								/* lower left is a drawn part of boundary */
								data[ij] |= J1_START;
								count_++;
							}
						}
						else if ((data[ij] & ZONE_EX)!=0)
						{
							if ((data[ij + imax - 1] & ZONE_EX)!=0)
							{
								/* weird case of open hole at lower left */
								data[ij] |= J0_START;
								count_++;
							}
						}
					}
				}

				i_was_chunk = (i == ichunk);
				if (i_was_chunk)
					ichunk += icsize + ((ichunk >= irem)?1:0);
			}

			if (j == jchunk)
				jchunk += jcsize + ((jchunk >= jrem)?1:0);

			/* place first START_ROW marker */
			if (count_!=0 && !started)
			{
				data[ij - imax] |= START_ROW;
				started = true;
			}
		}

		/* place immediate stop mark if nothing found */
		if (count_==0)
			data[0] |= ALL_DONE;
		else
			for (int i = 0; i < imax * jmax; ++i) saddle[i] = 0;

		/* initialize site */
		edge0 = edge00 = edge = 0;
		left0 = left = 0;
		n = 0;
		count = count_;
	}
}