/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobplugin.descriptors;

import icy.type.point.Point4D;

import java.util.Comparator;

/**
 * @class CellDescriptor
 * 
 *        Abstract class describing a cell.
 */
public interface CellDescriptor {
	/**
	 * Initialize the cells parameters.
	 * 
	 * @param center
	 *            Center of the cell. The fourth component is the initial
	 *            radius.
	 */
	public abstract void init(final Point4D center);

	/** Energy of the cell. */
	public abstract double energy();

	/**
	 * Return the normalize parameters of the cell.
	 * 
	 * @warning The last element contains the radius of the cell and has to be
	 *          not changed.
	 */
	public abstract double[] getParameters();

	/**
	 * Set all the parameters of the cell.
	 * 
	 * @param parameters
	 *            Normalized parameters. The last element has to be the radius
	 *            obtained from @ref getParameters().
	 */
	public abstract void setParameters(final double[] parameters);

	/**
	 * Compute the energy of the cell. The energy is computed with the formula:
	 * \f[ E = \frac{\left(\sqrt(2)\int_{\Omega_{shell}}
	 * mask\left(\textbf{x}\right)~d\Omega - \int_{\Omega_{cell}}
	 * mask\left(\textbf{x}\right)~d\Omega\right) - \int_{\Omega_{cell}}
	 * mask\left(\textbf{x}\right)~d\Omega}{Volume_{cell}} \f]
	 * 
	 * Shell design a scaled version of the cell by a factor \f$\sqrt(2)\f$ and
	 * \f$\Omega\f$ is the surface of the cell.
	 * 
	 * @param mask
	 *            Binarized image used to compute the energy of the cell.
	 */
	public abstract double computeEnergy(final ImageDescriptor imgInt);

	/**
	 * Test if two cells can be considered as equal.
	 * 
	 * @param other
	 *            Other cell to be compared with. Must be of the same type
	 *            (EllipseDescriptor or EllipsoidDescriptor).
	 * @param tolerance
	 *            Threshold criterion.
	 * @return Return true it the two cells are closed enough to be considered
	 *         as equal.
	 */
	public abstract boolean isClosedTo(final CellDescriptor other);

	/**
	 * @class CellComparator
	 * 
	 *        Comparison class between cell descriptors based on their energy.
	 */
	public static class CellComparator implements Comparator<CellDescriptor> {

		/**
		 * Compare two cells according to their energy.
		 * 
		 * @return Return -1 if cell1 has a lower energy than cell2, 1 if cell1
		 *         has higher energy and 0 if the two cells have the same
		 *         energy.
		 */
		@Override
		public int compare(CellDescriptor cell1, CellDescriptor cell2) {
			if (cell1.energy() > cell2.energy()) {
				return 1;
			}
			if (cell1.energy() < cell2.energy()) {
				return -1;
			}
			return 0;
		}
	}
}
