/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobplugin.descriptors;

import icy.type.point.Point4D;
import plugins.big.bigsnakeutils.icy.ellipsoid.Ellipsoid2D;

/**
 * @class EllipseDescriptor
 * 
 *        Describe a 2D cell.
 * 
 */
public class EllipseDescriptor extends Ellipsoid2D implements CellDescriptor {
	static final private double SQRT2 = Math.sqrt(2);
	static final private int SAMPLES_COUNT = 30;

	private double _energy;

	/**
	 * Constructor.
	 */
	public EllipseDescriptor() {
		this(0);
	}

	/**
	 * Constructor/
	 * 
	 * @param t
	 *            Time corresponding to this cell.
	 */
	public EllipseDescriptor(int t) {
		super(t);
		_energy = 0;
	}

	/**
	 * Constructor.
	 * 
	 * @param other
	 *            Cell to copy.
	 */
	public EllipseDescriptor(final EllipseDescriptor other) {
		super(other);
		_energy = 0;
	}

	@Override
	public void init(final Point4D center) {
		x0 = center.getX();
		y0 = center.getY();
		alpha = 0;
		a = b = center.getT();
	}

	@Override
	public double[] getParameters() {
		double r = (a + b) / 2;
		return new double[] { x0 / r, y0 / r, a / r, b / r, alpha / Math.PI, r };
	}

	@Override
	public void setParameters(final double[] parameters) {
		double r = parameters[5];
		x0 = r * parameters[0];
		y0 = r * parameters[1];
		a = r * parameters[2];
		b = r * parameters[3];
		alpha = Math.PI * parameters[4];
	}

	@Override
	public double energy() {
		return _energy;
	}

	@Override
	public double computeEnergy(final ImageDescriptor imgInt) {
		if (!isValid()) {
			return Double.POSITIVE_INFINITY;
		}

		// Create the shell
		EllipseDescriptor shell = new EllipseDescriptor();
		shell.a = SQRT2 * a;
		shell.b = SQRT2 * b;
		shell.x0 = x0;
		shell.y0 = y0;
		shell.alpha = alpha;

		// Compute the integrals
		double I_cell = computeIntegral(imgInt);
		double I_shell = shell.computeIntegral(imgInt);

		// Compute the visible area
		double A = computeVisibleArea(imgInt);

		_energy = (I_shell - 2 * I_cell) / A;

		return _energy;
	}

	@Override
	public boolean isClosedTo(final CellDescriptor other) {
		if (!(other instanceof EllipseDescriptor)) {
			return false;
		}

		EllipseDescriptor ellipse = (EllipseDescriptor) other;

		double rmean = (a + b + ellipse.a + ellipse.b) / 4.0;
		double dx = x0 - ellipse.x0;
		double dy = y0 - ellipse.y0;

		return (dx * dx + dy * dy < rmean * rmean);
	}

	private double computeIntegral(final ImageDescriptor imgInt) {
		double R[] = getRotationMatrix();
		R[0] *= a;
		R[1] *= b;
		R[2] *= a;
		R[3] *= b;
		double I = 0;

		for (int i = 0; i < SAMPLES_COUNT; ++i) {
			double t = (2 * i * Math.PI) / SAMPLES_COUNT;
			double c = Math.cos(t);
			double s = Math.sin(t);

			int x = (int) Math.min(Math.max(x0 + R[0] * c + R[1] * s, 0),
					imgInt.width - 1);
			int y = (int) Math.min(Math.max(y0 + R[2] * c + R[3] * s, 0),
					imgInt.height - 1);

			I += imgInt.value(x, y) * (-R[2] * s + R[3] * c);
		}

		return I / SAMPLES_COUNT;
	}

	private double computeVisibleArea(final ImageDescriptor imgInt) {
		double R[] = getRotationMatrix();
		R[0] *= a;
		R[1] *= b;
		R[2] *= a;
		R[3] *= b;
		double A = 0;

		for (int i = 0; i < SAMPLES_COUNT; ++i) {
			double t = (2 * i * Math.PI) / SAMPLES_COUNT;
			double c = Math.cos(t);
			double s = Math.sin(t);

			int x = (int) Math.min(Math.max(x0 + R[0] * c + R[1] * s, 0),
					imgInt.width - 1);

			A += x * (-R[2] * s + R[3] * c);
		}

		return A / SAMPLES_COUNT;
	}
}