/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobgenerator;

import icy.canvas.Canvas3D;
import icy.canvas.IcyCanvas;
import icy.canvas.IcyCanvas2D;
import icy.canvas.IcyCanvas3D;
import icy.sequence.Sequence;
import icy.vtk.VtkUtil;

import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;

import plugins.big.bigsnakeutils.icy.ellipsoid.AbstractEllipsoid;
import plugins.big.bigsnakeutils.icy.ellipsoid.Ellipsoid2D;
import plugins.big.bigsnakeutils.icy.ellipsoid.Ellipsoid3D;
import plugins.nchenouard.spot.Detection;
import vtk.vtkActor;
import vtk.vtkCellArray;
import vtk.vtkDoubleArray;
import vtk.vtkPoints;
import vtk.vtkPolyData;
import vtk.vtkPolyDataMapper;
import vtk.vtkRenderer;

/**
 * Detection class for Ellipsoid3D objects. Used to display Ellipsoid3D with the
 * track manager.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public class CellDetection3D extends Detection {
	/**
	 * Number of horizontal lines of the 3D representation of the ellipsoid.
	 */
	private static final int H_LINES_COUNT = 12;

	/**
	 * Number of horizontal lines of the 3D representation of the ellipsoid.
	 */
	private static final int V_LINES_COUNT = 12;

	private boolean _isInitialized = false;
	private vtkActor _actor = null;
	private final Ellipsoid3D _cell;

	public CellDetection3D(Ellipsoid3D cell) {
		super(cell.x0, cell.y0, cell.z0, cell.getT());
		_cell = cell;
	}

	@Override
	public void paint(Graphics2D g2, Sequence sequence, IcyCanvas canvas) {
		if (_cell.getT() != canvas.getPositionT()) {
			return;
		}

		if (canvas instanceof IcyCanvas2D) {
			int zSlide = canvas.getPositionZ();

			Ellipsoid2D ellipse = _cell.intersectionAtZ(zSlide);
			if (ellipse != null) {
				Shape baseShape = new Ellipse2D.Double(ellipse.x0 - ellipse.a,
						ellipse.y0 - ellipse.b, 2 * ellipse.a, 2 * ellipse.b);
				Shape rotatedShape = AffineTransform.getRotateInstance(
						ellipse.alpha, ellipse.x0, ellipse.y0)
						.createTransformedShape(baseShape);
				g2.draw(rotatedShape);
			}
		} else if (canvas instanceof IcyCanvas3D) {
			final Canvas3D canvas3d = (Canvas3D) canvas;
			final vtkRenderer renderer = canvas3d.getRenderer();

			if (!_isInitialized) {
				init3DRenderer(renderer, sequence.getPixelSizeX(),
						sequence.getPixelSizeY(), sequence.getPixelSizeZ());
			}
			if (!VtkUtil.findProp(renderer, _actor)) {
				renderer.AddActor(_actor);
			}
		}
	}

	protected void init3DRenderer(vtkRenderer renderer, double pixelSizeX,
			double pixelSizeY, double pixelSizeZ) {
		renderer.SetGlobalWarningDisplay(0);

		final vtkPoints points = new vtkPoints();
		final double[] coordinates = new double[3 * H_LINES_COUNT
				* V_LINES_COUNT];
		int[][] indices = new int[2 * H_LINES_COUNT * V_LINES_COUNT][2];
		int i = 0, j = 0, offset = H_LINES_COUNT * V_LINES_COUNT;

		final double[] R = AbstractEllipsoid.transposeMatrix(_cell
				.getRotationMatrix());
		for (int vi = 0; vi < H_LINES_COUNT; vi++) {
			double v = _cell.beta + Math.PI
					* (2.0 * vi / (H_LINES_COUNT - 1) - 1);
			double cb = Math.cos(v);
			double sb = Math.sin(v);

			for (int ui = 0; ui < V_LINES_COUNT; ui++) {
				double u = _cell.alpha + ui * Math.PI / (V_LINES_COUNT - 1);
				double ca = Math.cos(u);
				double sa = Math.sin(u);

				double X[] = { _cell.a * ca * cb, _cell.b * sa * cb,
						_cell.c * sb };
				X = AbstractEllipsoid.applyRotation(R, X);

				coordinates[3 * i] = pixelSizeX * (_cell.x0 + X[0]);
				coordinates[3 * i + 1] = pixelSizeY * (_cell.y0 + X[1]);
				coordinates[3 * i + 2] = pixelSizeZ * (_cell.z0 + X[2]);

				indices[j][0] = i;
				indices[j][1] = (ui != V_LINES_COUNT - 1) ? i + 1 : i + 1
						- V_LINES_COUNT;
				indices[j + offset][0] = i;
				indices[j + offset][1] = (vi != H_LINES_COUNT - 1) ? i
						+ V_LINES_COUNT : i - (H_LINES_COUNT - 1)
						* V_LINES_COUNT;

				i++;
				j++;
			}
		}

		final vtkDoubleArray array = new vtkDoubleArray();
		array.SetJavaArray(coordinates);
		array.SetNumberOfComponents(3);
		points.SetData(array);

		final vtkCellArray cells = VtkUtil.getCells(indices.length - 1,
				VtkUtil.prepareCells(indices));

		vtkPolyData data = new vtkPolyData();
		data.SetPoints(points);
		data.SetLines(cells);

		// Add actor to the renderer
		final vtkPolyDataMapper polyMapper = new vtkPolyDataMapper();
		polyMapper.SetInputData(data);
		_actor = new vtkActor();
		_actor.SetMapper(polyMapper);
		_actor.GetProperty().SetColor(1, 0, 0);
		renderer.AddActor(_actor);

		_isInitialized = true;
	}
}
