package plugins.fab.manualtnt;

import icy.canvas.Canvas2D;
import icy.canvas.IcyCanvas;
import icy.file.xls.XlsManager;
import icy.image.lut.LUT.LUTChannel;
import icy.painter.Overlay;
import icy.sequence.Sequence;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.font.FontRenderContext;
import java.awt.font.TextLayout;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.io.File;
import java.util.ArrayList;

import org.w3c.dom.Node;

public class ManualTNTOverlay extends Overlay {

	TNTAnnotation tntAnnotation = null ;
	
	enum Mode { NO_EDIT_MODE, EDIT_MODE };
	
	Mode mode = Mode.NO_EDIT_MODE; 
	ManualTNT manualTNT;
	
	public ManualTNTOverlay(String name, Node node , File xlsFile , double scale , ManualTNT manualTNT )
	{
		super(name);
	
		tntAnnotation = new TNTAnnotation( node , xlsFile , scale ); 
		tntAnnotation.loadPersistentData( );
		this.manualTNT = manualTNT;
	}

	@Override
	public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas) {

		if ( g == null ) return;
		
		
		tntAnnotation.paint( g, sequence, canvas );
		
		Graphics2D absoluteG = (Graphics2D) g.create();
		//absoluteG.
		Canvas2D canvas2D = (Canvas2D) canvas;
		absoluteG.transform( canvas2D.getInverseTransform() );
		
		int x = 20;
		int y = 20;
		Font font = new Font("Arial" , Font.BOLD , 20 );
		absoluteG.setFont( font );
		absoluteG.setColor( Color.white );
		switch ( mode ) {
		case NO_EDIT_MODE:
			drawString("No edition (space to edit)", x, y , absoluteG );
			break;
		case EDIT_MODE:
			absoluteG.setColor( Color.green );
			drawString("Edit mode (space to stop)", x, y , absoluteG );
			break;

		}
		
		y+=20;
		absoluteG.setColor( Color.white );
		drawString( "Press enter to process next image" , x, y , absoluteG );

		y+=20;
		absoluteG.setColor( Color.white );
		drawString( "#Cells: " + tntAnnotation.getCellList().size() , x, y , absoluteG );
		
		y+=20;
		absoluteG.setColor( Color.green );
		drawString( "#Cells class 0: " + tntAnnotation.getNumberOfCellsClass( 0 ) , x, y , absoluteG );
		
		y+=20;
		absoluteG.setColor( Color.red );
		drawString( "#Cells class 1: " + tntAnnotation.getNumberOfCellsClass( 1 ) , x, y , absoluteG );
		
		y+=20;
		absoluteG.setColor( Color.blue );
		drawString( "#Cells class 2: " + tntAnnotation.getNumberOfCellsClass( 2 ) , x, y , absoluteG );
		
		y+=20;
		absoluteG.setColor( Color.yellow );
		drawString( "#Cells class 3: " + tntAnnotation.getNumberOfCellsClass( 3 ) , x, y , absoluteG );
				
		if ( isDragging )
		{
			Line2D line = new Line2D.Double( pointA , pointB );
			
			g.setColor( Color.black );
			g.setStroke( new BasicStroke( 5 ) );
			g.draw( line );
			
			g.setColor( Color.yellow );
			g.setStroke( new BasicStroke( 4 ) );
			g.draw( line );
	
		}
		
	}
	
	private void drawString(String string, int x, int y , Graphics2D g ) {
		
		Color color = g.getColor();
		FontRenderContext frc = g.getFontRenderContext();
		
		TextLayout textTl = new TextLayout( string, g.getFont(), frc );
		g.translate( x , y );		
		Shape outline = textTl.getOutline(null);		
		
		g.drawString( string , 0, 0 );
		
		g.setColor( Color.black );
		g.draw( outline );
		
		g.translate( -x , -y );
		g.setColor( color );
	
	}

	boolean isDragging = false;
	Point2D pointA ;
	Point2D pointB ;
	
	@Override
	public void mouseDrag(MouseEvent e, Point2D imagePoint, IcyCanvas canvas) {

		if ( mode == Mode.EDIT_MODE )
		{
//			mouseClick(e, imagePoint, canvas);
//			e.consume();
		}

		if ( mode == Mode.EDIT_MODE )
		{
			if ( isDragging )
			{				
				pointB = new Point2D.Double( imagePoint.getX() , imagePoint.getY() );
			}else
			{			
				isDragging = true;
				pointA = new Point2D.Double( imagePoint.getX() , imagePoint.getY() );
				pointB = new Point2D.Double( imagePoint.getX() , imagePoint.getY() );				
			}
			canvas.getSequence().painterChanged( this );
			e.consume();
		}
		
	}
	
	@Override
	public void mouseReleased(MouseEvent e, Point2D imagePoint, IcyCanvas canvas) {

		if ( mode == Mode.EDIT_MODE )
		{
			if ( isDragging )
			{
				isDragging = false;
				if ( pointA.distance( pointB ) < 20 )
				{
					createCell( imagePoint );
					
					e.consume();
					canvas.getSequence().painterChanged( this );
					return;
				}
			
				tntAnnotation.addTNT( new TNT( pointA , pointB ) );
				
				e.consume();
				canvas.getSequence().painterChanged( this );
			}
		}
	}
	
	
	private void createCell(Point2D imagePoint) {

		Point2D cellMassCenter = new Point2D.Double( imagePoint.getX() , imagePoint.getY() );

		// check if an existing cell mark should be deleted or not.
		{
			for ( Cell cell : new ArrayList<Cell>(tntAnnotation.getCellList() ) )
			{
				if ( cell.massCenter.distance( cellMassCenter ) < 20 )
				{
					cell.cellClass++;
					
					if ( cell.cellClass > 3 )
					{
						//remove cell
						tntAnnotation.removeCell( cell );
					}
					return;
				}
			}
		}

		tntAnnotation.addCell( new Cell( cellMassCenter ) );		
		
	}

	@Override
	public void keyPressed(KeyEvent e, Point2D imagePoint, IcyCanvas canvas) {

		
		if ( e.getKeyCode() == KeyEvent.VK_ENTER )
		{
			manualTNT.openNextImage();
			tntAnnotation.saveXLS();
		}
		
		if ( e.getKeyCode() == KeyEvent.VK_SPACE )
		{
			switch ( mode ) {
			case NO_EDIT_MODE:
				mode = Mode.EDIT_MODE;
				break;
			case EDIT_MODE:
				mode = Mode.NO_EDIT_MODE;
				break;
			}
			
			canvas.getSequence().painterChanged( this );
			
		}
		
		if ( e.getKeyChar() == 'a' )
		{
			for ( Sequence sequence : getSequences() )
			{
				for ( LUTChannel channel : sequence.getViewers().get(0).getCanvas().getLut().getLutChannels() )
				{
					channel.setMax( channel.getMax() - channel.getMaxBound()/100 );
					
				}			
			}
		}

		if ( e.getKeyChar() == 's' )
		{
			tntAnnotation.saveXLS();
		}
		
		if ( e.getKeyChar() == 'z' )
		{
			for ( Sequence sequence : getSequences() )
			{
				for ( LUTChannel channel : sequence.getViewers().get(0).getCanvas().getLut().getLutChannels() )
				{
					channel.setMax( channel.getMax() + channel.getMaxBound()/100 );
				}			
			}
		}
		
		if ( e.getKeyChar() == 'q' )
		{
			Canvas2D c = (Canvas2D) canvas;
			c.fitImageToCanvas( true );
		}
		
		if ( e.getKeyCode() == KeyEvent.VK_DELETE )
		{
			
			if ( mode == Mode.EDIT_MODE )
			{
				// search for closest TNT mouse position
				double bestDistance = Double.MAX_VALUE;
				TNT bestTNT = null;
				Cell bestCell = null;
				
				for( TNT tnt : tntAnnotation.getTNTList() )
				{
					Line2D line = new Line2D.Double( tnt.a , tnt.b );
					double distance = line.ptSegDist( imagePoint );
					if ( distance < bestDistance)
					{
						bestDistance = distance;
						bestTNT= tnt;
						bestCell = null;
					}
				}

				// search for closest Cell mouse position
				for( Cell cell : tntAnnotation.getCellList() )
				{					
					double distance = cell.massCenter.distance( imagePoint );
					if ( distance < bestDistance)
					{
						bestDistance = distance;
						bestTNT = null;
						bestCell= cell;
					}
				}
				
				tntAnnotation.removeTNT( bestTNT );
				tntAnnotation.removeCell( bestCell );
				canvas.getSequence().painterChanged( this );				
				
			}
		}

	}
	
	@Override
	public void mouseClick(MouseEvent e, Point2D imagePoint, IcyCanvas canvas) {

		if ( mode == Mode.EDIT_MODE )
		{
			createCell(imagePoint);

			canvas.getSequence().painterChanged( this );

			e.consume();
		}
	}
	
}
