
package plugins.fab.manualtnt.toremove; 

/** 
 * @(#) WaveletEstimatorConfig.java 
 * 2002/10/24    
 * 
 * @version 1.0 
 * @author Zhang Bo & Lionel Penard
 */ 

import java.util.HashMap; 

/** 
 * Configuration before estimation by using the operator 
 * of wavelet estimation. 
 */ 
public final class WaveletEstimatorConfig 
{ 
	/** one dimension */ 
	public static final int D1 = 1; 
	/** two dimension */ 
	public static final int D2 = 2; 
	/** three dimension */ 
	public static final int D3 = 3; 
	/* 
 	// Estimators' type - Jeffrey's Prior 
 	public static final int TYPE_JEFFREY = 0; 
 	// Estimators' type - Donoho's Hard  
 	public static final int TYPE_HARD = 1; 
 	// Estimators' type - Donoho's Soft 
 	public static final int TYPE_SOFT = 2;
    // Estimators' type - Correlation based
 	public static final int TYPE_CORR = 3;
	 */

	public static final int TYPE_JEFFREY = 0; 
	/** Estimators' type - Donoho's Hard */ 
	public static final int TYPE_HARD_CONST = 1;
	public static final int TYPE_HARD_VAR = 2;
	/** Estimators' type - Donoho's Soft */ 
	public static final int TYPE_SOFT_CONST = 3;
	public static final int TYPE_SOFT_VAR = 4;
	/** Estimators' type - Correlation based */
	public static final int TYPE_CORR = 5;
	public static final int TYPE_DEFAULT = TYPE_JEFFREY;

	public static final String[] TYPES_TAGS = {"Bayes - Jeffrey's prior", "Hard - Const. weight", "Hard - Variable weight", "Soft - Const. weight", "Soft - Variable weight", "Correlation based"};
	public static final boolean[] CONST_WEIGHTS = {false, true, false, true, false, false};
	private int dim;
	private int type; 
	private HashMap hashMap; // for the array of weights of scale(i) and band(j) 
	private int maxScale; // Max-th scale 

	/** construction with max scale = 0, 3 dimension and Jeffrey's Prior as the estimator's type */ 
	public WaveletEstimatorConfig() 
	{ 
		this(0); 
	} 

	/** construction in specifying the max scale, dimension is 3 and the type of estimator is Jeffrey */ 
	public WaveletEstimatorConfig(int maxScale) 
	{ 
		/* 3D Jeffrey as default */ 
		this(D3, TYPE_JEFFREY, maxScale); 
	} 

	/** construction in specifying the dimension, estimator's type and max scale */ 
	public WaveletEstimatorConfig(int dim, int type, int maxScale) 
	{ 
		if (dim < D1 || dim > D3)  
 		    throw new IllegalArgumentException("Dimension doesn't exist"); 
 		if (type < TYPE_JEFFREY || type > TYPE_CORR ) 
 		    throw new IllegalArgumentException("Type of estimator doesn't exist"); 
 		if (maxScale < 0) 
 		    throw new IllegalArgumentException("Max-th scale cannot be less than zero");     
 		this.dim = dim; 
 		this.type = type; 
 		if (type != TYPE_JEFFREY) hashMap = new HashMap(); 
 		else hashMap = null; 
 		this.maxScale = maxScale;
	} 

	public int getDimension() 
	{ 
		return dim; 
	} 

	public int getType() 
	{ 
		return type; 
	} 

	public int getMaxScale() 
	{ 
		return maxScale; 
	} 

	public void setMaxScale(int s) 
	{ 
		if (s < 0) throw new IllegalArgumentException("Scale must be >= 0"); 
		maxScale = s; 
	} 

	public void setType(int type) 
	{ 
		if (this.type == type) return; 
		switch (type) 
		{ 
		case TYPE_JEFFREY:  
			hashMap = null; 
			break; 

		case TYPE_HARD_VAR:
		case TYPE_SOFT_VAR:
		case TYPE_HARD_CONST:
		case TYPE_SOFT_CONST:
		case TYPE_CORR:
			hashMap = new HashMap(); 
			break; 
		default: 
			throw new IllegalArgumentException("Type of estimator doesn't exist"); 
		}
	} 
	/** get the weight of threshold: k(scale,band)*sigma(scale,band) */ 
	public double getWeights(_SBKey _sbk) 
	{ 
		int scale = _sbk.getScale(); int band = _sbk.getBand(); 
		if (type == TYPE_JEFFREY)
			throw new IllegalArgumentException("getWeights() works only with Threshold estimator"); 
		if (scale < 0 || scale > maxScale || band < 0 || band > Math.pow(2, dim)-1) 
			throw new IllegalArgumentException("scale must >= 0 <= "+maxScale+" and band must be between 0 and " 
					+((int)Math.pow(2,dim)-1)); 
		if (!hashMap.containsKey(_sbk)) 
			return Math.sqrt(2*Math.log(7*maxScale+1)); //Donoho's threshold 
		return ((Double)hashMap.get(_sbk)).doubleValue(); 
	} 

	/** remove the weight of threshold: k(scale,band)*sigma(scale,band) */ 
	public void removeWeights(_SBKey _sbk) 
	{ 
		if (type == TYPE_JEFFREY) 
			throw new IllegalArgumentException("setWeights() works only with Threshold estimator"); 

		if (hashMap.containsKey(_sbk)) hashMap.remove(_sbk);
	} 

	/** set the weight of threshold: k(scale,band)*sigma(scale,band) */ 
	public void setWeights(_SBKey _sbk, double weight)
	{ 
		int scale = _sbk.getScale(); int band = _sbk.getBand(); 
		if (type == TYPE_JEFFREY) 
			throw new IllegalArgumentException("setWeights() works only with Threshold estimator"); 
		if (scale < 0 || scale > maxScale || band < 0 || band > Math.pow(2, dim)-1) 
			throw new IllegalArgumentException("scale must >= 0 <= "+maxScale+" and band must be between 0 and " 
					+((int)Math.pow(2,dim)-1)); 
		removeWeights(_sbk); // if already exists 
		hashMap.put(_sbk, new Double(weight));  
	}  
} 
