package plugins.worm.waveletfilter;
import java.util.Arrays;

/*
Copyright (C) 2010

Wavelet Class
written by Max Bügler
http://www.maxbuegler.eu/

Wavelet class is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

Wavelet class is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/


/*
    Daub3-Daub10 coefficients aren't very precise here.
    Generally there are some minor rounding errors to be expected.

 */
public class Wavelet {

    //Wavelet defintions
        //Daubechies Wavelets
        private static final double[] daub1={1,1};
        private static final double[] daub2={0.6830127,1.1830127,0.3169873,-0.1830127019};
        private static final double[] daub3={0.47,1.14,0.65,-0.19093442,-0.12083221,0.05};
        private static final double[] daub4={0.33,1.01,0.89,-0.03967503,-0.26450717,0.04,0.05,-0.01498699};
        private static final double[] daub5={0.23,0.85,1.02,0.2,-0.34265671,-0.04560113,0.11,-0.00882680,-0.01779187,4.71742793e-3};
        private static final double[] daub6={0.16,0.7,1.06,0.45,-0.31998660,-0.18351806,0.14,0.04,-0.04466375,7.83251152e-4,6.75606236e-3,-1.52353381e-3};
        private static final double[] daub7={0.11,0.56,1.03,0.66,-0.20351382,-0.31683501,0.1,0.11,-0.05378245,-0.02343994,0.02,6.07514995e-4,-2.54790472e-3,5.00226853e-4};
        private static final double[] daub8={0.08,0.44,0.96,0.83,-0.02238574,-0.40165863,6.68194092e-4,0.18,-0.02456390,-0.06235021,0.02,0.01,-6.88771926e-3,-5.54004549e-4,9.55229711e-4,-1.66137261e-4};
        private static final double[] daub9={0.05,0.34,0.86,0.93,0.19,-0.41475176,-0.13695355,0.21,0.04,-0.09564726,3.54892813e-4,0.03,-6.67962023e-3,-6.05496058e-3,2.61296728e-3,3.25814671e-4,-3.56329759e-4,-5.5645514e-5};
        private static final double[] daub10={0.04,0.27,0.75,0.97,0.4,-0.35333620,-0.27710988,0.18,0.13,-0.10096657,-0.04165925,0.05,5.10043697e-3,-0.01517900,1.97332536e-3,2.81768659e-3,-9.69947840e-4,-1.64709006e-4,1.32354367e-4,-1.875841e-5};
        //all / sqrt(2)

        //Coiflets
        private static final double[] coif6={-0.1028594569415,0.4778594569415,1.2057189138831,0.5442810861169,-0.1028594569415,-0.0221405430585};
        private static final double[] coif12={0.0231751934774,-0.0586402759669,-0.0952791806220,0.5460420930695,1.1493647877137,0.5897343873912,-0.1081712141834,-0.0840529609215,0.0334888203266,0.0079357672259,-0.0025784067123,-0.0010190107982};
        private static final double[] coif18={-0.0053648373418,0.0110062534157,0.0331671209583,-0.0930155289575,-0.0864415271204,0.5730066705473,1.1225705137407,0.6059671435456,-0.1015402815098,-0.1163925015232,0.0488681886423,0.0224584819241,-0.0127392020221,-0.0036409178311,0.0015804102019,0.0006593303476,-0.0001003855491,-0.0000489314685};
        private static final double[] coif24={0.0012619224229,-0.0023044502875,-0.0103890503269,0.0227249229665,0.0377344771391,-0.1149284838039,-0.0793053059249,0.5873348100322,1.1062529100791,0.6143146193358,-0.0942254750478,-0.1360762293560,0.0556272739169,0.0354716628454,-0.0215126323102,-0.0080020216899,0.0053053298271,0.0017911878554,-0.0008330003902,-0.0003676592334,0.0000881604532,0.0000441656938,-0.0000046098383,-0.0000025243584};
        private static final double[] coif30={-0.0002999290457,0.0005071055047,0.0030805734520,-0.0058821563281,-0.0143282246988,0.0331043666130,0.0398380343960,-0.1299967565094,-0.0736051069489,0.5961918029174,1.0950165427081,0.6194005181568,-0.0877346296565,-0.1492888402657,0.0583893855506,0.0462091445541,-0.0279425853728,-0.0129534995030,0.0095622335983,0.0034387669688,-0.0023498958688,-0.0009016444801,0.0004268915950,0.0001984938228,-0.0000582936878,-0.0000300806360,0.0000052336193,0.0000029150058,-0.0000002296399};
        //all / sqrt(2)

    //Integer constants
    public static final int WAVELET_DB1=0;
    public static final int WAVELET_DB2=1;
    public static final int WAVELET_DB3=2;
    public static final int WAVELET_DB4=3;
    public static final int WAVELET_DB5=4;
    public static final int WAVELET_DB6=5;
    public static final int WAVELET_DB7=6;
    public static final int WAVELET_DB8=7;
    public static final int WAVELET_DB9=8;
    public static final int WAVELET_DB10=9;
    public static final int WAVELET_COIF6=10;
    public static final int WAVELET_COIF12=11;
    public static final int WAVELET_COIF18=12;
    public static final int WAVELET_COIF24=13;
    public static final int WAVELET_COIF30=14;

    //private fields
    private double[] hp,lp; //high and low pass coefficients

    /**
     * Constructs wavelet of typ i
     * @param i wavelet index
     */
    public Wavelet(int i){
        double[] w=getWaveletScale(i);
        lp=new double[w.length];
        hp=new double[w.length];
        for (int x=0;x<w.length;x++){
            lp[x]=w[x]/Math.sqrt(2);
            hp[w.length-1-x]=Math.pow(-1,x+1)*lp[x];
        }
    }

    /**
     * Construct new Wavelet given high and low pass coefficients
     * @param lp low pass coefficients
     * @param hp high pass coefficients
     */
    public Wavelet(double[] lp, double[] hp) {
        this.hp = hp;
        this.lp = lp;
    }

    /**
     * Returns high pass coefficients
     * @return high pass coefficients
     */
    public double[] getHP() {
        return hp;
    }

    /**
     * Returns low pass coefficients
     * @return low pass coefficients
     */
    public double[] getLP() {
        return lp;
    }

    /**
     * Performs an n-level wavelet decomposition on the input signal
     * @param input double array
     * @param n level of decomposition
     * @return decomposed signal
     */
    public double[] decompose(double[] input, int n){
        input=addZeros(input,n);
        double[] dec=new double[input.length];
        for (int l=0;l<n&&input.length>=hp.length;l++){
            for (int x=0;x<input.length&&x/2<dec.length;x+=2){
                dec[x/2]=0;
                dec[(input.length/2)+(x/2)]=0;
                for (int y=0;y<lp.length;y++){
                    int x2=x+y;
                    if (x2>=input.length)x2-=input.length;//Wrap around
                    dec[x/2]+=input[x2]*lp[y];
                    dec[(input.length/2)+(x/2)]+=input[x2]*hp[y];
                }
            }
            input=new double[input.length/2];
            for (int x=0;x<input.length;x++){
                input[x]=dec[x];
            }
        }
        return dec;
    }

    /**
     * Reconstructs a signal given a n-level decomposition
     * @param input double array containing decomposition
     * @param n level of decomposition
     * @return reconstructed signal
     */
    public double[] reconstruct(double[] input, int n){
        double[] rec=new double[input.length];
        for(int x=0;x<rec.length;x++){
            rec[x]=input[x];
        }
        for (int l=0;l<n;l++){
            int width=(int)(input.length/Math.pow(2,n-l));

            //Upscaling
            double[] upl=new double[2*width];
            double[] uph=new double[2*width];
            for(int x=0;x<width;x++){
                upl[2*x+1]=0;
                uph[2*x+1]=0;
                upl[2*x]=rec[x];
                uph[2*x]=rec[width+x];
            }

            //Recomposition
            for (int x=0;x<2*width;x++){
                double invl=0;
                double invh=0;
                for (int y=0;y<lp.length;y++){
                    int x2=x-y;
                    if (x2<0)x2+=2*width;//Wrap around
                    if (x2>=2*width)x2-=2*width;//Wrap around
                    invl+=upl[x2]*lp[y];
                    invh+=uph[x2]*hp[y];
                }
                rec[x]=invl+invh;
            }
        }
        return rec;
    }


    /**
     * Creates a n-level decomposition of a 2D input signal
     * @param input 2D input signal
     * @param n level of decomposition
     * @return decomposed signal
     */
    public double[][] decompose2D(double[][] input, int n){
        input=addZeros(input, n);
        double[][] dec=new double[input.length][input[0].length];
        for (int l=0;l< n &&input.length>=hp.length&&input[0].length>=hp.length;l++){
            for (int x=0;x<input.length;x++){
                for (int y=0;y<input[x].length;y+=2){
                    dec[x][y/2]=0;
                    dec[x][(input[x].length/2)+(y/2)]=0;
                    for (int z=0;z<lp.length;z++){
                        int y2=y+z;
                        if (y2>=input[x].length)y2-=input[x].length;//Wrap around
                        dec[x][y/2]+=input[x][y2]*lp[z];
                        dec[x][(input[x].length/2)+(y/2)]+=input[x][y2]*hp[z];
                    }
                }
            }
            for (int x=0;x<input.length;x++){
                for (int y=0;y<input[x].length;y++){
                    input[x][y]= dec[x][y];
                }
            }
            for (int y=0;y<input[0].length;y++){
                for (int x=0;x<input.length;x+=2){
                    dec[x/2][y]=0;
                    dec[(input.length/2)+(x/2)][y]=0;
                    for (int z=0;z<hp.length;z++){
                        int x2=x+z;
                        if (x2>=input.length)x2-=input.length;//Wrap around
                        dec[x/2][y]+=input[x2][y]*lp[z];
                        dec[(input.length/2)+(x/2)][y]+=input[x2][y]*hp[z];
                    }
                }
            }
            input=new double[input.length/2][input[0].length/2];
            for (int x=0;x<input.length;x++){
                for (int y=0;y<input[x].length;y++){
                    input[x][y]=dec[x][y];
                }
            }
            if (input.length<hp.length||input[0].length<hp.length) n =l+1;
        }
        return dec;
    }

    /**
     * Reconstruct a 2D signal given a n-level decomposition
     * @param input 2D array containing decomposition
     * @param n level of decomposition
     * @return reconstructed signal
     */
    public double[][] reconstruct2D(double[][] input, int n){
        double[][] out=new double[input.length][input[0].length];
        for (int x=0;x<out.length;x++){
            for (int y=0;y<out[x].length;y++){
                out[x][y]=input[x][y];
            }
        }

        for (int l=0;l< n;l++){
            int iw=(int)(input.length/Math.pow(2, n -l));
            int ih=(int)(input[0].length/Math.pow(2, n -l));
            double[][] upl=new double[2*iw][2*ih];
            double[][] uph=new double[2*iw][2*ih];
            for (int x=0;x<iw;x++){
                for (int y=0;y<2*ih;y++){
                    upl[2*x+1][y]=0;
                    uph[2*x+1][y]=0;
                    upl[2*x][y]=out[x][y];
                    uph[2*x][y]=out[iw+x][y];
                }
            }
            for (int x=0;x<2*iw;x++){
                for (int y=0;y<2*ih;y++){
                    double invl=0;
                    double invh=0;
                    for (int z=0;z<lp.length;z++){
                        int x2=x-z;
                        if (x2<0)x2+=2*iw;//Wrap around
                        invl+=upl[x2][y]*lp[z];
                        invh+=uph[x2][y]*hp[z];
                    }
                    out[x][y]=(invl+invh);
                }
            }

            for (int x=0;x<2*iw;x++){
                for (int y=0;y<ih;y++){
                    upl[x][2*y+1]=0;
                    uph[x][2*y+1]=0;
                    upl[x][2*y]=out[x][y];
                    uph[x][2*y]=out[x][ih+y];
                }
            }

            for (int y=0;y<2*ih;y++){
                for (int x=0;x<2*iw;x++){
                    double invl=0;
                    double invh=0;
                    for (int z=0;z<lp.length;z++){
                        int y2=y-z;
                        if (y2<0)y2+=2*ih;//Wrap around
                        invl+=upl[x][y2]*lp[z];
                        invh+=uph[x][y2]*hp[z];
                    }
                    out[x][y]=(invl+invh);
                }
            }
        }
        return out;
    }

    /**
     * Returns the optimal threshold according to the visushrink formula
     * @param input input signal
     * @param n level of decomposition
     * @return threshold value
     */
    public double getVisushrinkThreshold(double[] input, int n){
        int dc=0;
        for (int x=0;x<n;x++){
           dc+=Math.round(input.length*Math.pow(0.5,x+1));
        }
        double med=median(input);
        double[] mad=new double[input.length];
        for (int x=0;x<input.length;x++){
            mad[x]=Math.abs(input[x]-med);
        }
        double sigma=median(mad)/0.6745;
        return sigma*Math.sqrt(2*Math.log(dc));
    }

    /**
     * Apply soft/hard thresholding to the detail coefficients
     * @param input decomposed signal
     * @param n level of decomposition
     * @param threshold threshold
     * @param hard false=soft, true=hard
     * @return thresholded coefficients
     */
    public static double[] thresholdDetails(double[] input, int n, double threshold, boolean hard){
        int detailstart=(int)(input.length/Math.pow(2,n));

        double[] out=new double[input.length];

        for (int x=0;x<detailstart;x++){
            out[x]=input[x];
        }

        for (int x=detailstart;x<input.length;x++){
            double v=input[x];
            if (Math.abs(v)<=threshold||v==0){
                v=0;
            }
            else if (!hard)v-=Math.signum(v)*threshold; //Soft Thresholding
            out[x]=v;
        }
        
        return out;
    }

    /**
     * Apply soft/hard thresholding to the detail coefficients of a 2D signal
     * @param input decomposed 2D signal
     * @param n level of decomposition
     * @param threshold threshold
     * @param hard false=soft, true=hard
     * @return thresholded coefficients
     */
    public static double[][] thresholdDetails2D(double[][] input, int n, double threshold, boolean hard){
        int iw=(int)(input.length/Math.pow(2,n));
        int ih=(int)(input[0].length/Math.pow(2,n));
        double[][] out=new double[input.length][input[0].length];
        for (int x=0;x<iw;x++){
            for (int y=0;y<ih;y++){
                out[x][y]=input[x][y];
            }
        }
        long zeros=0;
        for (int x=iw;x<input.length;x++){ //Right side
            for (int y=0;y<input[x].length;y++){
                double v=input[x][y];
                if (Math.abs(v)<=threshold){
                    v=0;
                    zeros++;
                }
                else if (!hard)v-=Math.signum(v)*threshold; //Soft Thresholding
                out[x][y]=v;
            }
        }
        for (int x=0;x<iw;x++){ //Bottom left side
            for (int y=ih;y<input[x].length;y++){
                double v=input[x][y];
                if (Math.abs(v)<threshold){
                    v=0;
                    zeros++;
                }
                else if (!hard)v-=Math.signum(v)*threshold; //Soft Thresholding
                out[x][y]=v;
            }
        }
        return out;
    }

    /**
     * Fills up a signal with zeroes until it reaches sufficient length
     * @param input input signal
     * @param level desired level of decomposition
     * @return output signal
     */
    private double[] addZeros(double[] input, int level){
        int mult=lp.length*(int)Math.pow(2,level);
        if (input.length%mult==0)return input;
        int newlength=input.length+(mult-input.length%mult);
        double[] out=new double[newlength];
        for (int x=0;x<out.length;x++){
            if (x<input.length)
                out[x]=input[x];
            else
                out[x]=0;
        }
        return out;
    }

    /**
     * Fills up a 2D signal with zeroes until it reaches sufficient length
     * @param input input signal
     * @param level desired level of decomposition
     * @return output signal
     */
    private double[][] addZeros(double[][] input, int level){
        int mult=lp.length*(int)Math.pow(2,level);
        if (input.length%mult==0&&input[0].length%mult==0)return input;
        int newwidth=input.length+(mult-input.length%mult);
        int newheight=input[0].length+(mult-input[0].length%mult);
        double[][] out=new double[newwidth][newheight];
        for (int x=0;x<out.length;x++){
            for (int y=0;y<out[x].length;y++){
                if (x<input.length && y<input[0].length)
                    out[x][y]=input[x][y];
                else
                    out[x][y]=0;
            }
        }
        return out;
    }

    /**
     * Return the n-level filter coefficients of the wavelet
     * @param n level
     * @return Wavelet with n-level filter coefficients
     */
    Wavelet getWaveletLevel(int n){
        if (n<=1)return this;
        double[] l=lp;
        double[] h=hp;
        double[] l1=lp;
        for (int a=0;a<n-1;a++){
            int pow=(int)Math.pow(2,a+1);
            int nw=l.length+pow*(l1.length-1);
            double[] nl=new double[nw];
            for (int x=0;x<nl.length;x++){
                nl[x]=0;
            }
            for (int x=0;x<l.length;x++){
                for (int y=0;y<l1.length;y++){
                    nl[2*x+y]+=l1[y]*l[x];
                }
            }
            l=nl;
            double[] nh=new double[nw];
            for (int x=0;x<nl.length;x++){
                nh[x]=0;
            }
            for (int x=0;x<h.length;x++){
                for (int y=0;y<l1.length;y++){
                    nh[2*x+y]+=l1[y]*h[x];
                }
            }
            h=nh;
        }
        return new Wavelet(l,h);
    }

    /**
     * Return median of double array
     * @param in double array
     * @return median
     */
    private static double median(double[] in){
        Arrays.sort(in);
        if (in.length%2==1)return in[in.length/2];
        else return (in[in.length/2]+in[in.length/2-1])/2.0;
    }

    /**
     * Get wavelet for index i
     * @param i wavelet index
     * @return wavelet
     */
    public static double[] getWaveletScale(int i){
        switch (i){
            case WAVELET_DB1:
                return daub1;
            case WAVELET_DB2:
                return daub2;
            case WAVELET_DB3:
                return daub3;
            case WAVELET_DB4:
                return daub4;
            case WAVELET_DB5:
                return daub5;
            case WAVELET_DB6:
                return daub6;
            case WAVELET_DB7:
                return daub7;
            case WAVELET_DB8:
                return daub8;
            case WAVELET_DB9:
                return daub9;
            case WAVELET_DB10:
                return daub10;
            case WAVELET_COIF6:
                return coif6;
            case WAVELET_COIF12:
                return coif12;
            case WAVELET_COIF18:
                return coif18;
            case WAVELET_COIF24:
                return coif24;
            case WAVELET_COIF30:
                return coif30;

        }
        return daub1;
    }

    /**
     * Some tests
     * @param args not used
     */
    public static void main(String[] args){

        //Create new DB2 Wavelet
        Wavelet w=new Wavelet(WAVELET_DB2);

        //Get low and high pass filter coefficients and print them
        double[] l=w.getLP();
        double[] h=w.getHP();
        System.out.println("Low pass coefficients:");
        for (int x=0;x<l.length;x++){
            System.out.print(l[x]+" , ");
        }
        System.out.println();
        System.out.println("High pass coefficients:");
        for (int x=0;x<h.length;x++){
            System.out.print(h[x]+" , ");
        }
        System.out.println();

        //Select level of decomposition
        int level=2;

        //Get the x level filter coefficients and print them
        w=w.getWaveletLevel(level);
        l=w.getLP();
        h=w.getHP();
        System.out.println(level+"-level Low pass coefficients:");
        for (int x=0;x<l.length;x++){
            System.out.print(l[x]+" , ");
        }
        System.out.println();
        System.out.println(level+"-level High pass coefficients:");
        for (int x=0;x<h.length;x++){
            System.out.print(h[x]+" , ");
        }
        System.out.println();

        //Some 1D signal
        double[] in={1,2,3,4,5,6,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2};

        System.out.println("Input signal");
        for (int x=0;x<in.length;x++){
            System.out.print(toString(in[x]));
        }
        System.out.println();

        //Create new Coif12 wavelet
        Wavelet wl= new Wavelet(WAVELET_COIF12);

        //Select level of decomposition
        level=2;

        double[] dec=wl.decompose(in,level); //decompose signal and print

        System.out.println("Decomposed signal");
        for (int x=0;x<dec.length;x++){
            System.out.print(toString(dec[x]));
        }
        System.out.println();

        double[] rec=wl.reconstruct(dec,level); //reconstruct signal and print

        System.out.println("Reconstructed signal");
        for (int x=0;x<rec.length;x++){
            System.out.print(toString(rec[x]));
        }
        System.out.println();

        //Determine threshold using visushrink
        double t=wl.getVisushrinkThreshold(in,level);
        System.out.println("Visushrink threshold: "+t);

        //Threshold detail coefficients
        double[] dec2=thresholdDetails(dec,level,t,false);

        System.out.println("Thresholded Decomposition");
        for (int x=0;x<dec2.length;x++){
            System.out.print(toString(dec2[x]));
        }
        System.out.println();

        //Reconstruct signal
        double[] rec2=wl.reconstruct(dec2,level);
        System.out.println("Reconstructed signal");
        for (int x=0;x<rec2.length;x++){
            System.out.print(toString(rec2[x]));
        }
        System.out.println();

        //Some 2D input signal
        double[][] in2D={{1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2},
                    {1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2},
                    {1,1,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2},
                    {1,2,1,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2},
                    {1,2,3,1,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2},
                    {1,2,3,3,1,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2},
                    {1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2},
                    {1,2,3,3,2,1,1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2,1,2,3,3,2,1,-1,-3,-2,0,1,2,5,-1,2,-2},};

        System.out.println("2D input signal:");
        for (int y=0;y<in2D.length;y++){
            for (int x=0;x<in2D[0].length;x++){
                System.out.print(toString(in2D[y][x]));
            }
            System.out.println();
        }

        //Decompose signal
        double[][] dec2D=wl.decompose2D(in2D,level);

        System.out.println("Decomposed 2D signal:");
        for (int y=0;y<in2D.length;y++){
            for (int x=0;x<in2D[0].length;x++){
                System.out.print(toString(dec2D[y][x]));
            }
            System.out.println();
        }
        //Reconstruct signal

        double[][] rec2D=wl.reconstruct2D(dec2D,level);

        //Print
        System.out.println("Reconstructed 2D signal:");
        for (int y=0;y<in2D.length;y++){
            for (int x=0;x<in2D[0].length;x++){
                System.out.print(toString(rec2D[y][x]));
            }
            System.out.println();
        }

        //Threshold
        double[][] dec2Dt=thresholdDetails2D(dec2D,level,0.1,true);

        //Print
        System.out.println("Thresholded decomposition:");
        for (int y=0;y<in2D.length;y++){
            for (int x=0;x<in2D[0].length;x++){
                System.out.print(toString(dec2Dt[y][x]));
            }
            System.out.println();
        }

        double[][] rec2Dt=wl.reconstruct2D(dec2Dt,level);

        //Print
        System.out.println("Reconstructed 2D signal:");
        for (int y=0;y<in2D.length;y++){
            for (int x=0;x<in2D[0].length;x++){
                System.out.print(toString(rec2Dt[y][x]));
            }
            System.out.println();
        }


    }

    /**
     * Convert double to string for printing
     * @param v double
     * @return string
     */
    private static String toString(double v){
        int v2=(int)Math.round(v*100.0);
        String s=v2/100.0+"";
        if (v2%10==0)s+=" ";
        if (v2%100==0)s+=" ";
        if (v2>=0)return " "+s+"\t";
        return s+"\t";
    }
}

