package plugins.ylemontag.noisegenerator.noisemodels;

import icy.util.StringUtil;

import java.util.Random;

import plugins.ylemontag.noisegenerator.IllegalNoiseModelArgumentException;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Modelize a salt & pepper noise
 * 
 * The salt & pepper degrading process works as follows: each sample is let
 * unchanged with probability p=1-intensity, or, with probability p=intensity,
 * is replaced a value that is selected in a random uniform manner between
 * a lower bound and an upper bound.
 * 
 * For instance, if intensity==0.2, lowerBound==0 and upperBound==255, then
 * 20% of the samples will be replaced with a random values between 0 and 255,
 * while the remaining 80% will be let unchanged.
 */
public class SaltPepperNoise extends PixelSeparableStationaryNoise
{
	private double _intensity ;
	private double _lowerBound;
	private double _upperBound;
	private Random _gen       ;
	
	/**
	 * Constructor
	 * @throws IllegalNoiseModelArgumentException If intensity<0 or intensity>1
	 * @throws IllegalNoiseModelArgumentException If lowerBound>upperBound
	 */
	public SaltPepperNoise(double intensity, double lowerBound, double upperBound)
	{
		if(intensity<0 || intensity>1) {
			throw new IllegalNoiseModelArgumentException(
				"The intensity parameter of a salt & pepper noise must be valued between 0 and 1."
			);
		}
		if(lowerBound>upperBound) {
			throw new IllegalNoiseModelArgumentException(
				"The upper bound parameter of a salt & pepper noise must be greater that its lower bound."
			);
		}
		_intensity  = intensity ;
		_lowerBound = lowerBound;
		_upperBound = upperBound;
		_gen        = new Random();
	}
	
	@Override
	protected double generateNoise(double in)
	{
		double replacementValue = _lowerBound + (_upperBound - _lowerBound)*_gen.nextDouble();
		double replacementCond  = _gen.nextDouble();
		return replacementCond<_intensity ? replacementValue : in;
	}
	
	@Override
	protected String describeNoise(String inName)
	{
		return inName + " + S&P(" + StringUtil.toString(_intensity*100) + "%)";
	}
}
