/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Zsuzsanna Puspoki (zsuzsanna.puspoki@epfl.ch)
 ******************************************************************************/
package plugins.big.steerablej;

import icy.main.Icy;
import icy.plugin.abstract_.PluginActionable;
import icy.sequence.Sequence;
import plugins.big.steerablej.core.AlphaValues;
import plugins.big.steerablej.core.ParameterSet;
import plugins.big.steerablej.gui.MainDialog;
import plugins.big.steerablej.process.Image2ArrayConverter;
import plugins.big.steerablej.process.SteerableDetector;

/**
 * Plug-in that implements 2D feature detectors. These detectors are a class of
 * steerable functions based on the optimization of a Canny-like criterion. In
 * contrast with previous computational designs, this approach is truly 2D and
 * provides filters that have closed-form expressions. It also yields operators
 * that have a better orientation selectivity than the classical gradient or
 * Hessian-based detectors.
 * 
 * @version April 22, 2013
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Zsuzsanna Puspoki (zsuzsanna.puspoki@epfl.ch)
 */
public class FeatureDetector extends PluginActionable {

	/** Main interface dialog. */
	private MainDialog mainDialog_ = null;

	/** Steerable filter detector. */
	private SteerableDetector detector_ = null;
	/** Thread where the analysis is performed. */
	private Thread detectorThread_ = null;

	/** Stores all parameters used for the steerable templates. */
	private ParameterSet[][] pSet_ = null;
	/**
	 * Array of weights that control the linear combination of Gaussian
	 * derivatives which constitute the feature template.
	 */
	private AlphaValues alphaValues_ = null;
	/** Order of the feature template. */
	private int M_ = 1;
	/**
	 * Standard deviation of the Gaussian on which the steerable templates are
	 * based.
	 */
	private double sigma_ = 2.0;

	// ============================================================================
	// PUBLIC METHODS

	/**
	 * First method executed when the plug-in is called. It builds the main
	 * interface and loads the parameters.
	 */
	@Override
	public void run() {
		alphaValues_ = new AlphaValues();
		pSet_ = ParameterSet.loadParameterSets(this.getClass()
				.getResourceAsStream("rsc/parameters.xml"));
		alphaValues_.setDefault(M_, 1, pSet_);
		mainDialog_ = new MainDialog(this);
	}

	// ----------------------------------------------------------------------------

	/** Filters the focused image. */
	public void processActiveImage() {
		Sequence focusedSequence = getFocusedSequence();
		if (focusedSequence == null) {
			System.err.println("No image opened.");
			return;
		}

		// focusedSequence.get

		double[] slice = Image2ArrayConverter.seqToDoubleArray(focusedSequence,
				0);

		detector_ = new SteerableDetector(slice, focusedSequence.getSizeX(),
				focusedSequence.getSizeY(), focusedSequence.getSizeT(), sigma_,
				M_, alphaValues_.getAlpha(1, sigma_, M_));

		Thread filteringThread = new Thread() {
			@Override
			public void run() {
				detectorThread_ = new Thread(detector_);
				detectorThread_.setPriority(Thread.MIN_PRIORITY);
				detectorThread_.start();
				try {
					detectorThread_.join();
				} catch (InterruptedException ie) {
				}
				if (!detector_.getStop()) {
					showAllResults();
				}
				mainDialog_.detectorFinished();
			}
		};
		filteringThread.setPriority(Thread.MIN_PRIORITY);
		filteringThread.start();
	}

	// ----------------------------------------------------------------------------
	// METHODS TO MANAGE THE WORKFLOW OF THE PLUG-IN

	/** Returns <code>true</code> if the detector is running. */
	public boolean isRunning() {
		return !(detectorThread_ == null || !detectorThread_.isAlive());
	}

	// ----------------------------------------------------------------------------

	/** Stops the detector and released the memory associated with it. */
	public void killDetector() {
		if (detector_ != null) {
			stopDetector();
			detector_ = null;
		}
	}

	// ----------------------------------------------------------------------------

	/** Stops the detector. */
	public void stopDetector() {
		detector_.stop();
	}

	// ----------------------------------------------------------------------------
	// DISPLAY RESULTS METHODS

	/** Displays the filtered image (magnitude and phase). */
	public void showAllResults() {
		Sequence orientation = detector_.getOrientation();
		orientation.setName("Orientation map");
		Icy.getMainInterface().addSequence(orientation);

		Sequence response = detector_.getResponse();
		response.setName("Response");
		Icy.getMainInterface().addSequence(response);
	}

	// ----------------------------------------------------------------------------

	/**
	 * Computes and displays the response of the filter after suppressing the
	 * non-maxima elements.
	 */
	public void showNMS() {
		Sequence NMSSequence = detector_.computeNMS();
		NMSSequence.setName("Non maxima supression");
		Icy.getMainInterface().addSequence(NMSSequence);
	}

	// ----------------------------------------------------------------------------

	/**
	 * Displays a color-coded version of the local orientation of the features
	 * detected by the filter.
	 */
	public void showOrientations() {
		Sequence colorOrientationSequence = detector_.computeColorOrientation();
		colorOrientationSequence.setName("Orientation map");
		Icy.getMainInterface().addSequence(colorOrientationSequence);
	}

	// ----------------------------------------------------------------------------

	/**
	 * Computes and displays the response to rotations of the feature template
	 * over 2 pi, in (2*pi)/nIncrements increments.
	 */
	public void showRotations(int numRotations) {
		Sequence rotationSequence = detector_.computeRotations(numRotations);
		rotationSequence.setName("Rotations");
		Icy.getMainInterface().addSequence(rotationSequence);
	}

	// ============================================================================
	// GETTERS AND SETTERS

	/**
	 * Returns the array of weights that control the linear combination of
	 * Gaussian derivatives up to order M which constitute the feature template.
	 * For more information, see the documentation for the
	 * <code>ParameterSet</code> class.
	 * 
	 */
	public AlphaValues getAlpha() {
		return alphaValues_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns the order of the feature template. Order up to 5 are supported in
	 * the current implementation.
	 */
	public int getM() {
		return M_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Sets the order of the feature template. Order up to 5 are supported in
	 * the current implementation.
	 */
	public void setM(int M) {
		M_ = M;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Gets the all the sets of parameters used in the
	 * <code>FeatureDetector</code> class.
	 */
	public ParameterSet[][] getParameterSet() {
		return pSet_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns the standard deviation of the Gaussian on which the steerable
	 * templates are based. This controls the feature width.
	 */
	public double getSigma() {
		return sigma_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Sets the standard deviation of the Gaussian on which the steerable
	 * templates are based. This controls the feature width.
	 */
	public void setSigma(double sigma) {
		sigma_ = sigma;
	}
}
