package plugins.tprovoost.box3d;

import icy.canvas.Canvas3D;
import icy.canvas.IcyCanvas;
import icy.math.UnitUtil;
import icy.math.UnitUtil.UnitPrefix;
import icy.painter.Overlay;
import icy.painter.VtkPainter;
import icy.sequence.Sequence;

import java.awt.Color;
import java.awt.Graphics2D;

import vtk.vtkAxesActor;
import vtk.vtkCamera;
import vtk.vtkCubeAxesActor;
import vtk.vtkOrientationMarkerWidget;
import vtk.vtkProp;
import vtk.vtkRenderWindowInteractor;
import vtk.vtkRenderer;
import vtk.vtkTransform;

/**
 * @author stephane
 */
public class BBoxOverlay extends Overlay implements VtkPainter
{
	private static final double SMALL_VALUE = 0.00001d;
	// vtk object
	private vtkAxesActor axesForWidget;

	private vtkAxesActor axesAbsolute;

	private vtkCubeAxesActor axesValues;
	private vtkCubeAxesActor boundingCube;
	private vtkCubeAxesActor grid1;
	private vtkCubeAxesActor grid2;
	private vtkCubeAxesActor grid3;
	private vtkCubeAxesActor grid4;
	private vtkCubeAxesActor grid5;
	private vtkCubeAxesActor grid6;

	private vtkOrientationMarkerWidget widget;

	public BBoxOverlay(final Sequence s)
	{
		super("Bounding Box");
		init(s);
	}

	// init vtk objects
	private void init(Sequence s)
	{
		double seqZ = s.getSizeZ() * s.getPixelSizeZ();
		double seqW = s.getWidth() * s.getPixelSizeX();
		double seqH = s.getHeight() * s.getPixelSizeY();

		// ---------
		// AXES WIDGET
		// ---------
		axesForWidget = new vtkAxesActor();
		widget = new vtkOrientationMarkerWidget();
		widget.SetOrientationMarker(axesForWidget);

		// -------
		// AXES
		// ------
		axesAbsolute = new vtkAxesActor();
		axesAbsolute.SetTotalLength(seqW, seqH, seqZ);
		axesAbsolute.SetShaftTypeToLine();
		axesAbsolute.SetTipTypeToCone();
		axesAbsolute.SetConeRadius(0.0001);

		axesValues = new vtkCubeAxesActor();
		axesValues.GetProperty().SetColor(0.8, 0.8, 0.8);
		axesValues.SetBounds(0, seqW, 0, seqH, 0, seqZ);

		// ---------
		// BOUNDING BOX
		// ---------
		boundingCube = new vtkCubeAxesActor();
		boundingCube.GetProperty().SetColor(0.8, 0.8, 0.8);
		boundingCube.SetBounds(0, seqW, 0, seqH, 0, seqZ);

		// ---------
		// GRIDs
		// ---------
		grid1 = new vtkCubeAxesActor();
		grid1.GetProperty().SetColor(0.8, 0.8, 0.8);
		grid1.SetBounds(0, SMALL_VALUE, 0, seqH, 0, seqZ);

		grid2 = new vtkCubeAxesActor();
		grid2.GetProperty().SetColor(0.8, 0.8, 0.8);
		grid2.SetBounds(0, seqW, 0, SMALL_VALUE, 0, seqZ);

		grid3 = new vtkCubeAxesActor();
		grid3.GetProperty().SetColor(0.8, 0.8, 0.8);
		grid3.SetBounds(0, seqW, 0, seqH, 0, SMALL_VALUE);

		grid4 = new vtkCubeAxesActor();
		grid4.GetProperty().SetColor(0.8, 0.8, 0.8);
		grid4.SetBounds(seqW - SMALL_VALUE, seqW, 0, seqH, 0, seqZ);

		grid5 = new vtkCubeAxesActor();
		grid5.GetProperty().SetColor(0.8, 0.8, 0.8);
		grid5.SetBounds(0, seqW, seqH - SMALL_VALUE, seqH, 0, seqZ);

		grid6 = new vtkCubeAxesActor();
		grid6.GetProperty().SetColor(0.8, 0.8, 0.8);
		grid6.SetBounds(0, seqW, 0, seqH, seqZ - SMALL_VALUE, seqZ);
	}

	@Override
	public vtkProp[] getProps()
	{
		return new vtkProp[]
		{ axesAbsolute, axesValues, boundingCube, grid1, grid2, grid3, grid4, grid5, grid6 };
	}

	@Override
	public void paint(Graphics2D g, Sequence s, IcyCanvas canvas)
	{
		if (canvas instanceof Canvas3D)
		{
			Box3dProps props = Box3dProps.getInstance();
			Color c = props.getColor();

			vtkRenderer renderer = ((Canvas3D) canvas).getPanel3D().GetRenderer();
			vtkCamera camera = renderer.GetActiveCamera();
			if (boundingCube.GetCamera() != camera)
			{
				initCamera(boundingCube, camera);
				initCamera(axesValues, camera);

				vtkRenderWindowInteractor interactor = new vtkRenderWindowInteractor();
				interactor.SetRenderWindow(renderer.GetRenderWindow());
				if (widget.GetInteractor() != null)
					widget.EnabledOff();
				widget.SetInteractor(interactor);
				widget.SetOutlineColor(0.93, 0.57, 0.13);
				widget.SetViewport(0, 0, 0.2, 0.2);
				widget.EnabledOn();
				widget.InteractiveOn();

				// initCamera(axesTriad, camera);
				initCamera(grid1, camera);
				initCamera(grid2, camera);
				initCamera(grid3, camera);
				initCamera(grid4, camera);
				initCamera(grid5, camera);
				initCamera(grid6, camera);
			}
			axesAbsolute.AxisLabelsOff();
			axesAbsolute.SetTipTypeToCone();
			axesAbsolute.SetShaftTypeToLine();
			axesAbsolute.SetConeRadius(0.0001);
			axesAbsolute.SetShaftTypeToCylinder();
			axesAbsolute.SetCylinderRadius(0.002);

			axesForWidget.AxisLabelsOn();
			axesForWidget.SetTipTypeToCone();
			axesForWidget.SetXAxisLabelText("x");
			axesForWidget.SetYAxisLabelText("y");
			axesForWidget.SetZAxisLabelText("z");

			boundingCube.SetFlyModeToStaticEdges();
			boundingCube.SetXTitle("");
			boundingCube.SetYTitle("");
			boundingCube.SetZTitle("");
			boundingCube.XAxisLabelVisibilityOff();
			boundingCube.YAxisLabelVisibilityOff();
			boundingCube.ZAxisLabelVisibilityOff();
			boundingCube.XAxisTickVisibilityOff();
			boundingCube.YAxisTickVisibilityOff();
			boundingCube.ZAxisTickVisibilityOff();
			boundingCube.SetTickLocationToInside();
			boundingCube.SetUseBounds(true);
			boundingCube.GetProperty().SetColor(c.getRed() / 255d, c.getGreen() / 255d, c.getBlue() / 255d);

			axesValues.XAxisLabelVisibilityOn();
			axesValues.YAxisLabelVisibilityOn();
			axesValues.ZAxisLabelVisibilityOn();
			axesValues.XAxisTickVisibilityOn();
			axesValues.YAxisTickVisibilityOn();
			axesValues.ZAxisTickVisibilityOn();
			axesValues.SetXTitle("");
			axesValues.SetYTitle("");
			axesValues.SetZTitle("");
			axesValues.GetProperty().SetColor(c.getRed() / 255d, c.getGreen() / 255d, c.getBlue() / 255d);

			if (props.areAxesAbsolute())
			{
				axesAbsolute.VisibilityOn();
				axesValues.SetFlyModeToStaticTriad();
			} else
			{
				axesAbsolute.VisibilityOff();
				axesValues.SetFlyModeToOuterEdges();
			}
			if (props.isBBoxVisible())
			{
				boundingCube.VisibilityOn();
			} else
			{
				boundingCube.VisibilityOff();
			}
			if (props.areAxesValuesVisible())
			{
				axesValues.VisibilityOn();
			} else
			{
				axesValues.VisibilityOff();
			}

			boundingCube.XAxisMinorTickVisibilityOff();
			boundingCube.YAxisMinorTickVisibilityOff();
			boundingCube.ZAxisMinorTickVisibilityOff();

			adaptValues(grid1, s);
			adaptValues(grid2, s);
			adaptValues(grid3, s);
			adaptValues(grid4, s);
			adaptValues(grid5, s);
			adaptValues(grid6, s);

			setAllGridsVisibility(props.isGridVisible());

			setGrid(grid1);
			setGrid(grid2);
			setGrid(grid3);
			setGrid(grid4);
			setGrid(grid5);
			setGrid(grid6);
		}
	}

	private void initCamera(vtkCubeAxesActor actor, vtkCamera camera)
	{
		actor.SetCamera(camera);
	}

	private void setGrid(vtkCubeAxesActor actor)
	{
		Box3dProps props = Box3dProps.getInstance();
		Color c = props.getColor();

		actor.SetFlyModeToClosestTriad();

		actor.SetTickLocationToInside();

		actor.XAxisLabelVisibilityOff();
		actor.YAxisLabelVisibilityOff();
		actor.ZAxisLabelVisibilityOff();

		actor.DrawXGridlinesOn();
		actor.DrawYGridlinesOn();
		actor.DrawZGridlinesOn();

		actor.XAxisMinorTickVisibilityOff();
		actor.YAxisMinorTickVisibilityOff();
		actor.ZAxisMinorTickVisibilityOff();

		actor.XAxisTickVisibilityOn();
		actor.YAxisTickVisibilityOn();
		actor.ZAxisTickVisibilityOn();

		actor.GetProperty().SetColor(c.getRed() / 255d, c.getGreen() / 255d, c.getBlue() / 255d);
	}

	private void adaptValues(vtkCubeAxesActor actor, Sequence s)
	{
		double seqZ = s.getSizeZ() * s.getPixelSizeZ();
		double seqW = s.getWidth() * s.getPixelSizeX();
		double seqH = s.getHeight() * s.getPixelSizeY();

		UnitPrefix xunit = UnitUtil.getBestUnit(s.getPixelSizeX(), UnitPrefix.MICRO);
		if (xunit == UnitPrefix.MICRO)
			actor.SetXUnits("" + xunit + "m");
		else
			actor.SetXUnits("um");

		UnitPrefix yunit = UnitUtil.getBestUnit(s.getPixelSizeY(), UnitPrefix.MICRO);
		if (yunit == UnitPrefix.MICRO)
			actor.SetYUnits("" + yunit + "m");
		else
			actor.SetYUnits("um");

		UnitPrefix zunit = UnitUtil.getBestUnit(s.getPixelSizeZ(), UnitPrefix.MICRO);
		if (zunit == UnitPrefix.MICRO)
			actor.SetZUnits("" + zunit + "m");
		else
			actor.SetZUnits("um");

		actor.SetXAxisRange(0, seqW);
		actor.SetYAxisRange(0, seqH);
		actor.SetZAxisRange(0, seqZ);

	}

	/**
	 * 
	 * @param defaultVisibility
	 *            : if false, never displays the grid
	 */
	private void setAllGridsVisibility(boolean defaultVisibility)
	{
		vtkCamera camera = grid1.GetCamera();
		vtkTransform matrix = camera.GetViewTransformObject();

		double[] norm = new double[]
		{ 1, 0, 0 };
		norm = matrix.TransformVector(norm);

		int visible = norm[2] >= 0 && defaultVisibility ? 1 : 0;
		grid1.SetVisibility(visible);
		// if (visible == 1)
		// {
		// grid1.YAxisLabelVisibilityOn();
		// grid1.ZAxisLabelVisibilityOn();
		// grid1.ZAxisLabelVisibilityOff();
		// } else
		// {
		// grid1.YAxisLabelVisibilityOff();
		// grid1.ZAxisLabelVisibilityOff();
		// grid1.ZAxisLabelVisibilityOff();
		// }

		norm = new double[]
		{ 0, 1, 0 };
		norm = matrix.TransformVector(norm);

		visible = norm[2] >= 0 && defaultVisibility ? 1 : 0;
		grid2.SetVisibility(visible);
		// if (visible == 1)
		// {
		// grid2.XAxisLabelVisibilityOn();
		// grid2.ZAxisLabelVisibilityOn();
		// grid2.ZAxisLabelVisibilityOff();
		// } else
		// {
		// grid2.XAxisLabelVisibilityOff();
		// grid2.ZAxisLabelVisibilityOff();
		// grid2.ZAxisLabelVisibilityOff();
		// }

		norm = new double[]
		{ 0, 0, 1 };
		norm = matrix.TransformVector(norm);

		visible = norm[2] >= 0 && defaultVisibility ? 1 : 0;
		grid3.SetVisibility(visible);
		// if (visible == 1)
		// {
		// grid3.XAxisLabelVisibilityOn();
		// grid3.YAxisLabelVisibilityOn();
		// grid3.ZAxisLabelVisibilityOn();
		// } else
		// {
		// grid3.XAxisLabelVisibilityOff();
		// grid3.YAxisLabelVisibilityOff();
		// grid3.ZAxisLabelVisibilityOff();
		// }

		norm = new double[]
		{ -1, 0, 0 };
		norm = matrix.TransformVector(norm);

		visible = norm[2] >= 0 && defaultVisibility ? 1 : 0;
		grid4.SetVisibility(visible);
		// if (visible == 1)
		// {
		// grid4.XAxisLabelVisibilityOn();
		// grid4.YAxisLabelVisibilityOn();
		// grid4.ZAxisLabelVisibilityOn();
		// } else
		// {
		// grid4.XAxisLabelVisibilityOff();
		// grid4.YAxisLabelVisibilityOff();
		// grid4.ZAxisLabelVisibilityOff();
		// }

		norm = new double[]
		{ 0, -1, 0 };
		norm = matrix.TransformVector(norm);

		visible = norm[2] >= 0 && defaultVisibility ? 1 : 0;
		grid5.SetVisibility(visible);
		// if (visible == 1)
		// {
		// grid5.XAxisLabelVisibilityOn();
		// grid5.YAxisLabelVisibilityOn();
		// grid5.ZAxisLabelVisibilityOn();
		// } else
		// {
		// grid5.XAxisLabelVisibilityOff();
		// grid5.YAxisLabelVisibilityOff();
		// grid5.ZAxisLabelVisibilityOff();
		// }

		norm = new double[]
		{ 0, 0, -1 };
		norm = matrix.TransformVector(norm);

		visible = norm[2] >= 0 && defaultVisibility ? 1 : 0;
		grid6.SetVisibility(visible);
		// if (visible == 1)
		// {
		// grid6.XAxisLabelVisibilityOn();
		// grid6.YAxisLabelVisibilityOn();
		// grid6.ZAxisLabelVisibilityOn();
		// } else
		// {
		// grid6.XAxisLabelVisibilityOff();
		// grid6.YAxisLabelVisibilityOff();
		// grid6.ZAxisLabelVisibilityOff();
		// }
	}
}
