/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobplugin.operations;

import plugins.big.blobplugin.descriptors.ImageDescriptor;

/**
 * @class Filters
 * 
 *        Abstract class implementing some basic filtering operations.
 * 
 */
public abstract class Filters {
	/**
	 * Apply a gaussian blur on a 2D or 3D image.
	 * 
	 * @param src
	 *            The input image
	 * @param dst
	 *            The output image. Must have the same size as src.
	 * @param sigma
	 *            Standard deviation of the gaussian filter.
	 * 
	 * @throws Exception
	 *             Throw an exception if src and dst have not the same size.
	 */
	public static void gaussianBlur(final ImageDescriptor src,
			ImageDescriptor dst, double sigma) throws Exception {
		if (src.width != dst.width || src.height != dst.height
				|| src.depth != dst.depth) {
			throw new Exception(
					"src and dst must have the same size in Filters.gaussianBlur");
		}

		double[] kernel = new double[7];
		double sum = 0;
		for (int i = 0; i < 7; i++) {
			sum += kernel[i] = Math.exp(-(i - 3) * (i - 3)
					/ (2 * sigma * sigma));
		}
		for (int i = 0; i < 7; i++) {
			kernel[i] /= sum;
		}

		filter(src, dst, kernel, kernel, kernel);
	}

	/**
	 * Compute the gradient of an image.
	 * 
	 * @param src
	 *            The input image
	 * @param Fx
	 *            Derivative along the x direction.
	 * @param Fy
	 *            Derivative along the y direction.
	 * @param Fz
	 *            Derivative along the z direction if it's a 3D image.
	 * 
	 * @throws Exception
	 *             Throw an exception if src, Fx, Fy (and Fz) have not the same
	 *             size.
	 */
	static public void gradient(final ImageDescriptor src, ImageDescriptor Fx,
			ImageDescriptor Fy, ImageDescriptor Fz) throws Exception {
		if (src.width != Fx.width
				|| src.height != Fx.height
				|| src.depth != Fx.depth
				|| src.width != Fy.width
				|| src.height != Fy.height
				|| src.depth != Fy.depth
				|| (src.depth > 1 && (src.width != Fz.width
						|| src.height != Fz.height || src.depth != Fz.depth))) {
			throw new Exception(
					"src, Fx, Fy (and Fz) must have the same size in Filters.gradient");
		}

		double[] kernel1 = new double[] { -0.5, 0.0, 0.5 };
		double[] kernel2 = new double[] { 0.25, 0.5, 0.25 };

		filter(src, Fx, kernel1, kernel2, kernel2);
		filter(src, Fy, kernel2, kernel1, kernel2);
		if (src.depth > 1) {
			filter(src, Fz, kernel2, kernel2, kernel1);
		}
	}

	/**
	 * Compute the gradient direction.
	 * 
	 * @param Fx
	 *            Derivative along the x direction.
	 * @param Fy
	 *            Derivative along the y direction.
	 * @param Fz
	 *            Derivative along the z direction if it's a 3D image.
	 * @param Alpha
	 *            Orientation of the gradient from x and y directions.
	 * @param Beta
	 *            Orientation of the gradient from z and y directions.
	 * @param Gamma
	 *            Orientation of the gradient from x and z directions.
	 * 
	 * @throws Exception
	 *             Throw an exception if the images have not the same size.
	 */
	static public void gradientDirection(final ImageDescriptor Fx,
			final ImageDescriptor Fy, final ImageDescriptor Fz,
			ImageDescriptor Alpha, ImageDescriptor Beta, ImageDescriptor Gamma)
			throws Exception {
		if (Fx.width != Fy.width
				|| Fx.height != Fy.height
				|| Fx.depth != Fy.depth
				|| (Fx.depth > 1 && (Fx.width != Fz.width
						|| Fx.height != Fz.height || Fx.depth != Fz.depth))
				|| Fx.width != Alpha.width
				|| Fx.height != Alpha.height
				|| Fx.depth != Alpha.depth
				|| (Fx.depth > 1 && (Fx.width != Beta.width
						|| Fx.height != Beta.height || Fx.depth != Beta.depth))
				|| (Fx.depth > 1 && (Fx.width != Gamma.width
						|| Fx.height != Gamma.height || Fx.depth != Gamma.depth))) {
			throw new Exception(
					"Images must have the same size in Filters.gradientDirection");
		}

		for (int x = 0; x < Fx.width; ++x) {
			for (int y = 0; y < Fx.height; ++y) {
				for (int z = 0; z < Fx.depth; ++z) {
					Alpha.setValue(x, y, z,
							Math.atan2(Fy.value(x, y, z), Fx.value(x, y, z)));
				}
			}
		}

		if (Fx.depth > 1) {
			for (int x = 0; x < Fx.width; ++x) {
				for (int y = 0; y < Fx.height; ++y) {
					for (int z = 0; z < Fx.depth; ++z) {
						Beta.setValue(x, y, z, Math.atan2(Fz.value(x, y, z),
								Fy.value(x, y, z)));
						Gamma.setValue(x, y, z, Math.atan2(Fx.value(x, y, z),
								Fz.value(x, y, z)));
					}
				}
			}
		}
	}

	static private void filter(final ImageDescriptor src, ImageDescriptor dst,
			final double[] kernelX, final double[] kernelY,
			final double[] kernelZ) {
		double[] buffer1 = (src.depth == 1) ? new double[src.depth * src.width
				* src.height] : dst.getData();
		double[] buffer2 = (src.depth == 1) ? dst.getData()
				: new double[src.depth * src.width * src.height];

		for (int z = 0; z < src.depth; z++) {
			int offset = src.offset + z * src.zStep;
			for (int row = 0; row < src.height; row++) {
				convolveLine(src.getData(), buffer1, kernelX, offset + row
						* src.yStep, src.width, 1);
			}
			for (int column = 0; column < src.width; column++) {
				convolveLine(buffer1, buffer2, kernelY, offset + column,
						src.height, src.yStep);
			}
		}

		if (src.depth > 1) {
			int i = src.offset;
			for (int y = 0; y < src.height; y++) {
				for (int x = 0; x < src.width; x++) {
					convolveLine(buffer2, dst.getData(), kernelZ, i, src.depth,
							src.zStep);
					i++;
				}
				i += src.yStep - src.width;
			}
		}
	}

	private static void convolveLine(final double[] src, double[] dst,
			final double[] kernel, int offset, int size, int step) {
		int k0 = kernel.length / 2;

		int index = offset;
		for (int i = 0; i < size; i++) {
			dst[index] = 0;
			for (int k = 0; k < kernel.length; k++) {
				int j = i + k - k0;
				if (j < 0) {
					j = -j;
				} else if (j >= size) {
					j = 2 * size - j - 1;
				}

				dst[index] += src[offset + j * step] * kernel[k];
			}
			index += step;
		}
	}
}
