/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobplugin.operations;

import icy.type.DataType;
import plugins.big.blobplugin.descriptors.ImageDescriptor;
import plugins.big.blobplugin.misc.Parameters;

/**
 * @class Morphology
 * 
 *        Abstract class implementing some basic morphological operations.
 * 
 */
public abstract class Morphology {
	final static int WINDOW_SIZE = 16;

	static public void binarizeImage(final ImageDescriptor src,
			ImageDescriptor mask, final Parameters parameters) throws Exception {
		double scaleFactor = 1;
		switch (src.type) {
		case DOUBLE:
		case FLOAT:
			scaleFactor = 1;
			break;
		case BYTE:
		case INT:
		case LONG:
		case SHORT:
		case UBYTE:
		case UINT:
		case ULONG:
		case UNDEFINED:
		case USHORT:
			scaleFactor = 255;
			break;
		}
		double stepThreshold = scaleFactor * parameters.threshold();

		// Blur the image to reduce noise
		ImageDescriptor buffer = new ImageDescriptor(src.width, src.height,
				src.depth, DataType.DOUBLE);
		Filters.gaussianBlur(src, buffer, 1);

		mask.fill(0);

		int windowSize = WINDOW_SIZE;
		int x0 = 0, y0 = 0, z0 = 0;
		for (int d = 0; d < 4; ++d, x0 -= 1.5 * windowSize, y0 -= 1.5 * windowSize, z0 -= 1.5 * windowSize, windowSize *= 2) {
			for (int z = z0; z < src.depth; ++z) {
				for (int x = x0; x < src.width; x += windowSize) {
					for (int y = y0; y < src.height; y += windowSize) {
						ImageDescriptor subImg = buffer.subImage(x, y, z,
								windowSize, windowSize, 1);
						ImageDescriptor subMask = mask.subImage(x, y, z,
								windowSize, windowSize, 1);

						double kMeans[] = getKMeans(subImg, scaleFactor);
						boolean color = Math.abs(kMeans[0] - kMeans[1]) >= stepThreshold;
						double threshold = 0.5 * (kMeans[0] + kMeans[1]);
						if (color) {
							for (int i = 0; i < subMask.width; ++i) {
								for (int j = 0; j < subMask.height; ++j) {
									subMask.setValue(
											i,
											j,
											0,
											subMask.value(i, j, 0)
													+ ((subImg.value(i, j, 0) >= threshold) ? 1
															: -1));
								}
							}
						}
					}
				}
			}
		}

		for (int x = 0; x < src.width; ++x) {
			for (int y = 0; y < src.height; ++y) {
				for (int z = 0; z < src.depth; ++z) {
					mask.setValue(x, y, z,
							(mask.value(x, y, z) > 0) ? scaleFactor : 0);
				}
			}
		}
	}

	static private double[] getKMeans(final ImageDescriptor img, double maxValue) {
		double[] centers = { 0.25 * maxValue, 0.75 * maxValue };
		double[] d = new double[centers.length];
		double[] centers_p = new double[centers.length];
		int[] n = new int[centers.length];

		double error = Double.MAX_VALUE;
		double errMin = 1;
		while (error > errMin) {
			for (int i = 0; i < centers.length; i++) {
				centers_p[i] = 0;
				n[i] = 0;
			}

			for (int x = 0; x < img.width; ++x) {
				for (int y = 0; y < img.height; ++y) {
					for (int z = 0; z < img.depth; ++z) {
						double value = img.value(x, y, z);

						double dMin = Double.MAX_VALUE;
						int iMin = 0;

						for (int i = 0; i < centers.length; i++) {
							d[i] = value - centers[i];
							d[i] *= d[i];

							if (d[i] < dMin) {
								dMin = d[i];
								iMin = i;
							}
						}

						centers_p[iMin] += value;
						n[iMin]++;
					}
				}
			}

			error = 0;
			for (int i = 0; i < centers.length; i++) {
				if (n[i] > 0) {
					centers_p[i] /= n[i];
				}

				double d_c = centers[i] - centers_p[i];
				error += d_c * d_c;

				centers[i] = centers_p[i];
			}
		}

		if (n[0] == 0) {
			centers[0] = centers[1];
		} else if (n[1] == 0) {
			centers[1] = centers[0];
		} else if (centers[0] > centers[1]) {
			double temp = centers[1];
			centers[1] = centers[0];
			centers[0] = temp;
		}

		return centers;
	}
}
