package plugins.ylemontag.noisegenerator;

import icy.sequence.Sequence;
import icy.sequence.SequenceBuilder;
import icy.type.DataType;
import plugins.ylemontag.noisegenerator.noisemodels.GaussianNoise;
import plugins.ylemontag.noisegenerator.noisemodels.PoissonNoise;
import plugins.ylemontag.noisegenerator.noisemodels.SaltPepperNoise;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Base class implementing a noise model (Gaussian, Poisson, salt & pepper, etc...)
 */
public abstract class NoiseModel
{
	/**
	 * Available types of noise models
	 */
	public static enum Type
	{
		GAUSSIAN   ("Gaussian noise"),
		POISSON    ("Poisson noise" ),
		SALT_PEPPER("Salt & pepper" );
		
		private String _name;
		
		private Type(String name)
		{
			_name = name;
		}
		
		@Override
		public String toString()
		{
			return _name;
		}
	}
	
	/**
	 * Instantiate a new Gaussian noise model
	 * @throws IllegalNoiseModelArgumentException If sigma<0
	 */
	public static NoiseModel createGaussianNoiseModel(double sigma)
	{
		return new GaussianNoise(sigma);
	}
	
	/**
	 * Instantiate a new Poisson noise model
	 */
	public static NoiseModel createPoissonNoiseModel()
	{
		return new PoissonNoise();
	}
	
	/**
	 * Instantiate a new salt & pepper noise model
	 * @throws IllegalNoiseModelArgumentException If intensity<0 or intensity>1
	 * @throws IllegalNoiseModelArgumentException If lowerBound>upperBound
	 */
	public static NoiseModel createSaltPepperNoiseModel(double intensity, double lowerBound, double upperBound)
	{
		return new SaltPepperNoise(intensity, lowerBound, upperBound);
	}
	
	/**
	 * Constructor
	 */
	protected NoiseModel() {}
	
	/**
	 * Return a noised copy of the input sequence
	 */
	public Sequence addNoise(Sequence in)
	{
		try {
			return addNoise(in, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Return a noised copy of the input sequence
	 * @throws Controller.CanceledByUser Operation canceled through the Controller object
	 */
	public Sequence addNoise(Sequence in, Controller controller) throws Controller.CanceledByUser
	{
		Sequence retVal = new Sequence();
		retVal.setName(describeNoise(in.getName()));
		addNoise(in, retVal, controller);
		return retVal;
	}
	
	/**
	 * Clear the sequence 'out' and return a noised copy of the input sequence 'in'
	 * in the sequence 'out'
	 */
	public void addNoise(Sequence in, Sequence out)
	{
		try {
			addNoise(in, out, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException("Unreachable code point");
		}
	}
	
	/**
	 * Clear the sequence 'out' and return a noised copy of the input sequence 'in'
	 * in the sequence 'out'
	 * @throws Controller.CanceledByUser Operation canceled through the Controller object
	 */
	public void addNoise(Sequence in, Sequence out, Controller controller)
		throws Controller.CanceledByUser
	{
		SequenceBuilder builder = new SequenceBuilder(in.getSizeX(), in.getSizeY(), in.getSizeZ(),
			in.getSizeT(), in.getSizeC(), DataType.DOUBLE, out);
		builder.beginUpdate();
		try {
			generateNoise(in, builder, controller);
		}
		finally {
			builder.endUpdate();
		}
	}
	
	/**
	 * Noise generation function, that must be implemented in derived classes
	 */
	protected abstract void generateNoise(Sequence in, SequenceBuilder out, Controller controller)
		throws Controller.CanceledByUser;
	
	/**
	 * Generate a description of the noisy sequence, assuming that the name of
	 * the original sequence is 'inName'
	 */
	protected abstract String describeNoise(String inName);
}
