/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobplugin;

import icy.image.IcyBufferedImage;
import icy.image.ImageUtil;
import icy.roi.ROI;
import icy.sequence.Sequence;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceListener;
import icy.type.DataType;
import icy.type.point.Point4D;

import java.awt.Image;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.ImageIcon;

import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.ezplug.EzComponent;
import plugins.adufour.ezplug.EzGroup;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzStoppable;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.vars.lang.Var;
import plugins.adufour.vars.util.VarListener;
import plugins.big.bigsnakeutils.icy.ellipsoid.AbstractEllipsoid;
import plugins.big.blobplugin.descriptors.CellDescriptor;
import plugins.big.blobplugin.descriptors.CellDescriptor.CellComparator;
import plugins.big.blobplugin.descriptors.EllipseDescriptor;
import plugins.big.blobplugin.descriptors.EllipsoidDescriptor;
import plugins.big.blobplugin.descriptors.ImageDescriptor;
import plugins.big.blobplugin.misc.Parameters;
import plugins.big.blobplugin.misc.ParametersCategory;
import plugins.big.blobplugin.misc.TargetBrightness;
import plugins.big.blobplugin.operations.MathOperations;
import plugins.big.blobplugin.operations.Morphology;
import plugins.big.blobplugin.operations.Optimization;

/**
 * 
 * @author Julien Jacquemot
 * @author Cl&#233;ment Marti
 * 
 * @mainpage Blob Detector
 * 
 * @section intro_sec Introduction
 *
 * Microscopy has been one of the most powerful tools in biology since its invention in the XVIIth century. 
 * It has seen many developments along the years, and now microscopes are complex, expensive pieces of engineering, able to extract much information out of a sample. 
 * One of the most popular techniques is fluorescence microscopy, in which the light is not reflected by the sample but emitted by chemicals specifically attached to parts of it. 
 * Another is confocal microscopy, enabling to take 3D images of sample, instead of the classical 2D images of wide field microscopes. 
 * Finally, the development of cheap storage and processing power made possible the recording  and processing of videos taken over a long period of time, for instance to analyze the behaviour of  cells.
 * @n @n
 * Modern microscopes enable to use all these techniques at the same time, and consequently enable to track the 3D position of specific molecules or cells over time. 
 * However, this result in sizeable datasets of a few gigabytes, and thus prohibits a manual analysis. 
 * So there is a need for a computer program capable of giving the positions of a given cell over time with minimal user intervention. 
 * Moreover, this program should be easy to use since it is aimed at biologists who are more interested in the conclusion they can infer from the tracking than in the details of the processing. 
 * This would enable biologist to know the distribution of cell positions, velocities and sizes, and their appearance and death rates.
 * @n @n
 * However, biologists are also interested in one more cell feature: their shape. 
 * Dividing, dying, damaged and healthy cells can have very different shapes. 
 * Thus cell shapes convey a lot of information about what is happening in the sample, and enable to distinguish between different cell populations. 
 * Different methods exist to describe shapes, but we will focus on parametric snakes. 
 * Parametric snakes enable to describe a shape with a small set of numbers, and choose these numbers by automatically fitting shapes on images with active contours.
 * Still, for a good fit, the snakes should not be completely different from the target shape: they need to be initialized correctly.
 *
 * @section tools_sec Existing Tools
 * @subsection snakes_subsec Ellipse-reproducing Snakes
 * Parametric snakes use a few coefficients to describe a shape. 
 * The shape they have consists of interpolated values around their nodes (defined by the parameters), with specific basis functions for interpolation. 
 * Thus, they can not reproduce all shapes with the same accuracy because actual shapes are very different: although a square and a circle can be described with one parameter (the side or the radius), trying to fit a square on a circle will not give good fit. 
 * Thus, it is important to have prior information about what the shapes to fit look like to choose a parametric snake model able to give a close fit on these. 
 * In the case of bioimaging and especially of cell tracking, the shapes of interest are the nuclei or the membranes of cells. 
 * Nucleis tend to be spherical, and a significant fraction of cells are close to ellipsoids. 
 * Thus, snakes capable to reproduce ellipsoids perfectly are an interesting candidate. 
 * These snakes have been implemented as plugins named <a href="http://icy.bioimageanalysis.org/plugin/Active_Cells"><b>Active Cells</b></a> and <a href="http://icy.bioimageanalysis.org/plugin/Active_Cells_3D"><b>Active Cells 3D</b></a> for the <a href="http://fiji.sc/Fiji"><b>ImageJ</b></a> and <a href="http://icy.bioimageanalysis.org/"><b>ICY</b></a> bioimaging softwares. 
 *
 * @subsection icy_subsec ICY 
 * <a href="http://icy.bioimageanalysis.org/"><b>ICY</b></a> is a new bioimaging software for image processing competing with the current standard <a href="http://fiji.sc/Fiji"><b>ImageJ</b></a>. It has several additional features over <a href="http://fiji.sc/Fiji"><b>ImageJ</b></a>:
 * <ul>
 * <li> A more user-friendly interface.
 * <li> A graphical programming language called <a href="http://icy.bioimageanalysis.org/plugin/Protocols"><b>Protocol</b></a>.
 * <li> A central server regrouping all <a href="http://icy.bioimageanalysis.org/"><b>ICY</b></a> plugins and their documentation. Users can search, share and install these plugins directly through <a href="http://icy.bioimageanalysis.org/"><b>ICY</b></a>, which will automatically look for updates. 
 * <li> <a href="http://icy.bioimageanalysis.org/plugin/EzPlug_SDK"><b>EzPlug</b></a>, a simple tool to build graphical interfaces.
 * <li> A dedicated community which answers questions fast. 
 * </ul>
 * It also natively incorporates <a href="http://fiji.sc/Fiji"><b>ImageJ</b></a>, and is also multi-platform since it is coded in Java. Scripts can be made in Javascript or Python.
 * @n
 * @section user_sec User Manual 
 * @subsection estup_subsec Basic Setup to detect blobs, create snakes, optimization and recording using Protocol
 * <ul> <li> Download the plugin <a href=""><b>SnakeTracker</b></a> from the Icy Online Plugin Repository. It will automatically download the other required plugins (
 * <a href="http://icy.bioimageanalysis.org/plugin/Active_Cells"><b>Active Cells</b></a>, 
 * <a href="http://icy.bioimageanalysis.org/plugin/Active_Cells_3D"><b>Active Cells 3D</b></a>,
 * <a href=""><b>Blob Detector</b></a>, 
 * <a href="http://icy.bioimageanalysis.org/plugin/Workbooks"><b>Workbooks</b></a>, 
 * <a href="http://icy.bioimageanalysis.org/plugin/EzPlug_SDK"><b>EzPlug</b></a>,
 * <a href="http://icy.bioimageanalysis.org/plugin/Filter_Toolbox"><b>Filter Toolbox</b></a>,
 * <a href=""><b>2DAutomaticSnake</b></a> and 
 * <a href=""><b>3DAutomaticSnake</b></a>)
 * <li> Launch <a href="http://icy.bioimageanalysis.org/plugin/Protocols"><b>Protocol</b></a>.
 * <li> Let us say you want to extract a workbook of snake parameters from a 2D sequence. 
 * <ul> 
 * <li> Import the blocks <a href=""><b>Blob Detector</b></a> and <a href=""><b>2DAutomaticSnake</b></a>. 
 * <li> Add a block File to sequence (I/O menu) and import your sequence of interest. 
 * <li> Link this block to the sequence input of <a href=""><b>Blob Detector</b></a> (blue arrow). 
 * <li> Link the sequence output of this block to the sequence input of <a href=""><b>2DAutomaticSnake</b></a>. 
 * <li> Link the result output of <a href=""><b>Blob Detector</b></a> to the list of ROI or snake input of <a href=""><b>2DAutomaticSnake</b></a>.
 * <li> In both blocks, choose in the target brightness menu the brightness of the objects you want to detect compared to the background. 
 * <li> Add a block <a href="http://icy.bioimageanalysis.org/plugin/Workbooks"><b>Workbooks</b></a> to File (I/O) menu and link the output list of snake parameters of <a href=""><b>2DAutomaticSnake</b></a> to it. 
 * <li> Select an Excel file to save the parameters in. 
 * <li> Add a display block and link the sequence output of <a href=""><b>2DAutomaticSnake</b></a> to it if you want to visualize the fitted snakes. 
 * </ul>
 * <li> Your <a href="http://icy.bioimageanalysis.org/plugin/Protocols"><b>Protocol</b></a> is now ready. Hit the Run button to execute it. The snakes parameters are now stored in your excel file, with each line corresponding to a snake with its energy, area, the values of its nodes, and the time in the sequence it was detected (if your sequence has several frames over T)
 * <li> For a 3D sequence, just use 3DAutomaticSnake instead of 2DAutomaticSnake.
 * </ul>
 * You can also use the <a href=""><b>Blob Detector</b></a> without protocol, it shows up in among the other <a href="http://icy.bioimageanalysis.org/"><b>ICY</b></a> plugins you installed.
 * @subsection cell_tracks_subsec Get the cell tracks 
 * <ul>
 * <li> Create a new protocol. 
 * <li> Add the block <a href =""><b>Snake Tracker</b></a>. 
 * <li> Add a block <a href="http://icy.bioimageanalysis.org/plugin/Workbooks"><b>Workbooks</b></a> to file, select an excel file to saves the tracks in, and link workbook of cell tracks to it. 
 * <li> Add a File to <a href="http://icy.bioimageanalysis.org/plugin/Workbooks"><b>Workbooks</b></a> block to import the excel file with snake parameters you created in 1), and link it to the input workbook of snake parameters. 
 * <li> Load the sequence used in the previous section and select it in the display sequence menu. 
 * <li> Set the number of spatial dimensions to 2, choose what weight spatial distance (Euclidean distance between snake centers) should have versus shape distance (distance on the node coefficients describing the shape of the snake), and set a maximum acceptable value between two successive positions of the same track.
 * <li> Hit run. The tracks are now saved in the excel file, with each line being a track, namely a list of IDs. These IDs are the IDs of the snakes defined in the worbook of snake parameters (first column).
 * </ul>
 * You can combine the various blocks to create the protocol suited to your needs.
 * @subsection options_subsec Advanced Options 
 * You can fine tune the <a href=""><b>Blob Detector</b></a> to improve its performance:
 * <ul>
 * <li> <b>Radius min         :</b> Minimal radius of a cell.
 * <li> <b>Gaussian blur sigma:</b> Standard deviation used for the initial gaussian blur (pre-treatment of the data). A value of 0 means that no filtering will be applied.
 * <li> <b>Threshold Method   :</b> Define which kind of thresholds will be used for the binarization step. <a href="http://en.wikipedia.org/wiki/Lloyd%27s_algorithm"><b>Kmean</b></a> is automatic, manual will require you to select the two threshold of a hysteresis <a href="http://en.wikipedia.org/wiki/Canny_edge_detector"><b>threshold method</b></a>.
 * <li> <b>Threshold1         :</b> Weak threshold used during the binarization step. It is only used if  thresholdMethod has the value Manual.
 * <li> <b>Threshold2         :</b> Weak threshold used during the binarization step. It is only used if  thresholdMethod has the value Manual.
 * <li> <b>Cluster Count      :</b> Determine how many clusters are created for the computation of the two thresholds required by the binarization step. It is only used if thresholdMethod has the value KMean.
 * <li> <b>Tolerance          :</b> Determine how much two cells are closed. 
 * <li> <b>Delta energy       :</b> Stop criterion for the optimization
 * <li> <b>Gradient Step Size :</b> Variation of the parameters used to estimate the gradient of the energy.
 * <li> <b>Gradient Factor    :</b> Describe how the parameters are updated in the direction of the gradient
 * </ul>
 * You can also fine tune the <a href=""><b>2DAutomaticSnake</b></a> and <a href=""><b>3DAutomaticSnake</b></a> to have a better fit:
 * <ul>
 * <li> <b>Output sequence	  :</b> a copy of sequence with snake ROIs displayed on it
 * <li> <b>MaxIterations      :</b> how many maximum steps the snake iteration can do alpha: the tradeoff between edge and region energies for the snakes (used only if energy type is MIX)
 * <li> <b>Gamma              :</b> the stiffness of the contour of the snake
 * <li> <b>Sigma              :</b> the standard deviation of the gaussian kernel used to blurred the image before fitting the snakes
 * <li> <b>Energy type        :</b> choose the energy between EDGES,REGION or MIXTURE
 * <li> <b>Format             :</b> the format of the object describing the ROIs (a workbook or an array of ROIs)
 * </ul>
 */

/**
 * @class BlobPlugin
 * 
 *        Main class containing the methods to initialize the interface and
 *        execute our algorithm.
 */
public class BlobPlugin extends EzPlug implements Block, EzStoppable {
	private Parameters _parameters = null;

	private EzVarBoolean _previewBinarization = null;
	private Sequence _previewBinarizationSequence = null;

	private boolean _stop;

	private Var<ROI[]> _outputVar = null;

	public BlobPlugin() {
	}

	@Override
	public void clean() {
	}

	/**
	 * Initialize the interface.
	 */
	@Override
	protected void initialize() {
		_parameters = new Parameters(this, true);

		initializeGroup(ParametersCategory.Global, "");
		EzGroup binarizationGroup = initializeGroup(
				ParametersCategory.Binarization, "Binarization");
		initializeGroup(ParametersCategory.Optimization, "Optimization");

		_previewBinarization = new EzVarBoolean("Preview", false);
		_previewBinarization.getVariable().addListener(
				new VarListener<Boolean>() {
					@Override
					public void valueChanged(Var<Boolean> source,
							Boolean oldValue, Boolean newValue) {
						if (newValue) {
							addSequence(_previewBinarizationSequence);
							BlobPlugin.this.valueChanged(null);
						} else {
							removeSequence(_previewBinarizationSequence);
						}
					}

					@Override
					public void referenceChanged(Var<Boolean> source,
							Var<? extends Boolean> oldReference,
							Var<? extends Boolean> newReference) {
					}
				});
		binarizationGroup.addEzComponent(_previewBinarization);

		_previewBinarizationSequence = new Sequence("Binarization");
		_previewBinarizationSequence.addListener(new SequenceListener() {
			@Override
			public void sequenceClosed(Sequence sequence) {
				_previewBinarization.setValue(false);
			}

			@Override
			public void sequenceChanged(SequenceEvent sequenceEvent) {
			}
		});

		// Set icons
		try {
			List<Image> icons = new ArrayList<Image>();
			icons.add(ImageUtil.load(this.getClass().getResourceAsStream(
					"icons/icon_16.png")));
			icons.add(ImageUtil.load(this.getClass().getResourceAsStream(
					"icons/icon_32.png")));
			icons.add(ImageUtil.load(this.getClass().getResourceAsStream(
					"icons/icon_64.png")));

			super.getUI().getExternalFrame().setIconImages(icons);
			super.getUI().getInternalFrame()
					.setFrameIcon(new ImageIcon(icons.get(0)));
		} catch (Exception e) {
			System.err.println("Failed to set the window icon.");
			e.printStackTrace();
		}
	}

	@Override
	protected void execute() {
		Sequence sequence = _parameters.sequence();
		if (sequence == null) {
			return;
		}
		_stop = false;

		ArrayList<ROI> rois = new ArrayList<ROI>();
		for (int t = 0; t < sequence.getSizeT(); ++t) {
			if (_stop) {
				return;
			}

			// Retrieve the image at time t (with bright target)
			ImageDescriptor img = new ImageDescriptor(sequence, t);
			if (_parameters.targetBrightness() == TargetBrightness.Dark) {
				img.inverse();
			}

			if (_stop) {
				return;
			}

			// Binarize the image
			ImageDescriptor mask = new ImageDescriptor(img.width, img.height,
					img.depth, img.type);
			try {
				Morphology.binarizeImage(img, mask, _parameters);

				if (_stop) {
					return;
				}

				// Compute the distance map of the mask
				ImageDescriptor acc = new ImageDescriptor(img.width,
						img.height, img.depth, img.type);
				MathOperations.getDistanceMap(mask, acc);

				if (_stop) {
					return;
				}

				// Get the local maxima of the distance map. They are the center
				// of the potential cells.
				List<Point4D> maxima = MathOperations.getLocalMaxima(acc,
						_parameters.rMin());

				if (_stop) {
					return;
				}

				// Compute the integral of the input image
				ImageDescriptor imgInt = new ImageDescriptor(img.width,
						img.height, img.depth, img.type);
				MathOperations.getImageIntegral(img, imgInt);

				if (_stop) {
					return;
				}

				/*
				 * For each maxima found, we create a new CellDescriptor that we
				 * optimize.
				 * 
				 * If the cell is still valid and if its radius is bigger than
				 * the minimal radius allowed, we add it to the list of
				 * potential cells.
				 */
				ArrayList<CellDescriptor> cells = new ArrayList<CellDescriptor>();
				for (Point4D pt : maxima) {
					if (_stop) {
						return;
					}

					CellDescriptor descriptor = (img.depth == 1) ? new EllipseDescriptor(
							t) : new EllipsoidDescriptor(t);
					descriptor.init(pt);

					if (!_parameters.optimizeDescriptors()) {
						descriptor.computeEnergy(imgInt);
						cells.add(descriptor);
					} else if (Optimization.optimize(descriptor, imgInt,
							_parameters)) {
						cells.add(descriptor);
					}
				}

				/*
				 * After sorting the cells by their energy, we check that there
				 * are no cells to closed from each other. Only the one with the
				 * lowest energy is kept.
				 */
				Collections.sort(cells, new CellComparator());
				for (int i = 0; i < cells.size(); i++) {
					if (_stop) {
						return;
					}

					CellDescriptor cell1 = cells.get(i);

					for (int j = i + 1; j < cells.size();) {
						CellDescriptor cell2 = cells.get(j);

						if (cell1.isClosedTo(cell2)) {
							cells.remove(j);
							continue;
						}

						j++;
					}
				}

				for (CellDescriptor cell : cells) {
					rois.add(((AbstractEllipsoid) cell).toROI());
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}

		/*
		 * Add the ROIs to the output
		 */
		if (isHeadLess()) {
			// Protocol Output
			ROI[] result = new ROI[rois.size()];
			for (int i = 0; i < rois.size(); i++) {
				result[i] = rois.get(i);
			}
			_outputVar.setValue(result);
		} else {
			sequence.removeAllROI();
			for (ROI roi : rois) {
				sequence.addROI(roi);
			}
		}
	}

	private EzGroup initializeGroup(final ParametersCategory category,
			final String name) {
		if (name.isEmpty()) {
			for (EzComponent component : _parameters.ezVariables.get(category)) {
				addEzComponent(component);
			}
			return null;
		} else {
			EzGroup group = new EzGroup(name);
			for (EzComponent component : _parameters.ezVariables.get(category)) {
				group.addEzComponent(component);
			}

			addEzComponent(group);
			return group;
		}
	}

	private void initializeGroup(final ParametersCategory category,
			VarList inputMap) {
		for (Var<?> variable : _parameters.variables.get(category)) {
			inputMap.add(variable);
		}
	}

	@Override
	public void declareInput(VarList inputMap) {
		_parameters = new Parameters(this, false);

		initializeGroup(ParametersCategory.Global, inputMap);
		initializeGroup(ParametersCategory.Binarization, inputMap);
		initializeGroup(ParametersCategory.Optimization, inputMap);
	}

	@Override
	public void declareOutput(VarList outputMap) {
		_outputVar = new Var<ROI[]>("Result", ROI[].class);
		outputMap.add(_outputVar);
	}

	public void valueChanged(Var<?> source) {
		if (_previewBinarization.getValue()) {
			_previewBinarizationSequence.removeAllImages();
			Sequence sequence = _parameters.sequence();
			if (sequence == null) {
				return;
			}

			ImageDescriptor img = new ImageDescriptor(sequence, 0);
			if (_parameters.targetBrightness() == TargetBrightness.Dark) {
				img.inverse();
			}

			ImageDescriptor mask = new ImageDescriptor(img.width, img.height,
					img.depth, img.type);

			try {
				Morphology.binarizeImage(img, mask, _parameters);
				IcyBufferedImage[] images = new IcyBufferedImage[img.depth];
				for (int z = 0; z < img.depth; z++) {
					images[z] = new IcyBufferedImage(sequence.getWidth(),
							sequence.getHeight(), 1, DataType.DOUBLE);
					_previewBinarizationSequence.setImage(0, z, images[z]);
				}
				mask.fillImages(images, 0);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public void stopExecution() {
		_stop = true;
	}
}
