/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobplugin.descriptors;

import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;
import icy.type.DataType;

/**
 * @class ImageDescriptor
 * 
 *        Describe a 2D or 3D image.
 * 
 */
public class ImageDescriptor {
	/** Data of the image. */
	private double[] data = null;
	/** Width of the image. */
	final public int width;
	/** Height of the image. */
	final public int height;
	/** Depth of the image. In case of a 2D image it's 1. */
	final public int depth;

	public final DataType type;

	/** Offset of the beginning of the image. */
	final public int offset;

	/** Size of a line. */
	final public int yStep;

	/** Size of a layer (z=cte). */
	final public int zStep;

	/**
	 * Constructor. Create an empty 2D image.
	 * 
	 * @param width
	 *            Width of the image.
	 * @param height
	 *            Height of the image.
	 */
	public ImageDescriptor(int width, int height, DataType type) {
		this(width, height, 1, type);
	}

	/**
	 * Constructor. Create an empty 3D image.
	 * 
	 * @param width
	 *            Width of the image.
	 * @param height
	 *            Height of the image.
	 * @param depth
	 *            Depth of the image.
	 */
	public ImageDescriptor(int width, int height, int depth, DataType type) {
		this.width = this.yStep = width;
		this.height = height;
		this.depth = depth;
		this.zStep = width * height;
		this.type = type;
		this.offset = 0;

		data = new double[depth * width * height];
	}

	/**
	 * Constructor. Create a copy of an existing image.
	 * 
	 * @param other
	 *            Image to copy.
	 */
	public ImageDescriptor(final ImageDescriptor other) {
		this(other.width, other.height, other.depth, other.type);

		for (int i = 0; i < other.data.length; i++) {
			data[i] = other.data[i];
		}
	}

	/**
	 * Constructor. Create an image from a <a href=
	 * "http://www.bioimageanalysis.org/icy/doc/classicy_1_1sequence_1_1_sequence.html"
	 * ><b>ICY::Sequence</b></a>.
	 * 
	 * @n By default the image is taken at time t=0 and from the first canal.
	 * 
	 * @param sequence
	 *            Input sequence.
	 */
	public ImageDescriptor(Sequence sequence) {
		this(sequence, 0, 0);
	}

	/**
	 * Constructor. Create an image from a <a href=
	 * "http://www.bioimageanalysis.org/icy/doc/classicy_1_1sequence_1_1_sequence.html"
	 * ><b>ICY::Sequence</b></a>.
	 * 
	 * @n By default the image is taken from the first channel.
	 * 
	 * @param sequence
	 *            Input sequence.
	 * @param t
	 *            Time frame of the image.
	 */
	public ImageDescriptor(Sequence sequence, int t) {
		this(sequence, t, 0);
	}

	/**
	 * Constructor. Create an image from a <a href=
	 * "http://www.bioimageanalysis.org/icy/doc/classicy_1_1sequence_1_1_sequence.html"
	 * ><b>ICY::Sequence</b></a>.
	 * 
	 * @param sequence
	 *            Input sequence.
	 * @param t
	 *            Time frame of the image.
	 * @param canal
	 *            Canal of the image to copy.
	 */
	public ImageDescriptor(Sequence sequence, int t, int canal) {
		this(sequence.getWidth(), sequence.getHeight(), sequence.getSizeZ(),
				sequence.getDataType_());

		int index = 0;
		for (int z = 0; z < depth; z++) {
			final IcyBufferedImage src = sequence.getImage(t, z);

			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++) {
					data[index++] = src.getData(x, y, canal);
				}
			}
		}
	}

	public int index(int x, int y) {
		return index(x, y, 0);
	}

	public int index(int x, int y, int z) {
		return offset + x + y * yStep + z * zStep;
	}

	/** Return the value of a pixel at a given position. */
	public double value(int x, int y) {
		return value(x, y, 0);
	}

	/** Return the value of a pixel at a given position. */
	public double value(int x, int y, int z) {
		if (x < 0 || x >= width || y < 0 || y >= height || z < 0 || z >= depth) {
			return Double.NaN;
		}
		return data[index(x, y, z)];
	}

	/** Set the value of a pixel at a given position. */
	public void setValue(int x, int y, double value) {
		setValue(x, y, 0, value);
	}

	/** Set the value of a pixel at a given position. */
	public void setValue(int x, int y, int z, double value) {
		if (x < 0 || x >= width || y < 0 || y >= height || z < 0 || z >= depth) {
			return;
		}
		data[index(x, y, z)] = value;
	}

	/**
	 * Replace the image by its negative (assuming that the values are between 0
	 * and 255).
	 */
	public void inverse() {
		switch (type) {
		case DOUBLE:
		case FLOAT:
			inverse(1);
			break;
		case BYTE:
		case INT:
		case LONG:
		case SHORT:
		case UBYTE:
		case UINT:
		case ULONG:
		case UNDEFINED:
		case USHORT:
			inverse(255);
			break;
		}
	}

	public double[] getData() {
		return data;
	}

	/**
	 * Replace the image by its negative.
	 * 
	 * @param maxValue
	 *            Maximum value of the image.
	 */
	public void inverse(double maxValue) {
		for (int i = 0; i < data.length; i++) {
			data[i] = maxValue - data[i];
		}
	}

	public ImageDescriptor subImage(int x, int y, int width, int height) {
		return subImage(x, y, 0, width, height, depth);
	}

	public ImageDescriptor subImage(int x, int y, int z, int width, int height,
			int depth) {
		x = Math.max(x, 0);
		y = Math.max(y, 0);
		z = Math.max(z, 0);
		width = Math.min(x + width, this.width) - x;
		height = Math.min(y + height, this.height) - y;
		depth = Math.min(z + depth, this.depth) - z;

		if (width <= 0 || height <= 0 || depth <= 0) {
			return null;
		}
		return new ImageDescriptor(x, y, z, width, height, depth, this);
	}

	public void fill(double value) {
		for (int z = 0; z < depth; z++) {
			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++) {
					data[offset + x + y * yStep + z * zStep] = value;
				}
			}
		}
	}

	/**
	 * Copy the content of this image descriptor into an array of <a href=
	 * "http://www.bioimageanalysis.org/icy/doc/classicy_1_1image_1_1_icy_buffered_image.html"
	 * ><b>ICY::IcyBufferedImage</b></a>.
	 * 
	 * @param images
	 *            Array of images. Each image corresponds to a z layer.
	 * @param canal
	 *            Canal in which the data have to be copied.
	 * 
	 * @throws Exception
	 *             The number of images has to be equal to the depth of the
	 *             image descriptor.
	 */
	public void fillImages(IcyBufferedImage[] images, int canal) {
		int imgSize = width * height;
		int i1 = this.offset;
		double[] buffer = new double[imgSize];

		for (int z = 0; z < depth; z++) {
			int i2 = 0;

			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++) {
					buffer[i2] = data[i1];
					i1++;
					i2++;
				}
				i1 += yStep - width;
			}

			images[z].setDataXYAsDouble(canal, buffer);
			i1 += zStep - imgSize;
		}
	}

	private ImageDescriptor(int x, int y, int z, int width, int height,
			int depth, ImageDescriptor other) {
		this.width = width;
		this.height = height;
		this.depth = depth;

		this.offset = other.index(x, y, z);

		this.yStep = other.yStep;
		this.zStep = other.zStep;
		this.type = other.type;
		this.data = other.data;
	}
}
