/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobplugin.operations;

import plugins.big.bigsnakeutils.icy.ellipsoid.AbstractEllipsoid;
import plugins.big.blobplugin.descriptors.CellDescriptor;
import plugins.big.blobplugin.descriptors.ImageDescriptor;
import plugins.big.blobplugin.misc.Parameters;

/**
 * @class Optimization
 * 
 *        Abstract class implementing the optimization of the cell descriptors.
 * 
 */
public abstract class Optimization {
	private static final double GRADIENT_STEP_SIZE = 0.2;
	private static final double MU = 0.2;
	private static final double ETA = 0.1;
	private static final double MIN_DISPLACEMENT = 1E-7;

	/**
	 * Optimize the parameters of a cell descriptor. Perform a basic gradient
	 * descent.
	 * 
	 * @param cell
	 *            Cell descriptor to optimize.
	 * @param mask
	 *            Binarized mask of the image.
	 * @param parameters
	 *            Parameters of the algorithm.
	 */
	public static boolean optimize(final CellDescriptor cell,
			final ImageDescriptor img, final Parameters parameters) {
		double[] X = cell.getParameters();
		int K = X.length - 1;
		double[] dX = new double[K];
		double nDx;
		int iter = 0;
		double gradientStepSize = GRADIENT_STEP_SIZE;
		double eta = ETA;

		do {
			// Compute the energy gradient
			double[] G = energyGradient(cell, img, gradientStepSize);

			// Update the elementary displacement and the parameters
			nDx = 0;
			for (int k = 0; k < K; ++k) {
				dX[k] = -eta * (1 - MU) * G[k] + MU * dX[k];
				X[k] += dX[k];
				nDx += dX[k] * dX[k];
			}
			nDx /= K;

			double decreasingRate = Math.min(1.0 / nDx, 1);
			gradientStepSize *= decreasingRate;
			eta *= decreasingRate;

			cell.setParameters(X);

			// Check cell validity
			if (!((AbstractEllipsoid) cell).isValid()
					|| ((AbstractEllipsoid) cell).getMinimalRadius() < parameters
							.rMin()
					|| ((AbstractEllipsoid) cell).getMaximalRadius() > parameters
							.rMax()) {
				return false;
			}

		} while (nDx > MIN_DISPLACEMENT && ++iter < parameters.maxIter());

		cell.computeEnergy(img);

		return true;
	}

	private static double[] energyGradient(CellDescriptor cell,
			final ImageDescriptor img, double gradientStepSize) {
		final double[] X = cell.getParameters();
		final int K = X.length - 1;
		final double[] Y = X.clone();
		final double[] G = new double[K];

		for (int k = 0; k < K; ++k) {
			Y[k] = X[k] - gradientStepSize;
			cell.setParameters(Y);
			double f0 = cell.computeEnergy(img);

			Y[k] = X[k] + gradientStepSize;
			cell.setParameters(Y);
			double f1 = cell.computeEnergy(img);

			G[k] = 0.5 * (f1 - f0) / gradientStepSize;
			Y[k] = X[k];
		}

		cell.setParameters(X);
		return G;
	}
}
