/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.blobplugin.descriptors;
import icy.type.point.Point4D;
import plugins.big.bigsnakeutils.icy.ellipsoid.Ellipsoid3D;


public class EllipsoidDescriptor extends Ellipsoid3D implements CellDescriptor {
	static final private double SQRT2 = Math.sqrt(2);
	static final private int SAMPLES_COUNT = 30;

	public double _energy;

	/**
	 * Default Constructor.
	 */
	public EllipsoidDescriptor() {
		this(0);
	}

	/**
	 * Default Constructor.
	 * 
	 * @param t
	 *            Time corresponding to this cell.
	 */
	public EllipsoidDescriptor(int t) {
		super(t);
		_energy = 0;
	}

	/**
	 * Copy Constructor.
	 * 
	 * @param other
	 *            Cell to copy.
	 */
	public EllipsoidDescriptor(final EllipsoidDescriptor other) {
		super(other);
		_energy = other._energy;
	}

	@Override
	public void init(Point4D center) {
		x0 = center.getX();
		y0 = center.getY();
		z0 = center.getZ();
		alpha = beta = gamma = 0;
		a = b = c = center.getT();
	}

	@Override
	public double energy() {
		return _energy;
	}

	@Override
	public double[] getParameters() {
		double r = (a + b + c) / 3;
		return new double[] { x0 / r, y0 / r, z0 / r, a / r, b / r, c / r,
				alpha / Math.PI, beta / Math.PI, gamma / Math.PI, r };
	}

	@Override
	public void setParameters(final double[] parameters) {
		double r = parameters[9];
		x0 = r * parameters[0];
		y0 = r * parameters[1];
		z0 = r * parameters[2];
		a = r * parameters[3];
		b = r * parameters[4];
		c = r * parameters[5];
		alpha = Math.PI * parameters[6];
		beta = Math.PI * parameters[7];
		gamma = Math.PI * parameters[8];
	}

	@Override
	public double computeEnergy(final ImageDescriptor imgInt) {
		if (!isValid()) {
			return Double.POSITIVE_INFINITY;
		}

		// Create the shell
		EllipsoidDescriptor shell = new EllipsoidDescriptor();
		shell.a = SQRT2 * a;
		shell.b = SQRT2 * b;
		shell.c = SQRT2 * c;
		shell.x0 = x0;
		shell.y0 = y0;
		shell.z0 = z0;
		shell.alpha = alpha;
		shell.beta = beta;
		shell.gamma = gamma;

		// Compute the integrals
		double I_cell = computeIntegral(imgInt);
		double I_shell = shell.computeIntegral(imgInt);

		// Compute the visible volume
		double V = computeVisibleVolume(imgInt);

		_energy = (I_shell - 2 * I_cell) / V;

		return _energy;
	}

	@Override
	public boolean isClosedTo(final CellDescriptor other) {
		if (!(other instanceof EllipsoidDescriptor)) {
			return false;
		}

		EllipsoidDescriptor ellipsoid = (EllipsoidDescriptor) other;

		double rmean = (a + b + c + ellipsoid.a + ellipsoid.b + ellipsoid.c) / 6.0;
		double dx = x0 - ellipsoid.x0;
		double dy = y0 - ellipsoid.y0;
		double dz = z0 - ellipsoid.z0;

		return (dx * dx + dy * dy + dz * dz < rmean * rmean);
	}

	private double computeIntegral(final ImageDescriptor imgInt) {
		double R[] = getRotationMatrix();
		double I = 0;

		for (int i = 0; i < SAMPLES_COUNT; ++i) {
			double t1 = (2 * i * Math.PI) / SAMPLES_COUNT;
			double c1 = Math.cos(t1);
			double s1 = Math.sin(t1);
			for (int j = 0; j < 0.5 * SAMPLES_COUNT; ++j) {
				double t2 = (2 * j * Math.PI) / SAMPLES_COUNT - 0.5 * Math.PI;
				double c2 = Math.cos(t2);
				double s2 = Math.sin(t2);

				double xp = a * c1 * c2;
				double yp = b * s1 * c2;
				double zp = c * s2;

				int x = (int) Math.min(
						Math.max(x0 + R[0] * xp + R[1] * yp + R[2] * zp, 0),
						imgInt.width - 1);
				int y = (int) Math.min(
						Math.max(y0 + R[3] * xp + R[4] * yp + R[5] * zp, 0),
						imgInt.height - 1);
				int z = (int) Math.min(
						Math.max(z0 + R[6] * xp + R[7] * yp + R[8] * zp, 0),
						imgInt.depth - 1);

				double nx = R[0] * b * c * c1 * c2 * c2 + R[1] * a * c * s1
						* c2 * c2 + R[2] * a * b * s2 * c2;

				I += imgInt.value(x, y, z) * nx;
			}
		}

		return I / (0.5 * SAMPLES_COUNT * SAMPLES_COUNT);
	}

	private double computeVisibleVolume(final ImageDescriptor imgInt) {
		double R[] = getRotationMatrix();
		double V = 0;

		for (int i = 0; i < SAMPLES_COUNT; ++i) {
			double t1 = (2 * i * Math.PI) / SAMPLES_COUNT;
			double c1 = Math.cos(t1);
			double s1 = Math.sin(t1);
			for (int j = 0; j < 0.5 * SAMPLES_COUNT; ++j) {
				double t2 = (2 * j * Math.PI) / SAMPLES_COUNT - 0.5 * Math.PI;
				double c2 = Math.cos(t2);
				double s2 = Math.sin(t2);

				double xp = a * c1 * c2;
				double yp = b * s1 * c2;
				double zp = c * s2;

				int x = (int) Math.min(
						Math.max(x0 + R[0] * xp + R[1] * yp + R[2] * zp, 0),
						imgInt.width - 1);

				double nx = R[0] * b * c * c1 * c2 * c2 + R[1] * a * c * s1
						* c2 * c2 + R[2] * a * b * s2 * c2;

				V += x * nx;
			}
		}

		return V / (0.5 * SAMPLES_COUNT * SAMPLES_COUNT);
	}
}
