package plugins.Yrvann.zhoubaoshi.SphericalParameterization;

import icy.gui.dialog.MessageDialog;

import java.util.ArrayList;
import java.lang.Double;
import java.lang.Math;

import javax.vecmath.Matrix3d;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

import plugins.adufour.activemeshes.mesh.Face;
import plugins.adufour.activemeshes.mesh.Mesh;
import plugins.adufour.activemeshes.mesh.Vertex;


/** 
 *  The following methods are based on the algorithm published in the following article: 
 * 		Fast and Memory Efficient Polygonal Simplification
 *  	Peter Lindstrom, Greg Turk
 *  	In: Proc. IEEE Visualization 1998, 279-286 (1998)
 *   
 *  [Part II.1.1 - Simplification algorithm (initialization)]
 *  - 1) Building the edges
 *  - 2) Adopting Lindstrom's volume preserving method
 *      a) Computing vertex placement (volume preservation & volume optimization)
 *      b) Weighing the edges
 *      
 *  [Part II.1.2 - Simplification algorithm (update)]
 *  - 1) Choosing the edge
 *  - 2) Modifying the mesh
 *  - 3) Capturing the local parameterization
 *  - 4) Updating the weights
 *  
 *  @author Yrvann EMZIVAT (March 2014)
 *  
 **/

public class WeightedEdge {

	static final private double ALPHA = 1; // ALPHA is given in degrees
	
	public ArrayList<Edge> edges;
	public ArrayList<Vertex> targets;
	public ArrayList<Double> weights;

	public WeightedEdge() {
		edges = new ArrayList<Edge>();
		targets = new ArrayList<Vertex>();
		weights = new ArrayList<Double>();
	}
		
	public void init(Mesh Mp) {

	// 1) Building the edges
		for (int i=0; i<Mp.faces.size(); i++) {
			int a = Mp.faces.get(i).v1;
			int b = Mp.faces.get(i).v2;			
			int c = Mp.faces.get(i).v3;	
			boolean Qa = true; 
			boolean Qb = true; 
			boolean Qc = true; 
			
			for (int j=0; j<this.edges.size(); j++) {
				if (((this.edges.get(j).v1 == a)&&(this.edges.get(j).v2 == b)) || ((this.edges.get(j).v2 == a)&&(this.edges.get(j).v1 == b)))
					Qa = false;
				if (((this.edges.get(j).v1 == b)&&(this.edges.get(j).v2 == c)) || ((this.edges.get(j).v2 == b)&&(this.edges.get(j).v1 == c)))
					Qb = false;					
				if (((this.edges.get(j).v1 == c)&&(this.edges.get(j).v2 == a)) || ((this.edges.get(j).v2 == c)&&(this.edges.get(j).v1 == a)))
					Qc = false;	
			}
			if (Qa)
				this.edges.add(new Edge(a,b));
			if (Qb)
				this.edges.add(new Edge(b,c));		
			if (Qc)
				this.edges.add(new Edge(c,a));	
		}
		
		if (Mp.vertices.size() + Mp.faces.size() - 2 != this.edges.size())
			MessageDialog.showDialog(">>> Warning : Processing non zero-genus mesh <<<");			
		

	// 2) Adopting Lindstrom's volume preserving method
		for (int i=0; i<this.edges.size(); i++) {
			int nbConstraints = 0;
			Vector3d recall = new Vector3d();
			int vA = this.edges.get(i).v1;
			int vB = this.edges.get(i).v2;
			

	/** A) Determining the position to which the chosen edge collapses */
			
		// Volume preservation
			Matrix3d A  = new Matrix3d();
			Vector3d a1 = new Vector3d();
			Vector3d a2 = new Vector3d(); 
			Vector3d a3 = new Vector3d();
			Vector3d b  = new Vector3d();
		// Volume optimization		
			Matrix3d Z  = new Matrix3d();
			Matrix3d H  = new Matrix3d();
			Vector3d c  = new Vector3d();
			double k = 0;
			for (int j=0; j<Mp.faces.size(); j++) {	
				Face F = Mp.faces.get(j);
				Vector3d nj = new Vector3d();
				if ((F != null)&&((F.v1 == vA)||(F.v2 == vA)||(F.v3 == vA)||(F.v1 == vB)||(F.v2 == vB)||(F.v3 == vB))) {
					Vector3d Fv1 = new Vector3d();
					Vector3d Fv2 = new Vector3d();
					Vector3d Fv3 = new Vector3d();
					Fv1.x = Mp.vertices.get(F.v1).position.x;
					Fv1.y = Mp.vertices.get(F.v1).position.y;
					Fv1.z = Mp.vertices.get(F.v1).position.z;
					Fv2.x = Mp.vertices.get(F.v2).position.x;
					Fv2.y = Mp.vertices.get(F.v2).position.y;
					Fv2.z = Mp.vertices.get(F.v2).position.z;
					Fv3.x = Mp.vertices.get(F.v3).position.x;
					Fv3.y = Mp.vertices.get(F.v3).position.y;
					Fv3.z = Mp.vertices.get(F.v3).position.z;
					recall.cross(Fv1,Fv2);	nj.x += recall.x; nj.y += recall.y; nj.z += recall.z;
					recall.cross(Fv2,Fv3);	nj.x += recall.x; nj.y += recall.y; nj.z += recall.z;
					recall.cross(Fv3,Fv1);	nj.x += recall.x; nj.y += recall.y; nj.z += recall.z;
					Matrix3d B = new Matrix3d(Fv1.x,Fv2.x,Fv3.x,Fv1.y,Fv2.y,Fv3.y,Fv1.z,Fv2.z,Fv3.z);
					double bj = B.determinant();
					k += bj*bj;

					a1.x += nj.x; 
					a1.y += nj.y; 
					a1.z += nj.z; 
					b.x += bj;
					H.m00 += nj.x*nj.x; H.m11 += nj.y*nj.y; H.m22 += nj.z*nj.z;
					H.m01 += nj.x*nj.y; H.m12 += nj.y*nj.z; H.m20 += nj.x*nj.z;
					c.x += bj*nj.x;
					c.y += bj*nj.y;
					c.z += bj*nj.z;					
				}
			}
			
			H.m10 = H.m01; H.m02 = H.m20; H.m21 = H.m12; 
			H.mul(1./18.);
			c.x /= -18.;
			c.y /= -18.;
			c.z /= -18.;
			k /= 18.;
			
		// Adding the volume preservation constraint (constraint n1)
			Vector3d h1 = new Vector3d();
			Vector3d h2 = new Vector3d();
			Vector3d h3 = new Vector3d();
			H.getColumn(0, h1);
			H.getColumn(1, h2);
			H.getColumn(2, h3);
			
			if ((Math.abs(a1.x) > Double.MIN_VALUE)||(Math.abs(a1.y) > Double.MIN_VALUE)||(Math.abs(a1.z) > Double.MIN_VALUE)) {
				nbConstraints++;
				if (Math.abs(b.x) > Double.MIN_VALUE) {
					a1.x /= b.x;
					a1.y /= b.x;
					a1.z /= b.x;
					b.x = 1;
				}
				else
					a1.normalize();
				
		
		// Adding the volume optimization constraints (constraint n2 and constraint n3)
			
			// Building a basis of IR^3
				Z.setColumn(0,a1);	
				if ((Math.abs(a1.y) < 2*Double.MIN_VALUE) && (Math.abs(a1.z) < 2*Double.MIN_VALUE)) {
					Z.m01 = 0;
					Z.m11 = Math.sqrt(2.);
					Z.m21 = Math.sqrt(2.);
				}
				else if ((Math.abs(a1.z) < 2*Double.MIN_VALUE) && (Math.abs(a1.x) < 2*Double.MIN_VALUE)) {
					Z.m01 = Math.sqrt(2.);
					Z.m11 = 0;
					Z.m21 = Math.sqrt(2.);
				}
				else if ((Math.abs(a1.x) < 2*Double.MIN_VALUE) && (Math.abs(a1.y) < 2*Double.MIN_VALUE)) {
					Z.m01 = Math.sqrt(2.);
					Z.m11 = Math.sqrt(2.);
					Z.m21 = 0;
				}
				else if (Math.abs(a1.x) < 2*Double.MIN_VALUE) {
					Z.m01 = 1;
					Z.m11 = 0;
					Z.m21 = 0;
				}
				else if (Math.abs(a1.y) < 2*Double.MIN_VALUE) {
					Z.m01 = 0;
					Z.m11 = 1;
					Z.m21 = 0;
				}
				else if (Math.abs(a1.z) < 2*Double.MIN_VALUE) {
					Z.m01 = 0;
					Z.m11 = 0;
					Z.m21 = 1;
				}
				else {
					Z.m01 = 0;
					Z.m11 = 1;
					Z.m21 = -a1.y/a1.z;	
				}	
				
				recall.cross(a1,new Vector3d(Z.m01,Z.m11,Z.m21));
				recall.normalize();
				Z.m02 = recall.x; 
				Z.m12 = recall.y;
				Z.m22 = recall.z;
				// System.out.print(Z);
				// *** Building a basis of IR^3 : check
				
				Matrix3d invZ = new Matrix3d();
				invZ.invert(Z);
				Vector3d z1 = new Vector3d();
				Vector3d z2 = new Vector3d();
				invZ.getRow(1, z1);
				invZ.getRow(2, z2);
				
			// Constraint n2
				a2.x = z1.dot(h1);
				a2.y = z1.dot(h2);
				a2.z = z1.dot(h3);
				b.y = -z1.dot(c);

				if (Math.pow(a1.dot(a2),2) < a1.lengthSquared()*a2.lengthSquared()*Math.pow(Math.cos(Math.toRadians(ALPHA)),2)) {
					nbConstraints++;
					if (Math.abs(b.y) > Double.MIN_VALUE) {
						a2.x /= b.y;
						a2.y /= b.y;
						a2.z /= b.y;
						b.y = 1;
					}
					else
						a2.normalize();	
				}		

			// Constraint n3
				a3.x = z2.dot(h1);
				a3.y = z2.dot(h2);
				a3.z = z2.dot(h3);
				b.z = -z2.dot(c);

				recall.cross(a1,a2);
				if (Math.pow(recall.dot(a3),2) > recall.lengthSquared()*a3.lengthSquared()*Math.pow(Math.sin(Math.toRadians(ALPHA)),2)) {
					nbConstraints++;
					if (Math.abs(b.z) > Double.MIN_VALUE) {
						a3.x /= b.z;
						a3.y /= b.z;
						a3.z /= b.z;
						b.z = 1;
					}
					else
						a3.normalize();	
				}
						
		// Matrix inversion
			A.setRow(0,a1);
			A.setRow(1,a2);
			A.setRow(2,a3);
			}

			double detA = A.determinant();
			if ((nbConstraints < 3)||(Math.abs(detA) < 2*Double.MIN_VALUE)) {
				this.targets.add(null);
				this.weights.add(Double.MAX_VALUE);				
			}
			else {
				Matrix3d invA = new Matrix3d();
				invA.invert(A);

			// New vertex position
				invA.getRow(0,recall);	double vx = recall.dot(b);
				invA.getRow(1,recall);	double vy = recall.dot(b);
				invA.getRow(2,recall);	double vz = recall.dot(b);
				this.targets.add(new Vertex(new Point3d(vx,vy,vz)));

				
	/** B) Determining the edge collapse order */		
				
			// Weighing the edge
				double Cv = 1; 	
				double fV = 0;
				
				Vector3d Vi = new Vector3d(vx,vy,vz);
				H.getRow(0, recall);
				fV += Vi.x*recall.dot(Vi);
				H.getRow(1, recall);
				fV += Vi.y*recall.dot(Vi);
				H.getRow(2, recall);
				fV += Vi.z*recall.dot(Vi);
				fV /= 2;			
				fV += c.dot(Vi);				
				fV += k/2;				
				this.weights.add(Cv*fV);			
				
			}	
		}	
		
	}
	
	public void update(Mesh Mp, SplitOperations Op) {
	
	/** A) Choosing the edge */	
		
		int p = 0; 
		double minWeight = Double.MAX_VALUE;
		for (int j=0; j<this.edges.size(); j++) {
			if ((this.weights.get(j) < minWeight)&&(this.targets.get(j) != null)) {
				int e1 = this.edges.get(j).v1; 
				int e2 = this.edges.get(j).v2; 
			
			// The edge is discarded if it causes faces to overlap
			// This happens when a vertex belongs to three faces only, one of which ceases to exist after the edge decimation process
				int safetylevel = 4;
				ArrayList<Integer> distressedVertices = new ArrayList<Integer>();
				for (int k=0; k<Mp.faces.size(); k++) {
					if (!Op.deadFaces.contains(k)) {
						int f1 = Mp.faces.get(k).v1;
						int f2 = Mp.faces.get(k).v2;
						int f3 = Mp.faces.get(k).v3;
	
						if (((f2 == e1)&&(f3 == e2)) || ((f3 == e1)&&(f2 == e2)))
							distressedVertices.add(f1);
						else if (((f3 == e1)&&(f1 == e2)) || ((f1 == e1)&&(f3 == e2)))
							distressedVertices.add(f2);
						else if (((f1 == e1)&&(f2 == e2)) || ((f2 == e1)&&(f1 == e2)))
							distressedVertices.add(f3);		
					}
				}
				for (int k=0; k<distressedVertices.size(); k++) {
					int vtrial = distressedVertices.get(k);
					int safetyK = 0;
					for (int l=0; l<Mp.faces.size(); l++) {
						if ((!Op.deadFaces.contains(l))&&((Mp.faces.get(l).v1 == vtrial)||(Mp.faces.get(l).v2 == vtrial)||(Mp.faces.get(l).v3 == vtrial)))
							safetyK++;
					}
					safetylevel = Math.min(safetylevel, safetyK);
				}

			// The edge is discarded if it severs a section of the mesh
			// This happens when an edge belongs to more than two faces as a result of the edge decimation
				ArrayList<Face> refreshedFaces = new ArrayList<Face>();
				for (int k=0; k<Mp.faces.size(); k++) {
					boolean refresh = false;
					refresh |= (!Op.deadFaces.contains(k)) &&(((Mp.faces.get(k).v1==e1)&&(Mp.faces.get(k).v2==e2)) || ((Mp.faces.get(k).v2==e1)&&(Mp.faces.get(k).v1==e2)));
					refresh |= (!Op.deadFaces.contains(k)) &&(((Mp.faces.get(k).v2==e1)&&(Mp.faces.get(k).v3==e2)) || ((Mp.faces.get(k).v3==e1)&&(Mp.faces.get(k).v2==e2)));
					refresh |= (!Op.deadFaces.contains(k)) && (((Mp.faces.get(k).v3==e1)&&(Mp.faces.get(k).v1==e2)) || ((Mp.faces.get(k).v1==e1)&&(Mp.faces.get(k).v3==e2)));
					if ((!refresh)&&(!Op.deadFaces.contains(k))) {
						Face F = new Face(Mp.faces.get(k).v1, Mp.faces.get(k).v2, Mp.faces.get(k).v3);
						refreshedFaces.add(F);
					}
				}
				for (int k=0; k<refreshedFaces.size(); k++) {
					if (refreshedFaces.get(k).v1 == e1) {
						Face F = new Face(e2,refreshedFaces.get(k).v2,refreshedFaces.get(k).v3);
						refreshedFaces.set(k,F);
					}
					else if (refreshedFaces.get(k).v2 == e1) {
						Face F = new Face(refreshedFaces.get(k).v1,e2,refreshedFaces.get(k).v3);
						refreshedFaces.set(k,F);
					}
					else if (refreshedFaces.get(k).v3 == e1) {
						Face F = new Face(refreshedFaces.get(k).v1,refreshedFaces.get(k).v2,e2);
						refreshedFaces.set(k,F);
					}
				}

				ArrayList<Edge> hangingEdges = new ArrayList<Edge>();
				ArrayList<Integer> count = new ArrayList<Integer>();
				for (int k=0; k<refreshedFaces.size(); k++) {
					int a = refreshedFaces.get(k).v1;
					int b = refreshedFaces.get(k).v2;			
					int c = refreshedFaces.get(k).v3;	
					boolean Qa = true; 
					boolean Qb = true; 
					boolean Qc = true; 
					
					for (int l=0; l<hangingEdges.size(); l++) {
						if (((hangingEdges.get(l).v1 == a)&&(hangingEdges.get(l).v2 == b)) || ((hangingEdges.get(l).v2 == a)&&(hangingEdges.get(l).v1 == b))) {
							Qa = false;		
							count.set(l,count.get(l)+1);
						}
						if (((hangingEdges.get(l).v1 == b)&&(hangingEdges.get(l).v2 == c)) || ((hangingEdges.get(l).v2 == b)&&(hangingEdges.get(l).v1 == c))) {
							Qb = false;	
							count.set(l,count.get(l)+1);
						}
						if (((hangingEdges.get(l).v1 == c)&&(hangingEdges.get(l).v2 == a)) || ((hangingEdges.get(l).v2 == c)&&(hangingEdges.get(l).v1 == a))) {
							Qc = false;	
							count.set(l,count.get(l)+1);
						}
					}
					if (Qa) {
						hangingEdges.add(new Edge(a,b));
						count.add(1);
					}
					if (Qb) {
						hangingEdges.add(new Edge(b,c));
						count.add(1);
					}
					if (Qc) {
						hangingEdges.add(new Edge(c,a));
						count.add(1);
					}
				}

				boolean refresh = true;
				for (int k=0; k<count.size(); k++) {
					if (count.get(k) > 2)
						refresh = false;
				}			
				if ((safetylevel > 3)&&(refresh)) {
					p = j;
					minWeight = this.weights.get(j);
				}
			}
		}

		int vA = this.edges.get(p).v1;
		int vB = this.edges.get(p).v2;
		
	// Absolutely necessary (do not delete) 
		if (vA > vB) {
			int v = vA;
			vA = vB;
			vB = v;
		}
		this.weights.set(p,Double.MAX_VALUE);
		
		
	/** B) Modifying the mesh */	

	// Adding and deleting vertices
		Vector3d vAoldPos = new Vector3d(Mp.vertices.get(vA).position.x,Mp.vertices.get(vA).position.y,Mp.vertices.get(vA).position.z);
		Vector3d vBoldPos = new Vector3d(Mp.vertices.get(vB).position.x,Mp.vertices.get(vB).position.y,Mp.vertices.get(vB).position.z);
		Op.vAlist.add(vA);
		Op.vBlist.add(vB);
		Mp.vertices.set(vA, this.targets.get(p));
		Mp.vertices.set(vB, null);
		
	// Adding and deleting edges
		this.edges.set(p,null);
		ArrayList<Integer> vAstar = new ArrayList<Integer>();
		
		for (int j=0; j<this.edges.size(); j++) {
			if ((this.edges.get(j) != null)&&(j != p)) {
				if (this.edges.get(j).v1 == vA)
					vAstar.add(this.edges.get(j).v2);
				else if (this.edges.get(j).v2 == vA)
					vAstar.add(this.edges.get(j).v1);
			}
		}
		for (int j=0; j<this.edges.size(); j++) {
			if ((this.edges.get(j) != null)&&(j != p)) {
				if ((this.edges.get(j).v1 == vB) && (vAstar.contains(this.edges.get(j).v2))) {
					this.weights.set(j,Double.MAX_VALUE);
					this.edges.set(j,null);
				}
				else if ((this.edges.get(j).v2 == vB) && (vAstar.contains(this.edges.get(j).v1))) {
					this.weights.set(j,Double.MAX_VALUE);
					this.edges.set(j,null);
				}
				else if ((this.edges.get(j).v1 == vB))
					this.edges.get(j).v1 = vA;
				else if ((this.edges.get(j).v2 == vB))
					this.edges.get(j).v2 = vA;
			}
		}
			
	// Adding and deleting faces
		for (int j=0; j<Mp.faces.size(); j++) {
			Face F = Mp.faces.get(j);

			if (!(Op.deadFaces.contains(j))&&((F.v1 == vA)||(F.v2 == vA)||(F.v3 == vA)) && ((F.v1 == vB)||(F.v2 == vB)||(F.v3 == vB)))
				Op.deadFaces.add(j);
			else if (!(Op.deadFaces.contains(j))&&(F.v1 == vB)) {
				if (Op.fupdatePos.size() == Op.nbOperations) {
					ArrayList<Integer> recall = new ArrayList<Integer>();
					recall.add(j);
					Op.fupdatePos.add(recall);
				}
				else
					Op.fupdatePos.get(Op.nbOperations).add(j);
				if (Op.fupdateList.size() == Op.nbOperations) {
					ArrayList<Face> recall = new ArrayList<Face>();
					Face Fcopy = new Face(F.v1,F.v2,F.v3);
					recall.add(Fcopy);
					Op.fupdateList.add(recall);
				}
				else {
					Face Fcopy = new Face(F.v1,F.v2,F.v3);
					Op.fupdateList.get(Op.nbOperations).add(Fcopy);
				}
				Mp.faces.get(j).v1 = vA;
			}
			else if (!(Op.deadFaces.contains(j))&&(F.v2 == vB)) {
				if (Op.fupdatePos.size() == Op.nbOperations) {
					ArrayList<Integer> recall = new ArrayList<Integer>();
					recall.add(j);
					Op.fupdatePos.add(recall);
				}
				else
					Op.fupdatePos.get(Op.nbOperations).add(j);
				if (Op.fupdateList.size() == Op.nbOperations) {
					ArrayList<Face> recall = new ArrayList<Face>();
					Face Fcopy = new Face(F.v1,F.v2,F.v3);
					recall.add(Fcopy);
					Op.fupdateList.add(recall);
				}
				else {
					Face Fcopy = new Face(F.v1,F.v2,F.v3);
					Op.fupdateList.get(Op.nbOperations).add(Fcopy);
				}
				Mp.faces.get(j).v2 = vA;
			}
			else if (!(Op.deadFaces.contains(j))&&(F.v3 == vB)) {
				if (Op.fupdatePos.size() == Op.nbOperations) {
					ArrayList<Integer> recall = new ArrayList<Integer>();
					recall.add(j);
					Op.fupdatePos.add(recall);
				}
				else
					Op.fupdatePos.get(Op.nbOperations).add(j);
				if (Op.fupdateList.size() == Op.nbOperations) {
					ArrayList<Face> recall = new ArrayList<Face>();
					Face Fcopy = new Face(F.v1,F.v2,F.v3);
					recall.add(Fcopy);
					Op.fupdateList.add(recall);
				}
				else {
					Face Fcopy = new Face(F.v1,F.v2,F.v3);
					Op.fupdateList.get(Op.nbOperations).add(Fcopy);
				}
				Mp.faces.get(j).v3 = vA;
			} 			
		}			

		
	/** C) Local Parameterization */ 
	
		ArrayList<Integer> vAfaceList_p = new ArrayList<Integer>();
		ArrayList<Integer> vBfaceList_p = new ArrayList<Integer>();	
		ArrayList<Vector3d> vAparamList_p = new ArrayList<Vector3d>();
		ArrayList<Vector3d> vBparamList_p = new ArrayList<Vector3d>();
		
		ArrayList<Face> Vstar = new ArrayList<Face>();
		ArrayList<Integer> VstarIndex = new ArrayList<Integer>();
		for (int i=0; i<Mp.faces.size(); i++) {	
			boolean containsA = ((!Op.deadFaces.contains(i)))&&(Mp.faces.get(i).v1 == vA || Mp.faces.get(i).v2 == vA || Mp.faces.get(i).v3 == vA);
			if (containsA &&(!Op.deadFaces.contains(i))) {
				Vstar.add(Mp.faces.get(i));
				VstarIndex.add(i);
			}
		}
		
		Vector3d G = new Vector3d();
		for (int i=0; i<Vstar.size(); i++) {	
			if (Vstar.get(i).v1 == vA) {
				G.x += Mp.vertices.get(Vstar.get(i).v2).position.x + Mp.vertices.get(Vstar.get(i).v3).position.x;
				G.y += Mp.vertices.get(Vstar.get(i).v2).position.y + Mp.vertices.get(Vstar.get(i).v3).position.y;	
				G.z += Mp.vertices.get(Vstar.get(i).v2).position.z + Mp.vertices.get(Vstar.get(i).v3).position.z;	
			}
			else if (Vstar.get(i).v2 == vA) {
				G.x += Mp.vertices.get(Vstar.get(i).v3).position.x + Mp.vertices.get(Vstar.get(i).v1).position.x;
				G.y += Mp.vertices.get(Vstar.get(i).v3).position.y + Mp.vertices.get(Vstar.get(i).v1).position.y;	
				G.z += Mp.vertices.get(Vstar.get(i).v3).position.z + Mp.vertices.get(Vstar.get(i).v1).position.z;	
			}
			else if (Vstar.get(i).v3 == vA) {
				G.x += Mp.vertices.get(Vstar.get(i).v1).position.x + Mp.vertices.get(Vstar.get(i).v2).position.x;
				G.y += Mp.vertices.get(Vstar.get(i).v1).position.y + Mp.vertices.get(Vstar.get(i).v2).position.y;	
				G.z += Mp.vertices.get(Vstar.get(i).v1).position.z + Mp.vertices.get(Vstar.get(i).v2).position.z;	
			}
		}		
		G.x /= 2*Vstar.size();
		G.y /= 2*Vstar.size();
		G.z /= 2*Vstar.size();
		
		Vector3d nMean = new Vector3d();

		for (int i=0; i<Vstar.size(); i++) {	
			Vector3d A = new Vector3d(Mp.vertices.get(Vstar.get(i).v1).position.x, Mp.vertices.get(Vstar.get(i).v1).position.y, Mp.vertices.get(Vstar.get(i).v1).position.z);			
			Vector3d B = new Vector3d(Mp.vertices.get(Vstar.get(i).v2).position.x, Mp.vertices.get(Vstar.get(i).v2).position.y, Mp.vertices.get(Vstar.get(i).v2).position.z);
			Vector3d C = new Vector3d(Mp.vertices.get(Vstar.get(i).v3).position.x, Mp.vertices.get(Vstar.get(i).v3).position.y, Mp.vertices.get(Vstar.get(i).v3).position.z);		
			Vector3d AB = new Vector3d();
			Vector3d AC = new Vector3d();
			Vector3d n = new Vector3d();
			
			AB.sub(B,A);
			AC.sub(C,A);
			n.cross(AB,AC);
			nMean.x += n.x;
			nMean.y += n.y;
			nMean.z += n.z;
		}		
		nMean.normalize();

		ArrayList<Vector3d> VstarFlat = new ArrayList<Vector3d>();
		for (int j=0; j<Mp.vertices.size();j++)
			VstarFlat.add(new Vector3d(0,0,0));
		for (int i=0; i<Vstar.size(); i++) {
			if (true) {
				Vector3d M = new Vector3d();
				M.x = Mp.vertices.get(Vstar.get(i).v1).position.x;
				M.y = Mp.vertices.get(Vstar.get(i).v1).position.y;
				M.z = Mp.vertices.get(Vstar.get(i).v1).position.z;
				double lambda = (nMean.x*(G.x-M.x) + nMean.y*(G.y-M.y) + nMean.z*(G.z-M.z));		
				Vector3d H = new Vector3d();
				H.x = M.x + lambda*nMean.x;
				H.y = M.y + lambda*nMean.y;
				H.z = M.z + lambda*nMean.z;
				Vector3d V = new Vector3d(H.x,H.y,H.z);
				VstarFlat.set(Vstar.get(i).v1, V);				
			}
			if (true) {
				Vector3d M = new Vector3d();
				M.x = Mp.vertices.get(Vstar.get(i).v2).position.x;
				M.y = Mp.vertices.get(Vstar.get(i).v2).position.y;
				M.z = Mp.vertices.get(Vstar.get(i).v2).position.z;
				double lambda = (nMean.x*(G.x-M.x) + nMean.y*(G.y-M.y) + nMean.z*(G.z-M.z));		
				Vector3d H = new Vector3d();
				H.x = M.x + lambda*nMean.x;
				H.y = M.y + lambda*nMean.y;
				H.z = M.z + lambda*nMean.z;
				Vector3d V = new Vector3d(H.x,H.y,H.z);
				VstarFlat.set(Vstar.get(i).v2, V);	
			}
			if (true) {
				Vector3d M = new Vector3d();
				M.x = Mp.vertices.get(Vstar.get(i).v3).position.x;
				M.y = Mp.vertices.get(Vstar.get(i).v3).position.y;
				M.z = Mp.vertices.get(Vstar.get(i).v3).position.z;
				double lambda = (nMean.x*(G.x-M.x) + nMean.y*(G.y-M.y) + nMean.z*(G.z-M.z));		
				Vector3d H = new Vector3d();
				H.x = M.x + lambda*nMean.x;
				H.y = M.y + lambda*nMean.y;
				H.z = M.z + lambda*nMean.z;
				Vector3d V = new Vector3d(H.x,H.y,H.z);
				VstarFlat.set(Vstar.get(i).v3, V);	
			}				
		}

		double lambda = 0;
		lambda = (nMean.x*(G.x-vAoldPos.x) + nMean.y*(G.y-vAoldPos.y) + nMean.z*(G.z-vAoldPos.z));			
		Vector3d Aproj = new Vector3d();
		Aproj.x = vAoldPos.x + lambda*nMean.x;
		Aproj.y = vAoldPos.y + lambda*nMean.y;
		Aproj.z = vAoldPos.z + lambda*nMean.z;
		lambda = (nMean.x*(G.x-vBoldPos.x) + nMean.y*(G.y-vBoldPos.y) + nMean.z*(G.z-vBoldPos.z));			
		Vector3d Bproj = new Vector3d();
		Bproj.x = vBoldPos.x + lambda*nMean.x;
		Bproj.y = vBoldPos.y + lambda*nMean.y;
		Bproj.z = vBoldPos.z + lambda*nMean.z;		
		
		for (int i=0; i<Vstar.size(); i++) {				
			Vector3d A = VstarFlat.get(Vstar.get(i).v1);
			Vector3d B = VstarFlat.get(Vstar.get(i).v2);
			Vector3d C = VstarFlat.get(Vstar.get(i).v3);
			TriangleArea ABC = new TriangleArea();	
			ABC.getArea(A,B,C);
		
			Vector3d AB = new Vector3d();
			Vector3d AC = new Vector3d();
			Vector3d n  = new Vector3d();
			
			AB.sub(B,A);
			AC.sub(C,A);
			n.cross(AB,AC);

			TriangleArea HBC = new TriangleArea();
			TriangleArea AHC = new TriangleArea();
			TriangleArea ABH = new TriangleArea();
			
		// Vertex A 						
			HBC.getArea(Aproj,B,C);
			AHC.getArea(A,Aproj,C);
			ABH.getArea(A,B,Aproj);

			double buffer = HBC.area + AHC.area + ABH.area - ABC.area;
			double epsilon = ABC.area / 10000;

			if ((Math.abs(buffer - 2*HBC.area)) < 2*epsilon) 
				HBC.area *= -1.;
			else if ((Math.abs(buffer - 2*AHC.area)) < 2*epsilon) 
				AHC.area *= -1.;
			else if ((Math.abs(buffer - 2*ABH.area)) < 2*epsilon) 
				ABH.area *= -1.;
			else if ((Math.abs(buffer - 2*(AHC.area+ABH.area))) < 2*epsilon)  {
				AHC.area *= -1.;	
				ABH.area *= -1.;
			}
			else if ((Math.abs(buffer - 2*(ABH.area+HBC.area))) < 2*epsilon)  {
				ABH.area *= -1.;
				HBC.area *= -1.;
			}
			else if ((Math.abs(buffer - 2*(HBC.area+AHC.area))) < 2*epsilon)  {
				HBC.area *= -1.;
				AHC.area *= -1.;					
			}
			buffer = HBC.area + AHC.area + ABH.area;
			HBC.area /= buffer;
			AHC.area /= buffer;
			ABH.area /= buffer;

			if ((HBC.area > Double.MIN_VALUE) && (AHC.area > Double.MIN_VALUE) && (ABH.area > Double.MIN_VALUE)) {
				vAfaceList_p.add(VstarIndex.get(i));
				vAparamList_p.add(new Vector3d(HBC.area,AHC.area,ABH.area));	
			}

		// Vertex B	
			HBC.getArea(Bproj,B,C);
			AHC.getArea(A,Bproj,C);
			ABH.getArea(A,B,Bproj);

			buffer = HBC.area + AHC.area + ABH.area - ABC.area;

			if ((Math.abs(buffer - 2*HBC.area)) < 2*epsilon) 
				HBC.area *= -1.;
			else if ((Math.abs(buffer - 2*AHC.area)) < 2*epsilon) 
				AHC.area *= -1.;
			else if ((Math.abs(buffer - 2*ABH.area)) < 2*epsilon) 
				ABH.area *= -1.;
			else if ((Math.abs(buffer - 2*(AHC.area+ABH.area))) < 2*epsilon)  {
				AHC.area *= -1.;	
				ABH.area *= -1.;
			}
			else if ((Math.abs(buffer - 2*(ABH.area+HBC.area))) < 2*epsilon)  {
				ABH.area *= -1.;
				HBC.area *= -1.;
			}
			else if ((Math.abs(buffer - 2*(HBC.area+AHC.area))) < 2*epsilon)  {
				HBC.area *= -1.;
				AHC.area *= -1.;					
			}
			buffer = HBC.area + AHC.area + ABH.area;
			HBC.area /= buffer;
			AHC.area /= buffer;
			ABH.area /= buffer;
			if ((HBC.area > Double.MIN_VALUE) && (AHC.area > Double.MIN_VALUE) && (ABH.area > Double.MIN_VALUE)) {	
				vBfaceList_p.add(VstarIndex.get(i)) ;
				vBparamList_p.add(new Vector3d(HBC.area,AHC.area,ABH.area));
			}

		}	
		

		if (vAfaceList_p.size() != 1) {
			Vector3d recall = new Vector3d();
			int triangleIndex = 0;
			int indexvA = 0;
			double maxDist = Double.MAX_VALUE;
			for (int i=0; i<Vstar.size(); i++) {
				double currentDist = 0.;
				if (Vstar.get(i).v1 != vA) {					
					recall.sub(VstarFlat.get(Vstar.get(i).v1),Aproj);
					currentDist += recall.length();
				}
				if (Vstar.get(i).v2 != vA) {
					recall.sub(VstarFlat.get(Vstar.get(i).v2),Aproj);		
					currentDist += recall.length();
					indexvA = 1;
				}
				if (Vstar.get(i).v3 != vA) {
					recall.sub(VstarFlat.get(Vstar.get(i).v3),Aproj);	
					currentDist += recall.length();
					indexvA = 2;
				}
				
				if (currentDist < maxDist) {
					triangleIndex = i;
					maxDist = currentDist;
				}
					
			}
			if (vAfaceList_p.size() == 0) {
				vAfaceList_p.add(VstarIndex.get(triangleIndex));
				if (indexvA == 0)
					vAparamList_p.add(new Vector3d(0.8,Math.sqrt(0.18),Math.sqrt(0.18)));
				else if (indexvA == 1)
					vAparamList_p.add(new Vector3d(Math.sqrt(0.18),0.8,Math.sqrt(0.18)));
				else if (indexvA == 2)
					vAparamList_p.add(new Vector3d(Math.sqrt(0.18),Math.sqrt(0.18),0.8));
			}
			else {
				vAfaceList_p.set(vAfaceList_p.size()-1,VstarIndex.get(triangleIndex));
				if (indexvA == 0)
					vAparamList_p.set(vAparamList_p.size()-1,new Vector3d(0.8,Math.sqrt(0.18),Math.sqrt(0.18)));
				else if (indexvA == 1)
					vAparamList_p.set(vAparamList_p.size()-1,new Vector3d(Math.sqrt(0.18),0.8,Math.sqrt(0.18)));
				else if (indexvA == 2)
					vAparamList_p.set(vAparamList_p.size()-1,new Vector3d(Math.sqrt(0.18),Math.sqrt(0.18),0.8));
			}
		}
		
		if (vBfaceList_p.size() != 1) {
			Vector3d recall = new Vector3d();
			int triangleIndex = 0;
			int indexvB = 0;
			double maxDist = Double.MAX_VALUE;
			for (int i=0; i<Vstar.size(); i++) {
				double currentDist = 0.;
				if (Vstar.get(i).v1 != vA) {					
					recall.sub(VstarFlat.get(Vstar.get(i).v1),Bproj);
					currentDist += recall.length();
				}
				if (Vstar.get(i).v2 != vA) {
					recall.sub(VstarFlat.get(Vstar.get(i).v2),Bproj);		
					currentDist += recall.length();
					indexvB = 1;
				}
				if (Vstar.get(i).v3 != vA) {
					recall.sub(VstarFlat.get(Vstar.get(i).v3),Bproj);	
					currentDist += recall.length();
					indexvB = 2;
				}
				
				if (currentDist < maxDist) {
					triangleIndex = i;
					maxDist = currentDist;
				}
					
			}
			if (vBfaceList_p.size() == 0) {
				vBfaceList_p.add(VstarIndex.get(triangleIndex));
				if (indexvB == 0)
					vBparamList_p.add(new Vector3d(0.8,Math.sqrt(0.18),Math.sqrt(0.18)));
				else if (indexvB == 1)
					vBparamList_p.add(new Vector3d(Math.sqrt(0.18),0.8,Math.sqrt(0.18)));
				else if (indexvB == 2)
					vBparamList_p.add(new Vector3d(Math.sqrt(0.18),Math.sqrt(0.18),0.8));
			}
			else {
				vBfaceList_p.set(vBfaceList_p.size()-1,VstarIndex.get(triangleIndex));
				if (indexvB == 0)
					vBparamList_p.set(vBparamList_p.size()-1,new Vector3d(0.8,Math.sqrt(0.18),Math.sqrt(0.18)));
				else if (indexvB == 1)
					vBparamList_p.set(vBparamList_p.size()-1,new Vector3d(Math.sqrt(0.18),0.8,Math.sqrt(0.18)));
				else if (indexvB == 2)
					vBparamList_p.set(vBparamList_p.size()-1,new Vector3d(Math.sqrt(0.18),Math.sqrt(0.18),0.8));	
			}
		}		
		
		Op.vAfaceList.add(vAfaceList_p.get(0));
		Op.vAparamList.add(vAparamList_p.get(0)); 	
		Op.vBfaceList.add(vBfaceList_p.get(0));
		Op.vBparamList.add(vBparamList_p.get(0)); 				
			
			
	/** D) Updating the weights */
		
		for (int i=0; i<this.edges.size(); i++) { 
			if ((this.edges.get(i) != null) && ((this.edges.get(i).v1 == Op.vAlist.get(Op.nbOperations))||(this.edges.get(i).v2 == Op.vAlist.get(Op.nbOperations)))) {
				int nbConstraints = 0;
				Vector3d recall = new Vector3d();
				vA = this.edges.get(i).v1;
				vB = this.edges.get(i).v2;

				
			// Volume preservation
				Matrix3d A  = new Matrix3d();
				Vector3d a1 = new Vector3d();
				Vector3d a2 = new Vector3d();
				Vector3d a3 = new Vector3d();
				Vector3d b  = new Vector3d();
			// Volume optimization		
				Matrix3d Z  = new Matrix3d();
				Matrix3d H  = new Matrix3d();
				Vector3d c  = new Vector3d();
				double k = 0;
				for (int j=0; j<Mp.faces.size(); j++) {	
					Face F = Mp.faces.get(j);
					Vector3d nj = new Vector3d();
					if ((!Op.deadFaces.contains(j))&&((F.v1 == vA)||(F.v2 == vA)||(F.v3 == vA)||(F.v1 == vB)||(F.v2 == vB)||(F.v3 == vB))) {
						Vector3d Fv1 = new Vector3d();
						Vector3d Fv2 = new Vector3d();
						Vector3d Fv3 = new Vector3d();
						Fv1.x = Mp.vertices.get(F.v1).position.x;
						Fv1.y = Mp.vertices.get(F.v1).position.y;
						Fv1.z = Mp.vertices.get(F.v1).position.z;
						Fv2.x = Mp.vertices.get(F.v2).position.x;
						Fv2.y = Mp.vertices.get(F.v2).position.y;
						Fv2.z = Mp.vertices.get(F.v2).position.z;
						Fv3.x = Mp.vertices.get(F.v3).position.x;
						Fv3.y = Mp.vertices.get(F.v3).position.y;
						Fv3.z = Mp.vertices.get(F.v3).position.z;
						recall.cross(Fv1,Fv2);	nj.x += recall.x; nj.y += recall.y; nj.z += recall.z;
						recall.cross(Fv2,Fv3);	nj.x += recall.x; nj.y += recall.y; nj.z += recall.z;
						recall.cross(Fv3,Fv1);	nj.x += recall.x; nj.y += recall.y; nj.z += recall.z;
						Matrix3d B = new Matrix3d(Fv1.x,Fv2.x,Fv3.x,Fv1.y,Fv2.y,Fv3.y,Fv1.z,Fv2.z,Fv3.z);
						double bj = B.determinant();
						k += bj*bj;

						a1.x += nj.x; 
						a1.y += nj.y; 
						a1.z += nj.z; 
						b.x += bj;
						H.m00 += nj.x*nj.x; H.m11 += nj.y*nj.y; H.m22 += nj.z*nj.z;
						H.m01 += nj.x*nj.y; H.m12 += nj.y*nj.z; H.m20 += nj.x*nj.z;
						c.x += bj*nj.x;
						c.y += bj*nj.y;
						c.z += bj*nj.z;					
					}
				}
				
				H.m10 = H.m01; H.m02 = H.m20; H.m21 = H.m12; 
				H.mul(1./18.);
				c.x /= -18.;
				c.y /= -18.;
				c.z /= -18.;
				k /= 18.;
				
			// Adding the volume preservation constraint (constraint n1)
				Vector3d h1 = new Vector3d();
				Vector3d h2 = new Vector3d();
				Vector3d h3 = new Vector3d();
				H.getColumn(0, h1);
				H.getColumn(1, h2);
				H.getColumn(2, h3);
				
				if ((Math.abs(a1.x) > Double.MIN_VALUE)||(Math.abs(a1.y) > Double.MIN_VALUE)||(Math.abs(a1.z) > Double.MIN_VALUE)) {
					nbConstraints++;
					if (Math.abs(b.x) > Double.MIN_VALUE) {
						a1.x /= b.x;
						a1.y /= b.x;
						a1.z /= b.x;
						b.x = 1;
					}
					else
						a1.normalize();
		
			// Adding the volume optimization constraints (constraint n2 and constraint n3)
			
				// Building a basis of IR^3
					Z.setColumn(0,a1);	
					if ((Math.abs(a1.y) < 2*Double.MIN_VALUE) && (Math.abs(a1.z) < 2*Double.MIN_VALUE)) {
						Z.m01 = 0;
						Z.m11 = Math.sqrt(2.);
						Z.m21 = Math.sqrt(2.);
					}
					else if ((Math.abs(a1.z) < 2*Double.MIN_VALUE) && (Math.abs(a1.x) < 2*Double.MIN_VALUE)) {
						Z.m01 = Math.sqrt(2.);
						Z.m11 = 0;
						Z.m21 = Math.sqrt(2.);
					}
					else if ((Math.abs(a1.x) < 2*Double.MIN_VALUE) && (Math.abs(a1.y) < 2*Double.MIN_VALUE)) {
						Z.m01 = Math.sqrt(2.);
						Z.m11 = Math.sqrt(2.);
						Z.m21 = 0;
					}
					else if (Math.abs(a1.x) < 2*Double.MIN_VALUE) {
						Z.m01 = 1;
						Z.m11 = 0;
						Z.m21 = 0;
					}
					else if (Math.abs(a1.y) < 2*Double.MIN_VALUE) {
						Z.m01 = 0;
						Z.m11 = 1;
						Z.m21 = 0;
					}
					else if (Math.abs(a1.z) < 2*Double.MIN_VALUE) {
						Z.m01 = 0;
						Z.m11 = 0;
						Z.m21 = 1;
					}
					else {
						Z.m01 = 0;
						Z.m11 = 1;
						Z.m21 = -a1.y/a1.z;	
					}	
					
					recall.cross(a1,new Vector3d(Z.m01,Z.m11,Z.m21));
					recall.normalize();
					Z.m02 = recall.x; 
					Z.m12 = recall.y;
					Z.m22 = recall.z;						
					//System.out.print(Z + "\n");
					Matrix3d invZ = new Matrix3d();
					invZ.invert(Z);
					Vector3d z1 = new Vector3d();
					Vector3d z2 = new Vector3d();
					invZ.getRow(1, z1);
					invZ.getRow(2, z2);
					
				// Constraint n2
					a2.x = z1.dot(h1);
					a2.y = z1.dot(h2);
					a2.z = z1.dot(h3);
					b.y = -z1.dot(c);

					if ((Math.pow(a1.dot(a2),2) < a1.lengthSquared()*a2.lengthSquared()*Math.pow(Math.cos(Math.toRadians(ALPHA)),2))) {
						nbConstraints++;
						if (Math.abs(b.y) > Double.MIN_VALUE) {
							a2.x /= b.y;
							a2.y /= b.y;
							a2.z /= b.y;
							b.y = 1;
						}
						else
							a2.normalize();	
					}			

				// Constraint n3
					a3.x = z2.dot(h1);
					a3.y = z2.dot(h2);
					a3.z = z2.dot(h3);
					b.z = -z2.dot(c);

					recall.cross(a1,a2);
					if (Math.pow(recall.dot(a3),2) > recall.lengthSquared()*a3.lengthSquared()*Math.pow(Math.sin(Math.toRadians(ALPHA)),2)) {
						nbConstraints++;
						if (Math.abs(b.z) > Double.MIN_VALUE) {
							a3.x /= b.z;
							a3.y /= b.z;
							a3.z /= b.z;
							b.z = 1;
						}
						else
							a3.normalize();	
					}
							
				// Matrix inversion
					A.setRow(0,a1);
					A.setRow(1,a2);
					A.setRow(2,a3);
				}

			double detA = A.determinant();
				if ((nbConstraints < 3)||(Math.abs(detA) < 2*Double.MIN_VALUE)) {
					this.targets.set(i,null);
					this.weights.set(i,Double.MAX_VALUE);				
				}
				else {
					Matrix3d invA = new Matrix3d();
					invA.invert(A);
						
				// New vertex position
					invA.getRow(0,recall);	double vx = recall.dot(b);
					invA.getRow(1,recall);	double vy = recall.dot(b);
					invA.getRow(2,recall);	double vz = recall.dot(b);
					this.targets.add(new Vertex(new Point3d(vx,vy,vz)));
					
	
			// Updating the edge collapse order	
					
				// Weighing the edge
					double Cv = 1; 	
					double fV = 0;
						
					Vector3d Vi = new Vector3d(vx,vy,vz);
					H.getRow(0, recall);
					fV += Vi.x*recall.dot(Vi);
					H.getRow(1, recall);
					fV += Vi.y*recall.dot(Vi);
					H.getRow(2, recall);
					fV += Vi.z*recall.dot(Vi);
					fV /= 2;			
					fV += c.dot(Vi);				
					fV += k/2;				
					this.weights.set(i,Cv*fV);
				}			
			}
		}		
		Op.nbOperations++;
	}
	
}
