/*
 *  This file is part of the StereoViewer plug-in for ICY.
 *  Copyright (C) 2012 Ricard Delgado-Gonzalo
 *  
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plugins.rdelgado.stereoviewer;

import icy.canvas.Canvas3D;
import icy.gui.viewer.Viewer;
import icy.image.lut.LUT;
import icy.sequence.Sequence;

import java.awt.Color;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import javax.swing.JFrame;

import plugins.rdelgado.stereoviewer.utils.ScreenUtils;
import vtk.vtkCamera;
import vtk.vtkPanel;
import vtk.vtkRenderer;

/**
 * Class that encapsulates the interaction with the stereo display.
 * 
 * @version February 13, 2012
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class StereoViewDisplayer implements MouseListener, MouseMotionListener,
		KeyListener {
	/** Focused sequence. */
	private Sequence focusedSequence_ = null;
	/** Frame that contains the left camera view. */
	private JFrame fullScreenLeftFrame_;
	/** Frame that contains the right camera view. */
	private JFrame fullScreenRightFrame_;
	/** VTK panel that contains the left camera view. */
	private vtkPanel leftVTKPanel_ = null;
	/** VTK panel that contains the right camera view. */
	private vtkPanel rightVTKPanel_ = null;
	/** Left camera view. */
	private vtkCamera leftActiveCam_ = null;
	/** Right camera view. */
	private vtkCamera rightActiveCam_ = null;
	/** ICY viewer that contains the left camera view. */
	private Viewer leftViewer_ = null;
	/** ICY viewer that contains the right camera view. */
	private Viewer rightViewer_ = null;
	/** VTK renderer of the left view. */
	private vtkRenderer leftRenderer_ = null;
	/** VTK renderer of the right view. */
	private vtkRenderer rightRenderer_ = null;
	/** Color of the background in both views. */
	private double[] backgroundColor_ = { 0, 0, 0 };
	/** LUT of the original view before the plug-in is executed. */
	private LUT originalLUT_ = null;
	/** Value of the separation between cameras. It is initialized as zero. */
	private double eyeDistance_ = 0;
	/**
	 * If <code>true</code>, it indicated that right and left views are swapped.
	 */
	private boolean inverted_ = true;

	// ============================================================================
	// PUBLIC METHODS

	/** Default constructor. */
	public StereoViewDisplayer(Sequence focusedSequence) {
		focusedSequence_ = focusedSequence;
		// create a empty compatible lut
		originalLUT_ = focusedSequence_.createCompatibleLUT();
		// copy the viewer lut in our save lut
		originalLUT_.copyFrom(focusedSequence_.getViewers().get(0).getLut());
	}

	// ----------------------------------------------------------------------------

	/** Creates the full screen views and sets-up the cameras and renderers. */
	public void display() {
		leftViewer_ = new Viewer(focusedSequence_, false);
		leftViewer_.detach();

		Canvas3D left3dCanvas = new Canvas3D(leftViewer_);
		left3dCanvas.setBackground(Color.BLACK);
		leftRenderer_ = left3dCanvas.getRenderer();
		leftRenderer_.LightFollowCameraOff();
		leftRenderer_.SetBackground(backgroundColor_);
		leftActiveCam_ = leftRenderer_.GetActiveCamera();
		leftViewer_.setCanvas(left3dCanvas);
		leftViewer_.setLut(originalLUT_);
		fullScreenLeftFrame_ = ScreenUtils.showOnFullScreen(0,
				leftViewer_.getCanvas());
		leftVTKPanel_ = left3dCanvas.getPanel3D();

		rightViewer_ = new Viewer(focusedSequence_, false);
		rightViewer_.detach();
		Canvas3D right3dCanvas = new Canvas3D(rightViewer_);
		right3dCanvas.setBackground(Color.BLACK);
		rightRenderer_ = right3dCanvas.getRenderer();
		rightRenderer_.LightFollowCameraOff();
		rightRenderer_.SetBackground(backgroundColor_);
		rightActiveCam_ = rightRenderer_.GetActiveCamera();
		rightViewer_.setCanvas(right3dCanvas);
		rightViewer_.setLut(originalLUT_);
		fullScreenRightFrame_ = ScreenUtils.showOnFullScreen(1,
				rightViewer_.getCanvas());
		rightVTKPanel_ = right3dCanvas.getPanel3D();

		// add listeners
		leftVTKPanel_.addMouseListener(this);
		rightVTKPanel_.addMouseListener(this);
		leftVTKPanel_.addMouseMotionListener(this);
		rightVTKPanel_.addMouseMotionListener(this);
		leftVTKPanel_.addKeyListener(this);
		rightVTKPanel_.addKeyListener(this);
	}

	// ----------------------------------------------------------------------------

	/**
	 * When a mouse event is detected in one view, this method fires a refresh
	 * of the complementary view.
	 */
	@Override
	public void mouseDragged(MouseEvent e) {
		Object source = e.getSource();
		if (source == leftVTKPanel_) {
			recomputeRightView();
		} else if (source == rightVTKPanel_) {
			recomputeLeftView();
		}
	}

	// ----------------------------------------------------------------------------

	/** Unused method. */
	@Override
	public void mouseMoved(MouseEvent e) {
	}

	// ----------------------------------------------------------------------------

	/** Handles the interaction with the stereo viewer through the keyboard. */
	@Override
	public void keyPressed(KeyEvent e) {
		char keyChar = e.getKeyChar();
		if ('q' == keyChar) {
			if (leftViewer_ != null) {
				fullScreenLeftFrame_.dispose();
				leftViewer_.close();
			}
			if (rightViewer_ != null) {
				fullScreenRightFrame_.dispose();
				rightViewer_.close();
			}
		} else if ('u' == keyChar) {
			eyeDistance_++;
			recomputeLeftView();
		} else if ('d' == keyChar) {
			if (eyeDistance_ > 0) {
				eyeDistance_--;
				recomputeLeftView();
			}
		} else if ('b' == keyChar) {
			if (backgroundColor_[0] == 0 && backgroundColor_[1] == 0
					&& backgroundColor_[2] == 0) {
				backgroundColor_[0] = 255;
				backgroundColor_[1] = 255;
				backgroundColor_[2] = 255;
			} else if (backgroundColor_[0] == 255 && backgroundColor_[1] == 255
					&& backgroundColor_[2] == 255) {
				backgroundColor_[0] = 0;
				backgroundColor_[1] = 0;
				backgroundColor_[2] = 0;
			}
			rightRenderer_.SetBackground(backgroundColor_);
			leftRenderer_.SetBackground(backgroundColor_);
			rightVTKPanel_.Render();
			leftVTKPanel_.Render();
		} else if ('i' == keyChar) {
			inverted_ = !inverted_;
			recomputeLeftView();
		}
	}

	// ----------------------------------------------------------------------------

	/** Unused method. */
	@Override
	public void keyReleased(KeyEvent e) {
	}

	// ----------------------------------------------------------------------------

	/** Unused method. */
	@Override
	public void keyTyped(KeyEvent e) {
	}

	// ----------------------------------------------------------------------------

	/** Unused method. */
	@Override
	public void mouseClicked(MouseEvent e) {
	}

	// ----------------------------------------------------------------------------

	/** Unused method. */
	@Override
	public void mouseEntered(MouseEvent e) {
	}

	// ----------------------------------------------------------------------------

	/** Unused method. */
	@Override
	public void mouseExited(MouseEvent e) {
	}

	// ----------------------------------------------------------------------------

	/**
	 * Adjusts the refresh rate of the views to provide a fast rendering during
	 * interaction.
	 */
	@Override
	public void mousePressed(MouseEvent e) {
		Object source = e.getSource();
		if (source == leftVTKPanel_) {
			rightRenderer_.GetRenderWindow().SetDesiredUpdateRate(5.0);
			rightVTKPanel_.Render();
		} else if (source == rightVTKPanel_) {
			leftRenderer_.GetRenderWindow().SetDesiredUpdateRate(5.0);
			leftVTKPanel_.Render();
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Adjusts the refresh rate of the views to provide a high definition
	 * rendering when the interaction with the mouse stops.
	 */
	@Override
	public void mouseReleased(MouseEvent e) {
		Object source = e.getSource();
		if (source == leftVTKPanel_) {
			rightRenderer_.GetRenderWindow().SetDesiredUpdateRate(0.01);
			rightVTKPanel_.Render();
		} else if (source == rightVTKPanel_) {
			leftRenderer_.GetRenderWindow().SetDesiredUpdateRate(0.01);
			leftVTKPanel_.Render();
		}
	}

	// ============================================================================
	// PRIVATE METHODS

	/** Renders the left camera view. */
	private void recomputeLeftView() {
		leftActiveCam_.SetClippingRange(rightActiveCam_.GetClippingRange());

		double[] position = rightActiveCam_.GetPosition();
		double[] focalPoint = rightActiveCam_.GetFocalPoint();
		double[] viewUp = rightActiveCam_.GetViewUp();
		double[] directionOfProjection = rightActiveCam_
				.GetDirectionOfProjection();

		// directionOfProjection x viewUp
		double[] eyeVector = {
				viewUp[1] * directionOfProjection[2] - directionOfProjection[1]
						* viewUp[2],
				directionOfProjection[0] * viewUp[2] - viewUp[0]
						* directionOfProjection[2],
				viewUp[0] * directionOfProjection[1] - directionOfProjection[0]
						* viewUp[1] };

		int sign = inverted_ ? 1 : -1;
		leftActiveCam_.SetPosition(position[0] - sign * eyeDistance_
				* eyeVector[0], position[1] - sign * eyeDistance_
				* eyeVector[1], position[2] - sign * eyeDistance_
				* eyeVector[2]);
		leftActiveCam_.SetFocalPoint(focalPoint[0] - sign * eyeDistance_
				* eyeVector[0], focalPoint[1] - sign * eyeDistance_
				* eyeVector[1], focalPoint[2] - sign * eyeDistance_
				* eyeVector[2]);
		leftActiveCam_.SetViewUp(viewUp);
		leftVTKPanel_.Render();
	}

	// ----------------------------------------------------------------------------

	/** Renders the right camera view. */
	private void recomputeRightView() {
		rightActiveCam_.SetClippingRange(leftActiveCam_.GetClippingRange());

		double[] position = leftActiveCam_.GetPosition();
		double[] focalPoint = leftActiveCam_.GetFocalPoint();
		double[] viewUp = leftActiveCam_.GetViewUp();
		double[] directionOfProjection = leftActiveCam_
				.GetDirectionOfProjection();

		// directionOfProjection x viewUp
		double[] eyeVector = {
				viewUp[1] * directionOfProjection[2] - directionOfProjection[1]
						* viewUp[2],
				directionOfProjection[0] * viewUp[2] - viewUp[0]
						* directionOfProjection[2],
				viewUp[0] * directionOfProjection[1] - directionOfProjection[0]
						* viewUp[1] };

		int sign = inverted_ ? 1 : -1;
		rightActiveCam_.SetPosition(position[0] + sign * eyeDistance_
				* eyeVector[0], position[1] + sign * eyeDistance_
				* eyeVector[1], position[2] + sign * eyeDistance_
				* eyeVector[2]);
		rightActiveCam_.SetFocalPoint(focalPoint[0] + sign * eyeDistance_
				* eyeVector[0], focalPoint[1] + sign * eyeDistance_
				* eyeVector[1], focalPoint[2] + sign * eyeDistance_
				* eyeVector[2]);
		rightActiveCam_.SetViewUp(viewUp);
		rightVTKPanel_.Render();
	}
}