package plugins.adufour.activemeshes.producers;

import java.util.ArrayList;

import javax.vecmath.Point3d;

import plugins.adufour.activemeshes.mesh.Face;
import plugins.adufour.activemeshes.mesh.Mesh;
import plugins.adufour.activemeshes.mesh.Vertex;

public class Icosahedron extends MeshProducer
{
	/**
	 * The golden number
	 */
	private static final double	PHI	= (1.0 + Math.sqrt(5.0)) / 2.0;
	
	private final double		radius;
	
	/**
	 * Creates a new mesh initialized by a regular icosahedron of specified size, location and
	 * resolution
	 * 
	 * @param radius
	 *            the new mesh radius
	 * @param center
	 *            the new mesh center
	 * @param resolution
	 *            the mesh resolution
	 */
	public Icosahedron(double radius, double resolution, boolean useVTK)
	{
		super(new Mesh(new ArrayList<Vertex>(12), new ArrayList<Face>(20), resolution, useVTK));
		this.radius = radius;
	}
	
	@Override
	public void run()
	{
		// The icosahedron is defined by 12 vertices enclosed in a cube of size
		// 2PHI. All the Vertices of the icosahedron are located by pair on the
		// cube faces. Each of these vertices is designated by 2 letters 'xy'
		// where:
		// x is the orientation of the cube face containing the vertex
		// y is the orientation of the closest cube face to the vertex
		// Orientations are as follows: (positive = toward the screen)
		// w (west) : x = -PHI
		// e (east) : x = PHI
		// n (north) : y = PHI
		// s (south) : y = -PHI
		// t (top) : z = PHI
		// b (bottom): z = -PHI
		
		mesh.topology.beginUpdate(false);

		// Declaration of the vertices
		int zA = mesh.topology.addVertex(new Point3d(PHI, 1, 0));
		int zB = mesh.topology.addVertex(new Point3d(-PHI, 1, 0));
		int zC = mesh.topology.addVertex(new Point3d(-PHI, -1, 0));
		int zD = mesh.topology.addVertex(new Point3d(PHI, -1, 0));
		int yA = mesh.topology.addVertex(new Point3d(1, 0, PHI));
		int yB = mesh.topology.addVertex(new Point3d(1, 0, -PHI));
		int yC = mesh.topology.addVertex(new Point3d(-1, 0, -PHI));
		int yD = mesh.topology.addVertex(new Point3d(-1, 0, PHI));
		int xA = mesh.topology.addVertex(new Point3d(0, PHI, 1));
		int xB = mesh.topology.addVertex(new Point3d(0, -PHI, 1));
		int xC = mesh.topology.addVertex(new Point3d(0, -PHI, -1));
		int xD = mesh.topology.addVertex(new Point3d(0, PHI, -1));
		
		// Declaration of the faces
		mesh.topology.addFace(yA, xA, yD);
		mesh.topology.addFace(yA, yD, xB);
		mesh.topology.addFace(yB, yC, xD);
		mesh.topology.addFace(yB, xC, yC);
		
		mesh.topology.addFace(zA, yA, zD);
		mesh.topology.addFace(zA, zD, yB);
		mesh.topology.addFace(zC, yD, zB);
		mesh.topology.addFace(zC, zB, yC);
		
		mesh.topology.addFace(xA, zA, xD);
		mesh.topology.addFace(xA, xD, zB);
		mesh.topology.addFace(xB, xC, zD);
		mesh.topology.addFace(xB, zC, xC);

		mesh.topology.addFace(xA, yA, zA);
		mesh.topology.addFace(xD, zA, yB);
		mesh.topology.addFace(yA, xB, zD);
		mesh.topology.addFace(yB, zD, xC);
		mesh.topology.addFace(yD, xA, zB);
		mesh.topology.addFace(yC, zB, xD);
		mesh.topology.addFace(yD, zC, xB);
		mesh.topology.addFace(yC, xC, zC);
		
		// the radius of a sphere bounding a regular icosahedron is:
		// radius = 0.5 * resolution * SQRT( PHI * SQRT(5) )
		// here our definition implies "resolution = 2":
		double currentResolution = 2;
		double currentRadius = 0.5 * currentResolution * Math.sqrt(PHI * Math.sqrt(5));
		
		// scale the mesh to the desired size
		mesh.topology.scale(radius / currentRadius);
		
		mesh.topology.endUpdate();
	}
}
