/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.gui.curve;

import java.awt.geom.Point2D;
import java.util.Vector;

/**
 * Abstract class describing a curve. The curve is defined by a set of control
 * points.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public abstract class Curve {
	protected Vector<Point2D> controlPoints_ = new Vector<Point2D>();

	/** Returns the list of the control points. */
	public Vector<Point2D> getControlPoints() {
		return controlPoints_;
	}

	/** Set the list of the control points. */
	public abstract void setControlPoints(final Vector<Point2D> points);

	/**
	 * Inserts a control point at the given index.
	 * 
	 * @throws IllegalArgumentException
	 *             if the abscissa of the first point isn't zero or if the
	 *             abscissa of the last point isn't one.
	 * @throws ArrayIndexOutOfBoundsException
	 *             if ( index < 0 ) or ( index > count() )
	 */
	public abstract void insertControlPoint(int index, Point2D point);

	/**
	 * Inserts a control point at the given index.
	 * 
	 * @throws IllegalArgumentException
	 *             if the abscissa of the first point isn't zero or if the
	 *             abscissa of the last point isn't one.
	 * @throws ArrayIndexOutOfBoundsException
	 *             if ( index < 0 ) or ( index > count() )
	 */
	public void insertControlPoint(int index, double x, double y) {
		insertControlPoint(index, new Point2D.Double(x, y));
	}

	/**
	 * Removes a control point.
	 * 
	 * @throws ArrayIndexOutOfBoundsException
	 *             if ( index < 0 ) or ( index > count() )
	 */
	public abstract void removeControlPoint(int index);

	/**
	 * Returns the control point for the given index.
	 * 
	 * @throws ArrayIndexOutOfBoundsException
	 *             if ( index < 0 ) or ( index > count() )
	 */
	public Point2D getControlPoint(int index) {
		return controlPoints_.get(index);
	}

	/**
	 * Set the control point for the given index.
	 * 
	 * @throws ArrayIndexOutOfBoundsException
	 *             if ( index < 0 ) or ( index > count() )
	 */
	public abstract void setControlPoint(int index, Point2D point);

	/**
	 * Set the control point for the given index.
	 * 
	 * @throws ArrayIndexOutOfBoundsException
	 *             if ( index < 0 ) or ( index > count() )
	 */
	public void setControlPoint(int index, double x, double y) {
		setControlPoint(index, new Point2D.Double(x, y));
	}

	/**
	 * Remove all the control points except the first and the last ones.
	 */
	public abstract void clear();

	/** Returns the number of control points. */
	public int count() {
		return controlPoints_.size();
	}

	/** Returns the value of the curve for the abscissa x. */
	public abstract double valueAt(double x);

	@Override
	public boolean equals(Object object) {
		if (object instanceof Curve) {
			Curve other = (Curve) object;
			if (count() != other.count()) {
				return false;
			}

			for (int i = 0; i < count(); ++i) {
				if (!controlPoints_.get(i).equals(other.controlPoints_.get(i))) {
					return false;
				}
			}

			return true;
		}
		return false;
	}

	@Override
	public abstract Curve clone();

	/**
	 * * @throws IllegalArgumentException if the index corresponds to the first
	 * point and the abscissa isn't zero or if the index corresponds to the last
	 * point and the abscissa isn't one.
	 */
	protected void checkPointValidity(int index, Point2D point) {
		if (index == 0 && point.getX() != 0) {
			throw new IllegalArgumentException(
					"First point of the curve must be 0");
		}
		if (index == count() && point.getX() != 1) {
			throw new IllegalArgumentException(
					"Last point of the curve must be 1");
		}
	}

}
