from array import array
import sys


# pack a 2D Numpy array in a list that can be sent to a remote 
def pack_image(a):
    # we do not import numpy globally so that this module can be loaded
    # and sent from Icy without errors, where numpy is not available
    import numpy as np

    # Not all the Numpy datatypes will be understood.
    # Not even all the basic numeric ones, since Icy color model is based on
    # java.awt.image.BandedSampleModel, which only supports
    # TYPE_BYTE, TYPE_USHORT, TYPE_SHORT, TYPE_INT, TYPE_FLOAT, and TYPE_DOUBLE.

    if a.ndim <> 2:
        raise ValueError("The array must be 2D.") 

    if a.dtype == np.uint8:
        typecode = 'B'
    elif a.dtype == np.int8:
        typecode = 'b'
    elif a.dtype == np.uint16:
        typecode = 'H'
    elif a.dtype == np.int16:
        typecode = 'h'
    elif a.dtype == np.int32:
        typecode = 'i'
    elif a.dtype == np.float32:
        typecode = 'f'
    elif a.dtype == np.float64:
        typecode = 'd'
    else:
        raise TypeError("Array dtype (%s) is not supported. It must be one of uint8, int8,\n"\
                        " uint16, int16, int32, float32, float64." %(a.dtype)) 

    return [a.shape[0],
            a.shape[1],
            typecode,
            sys.byteorder,
            a.tostring()]


# unpack image data from a remote to a 2D Numpy array
def unpack_image(packed_image):
    # we do not import numpy globally so that this module can be loaded
    # and sent from Icy without errors, where numpy is not available
    import numpy as np
    
    [rows, cols, typecode, byteorder, dataString] = packed_image

    # construct the Numpy data type out of the endianness and array typecode
    if byteorder == 'big':
        dtype = '>'
    else:
        dtype = '<'
    dtype += typecode
    
    # create a 2d Numpy array out of the data
    # Note that we make a copy because frombuffer returns a read-only array
    a = np.frombuffer(dataString, dtype=dtype).copy()
    a.shape = (rows, cols)

    return a
