package plugins.ylemontag.noisegenerator;

/**
 * 
 * @author Yoann Le Montagner
 *
 * Object used to control and interrupt long operations
 */
public class Controller
{
	/**
	 * Exception thrown when a computation have been interrupted
	 */
	public static class CanceledByUser extends Exception
	{
		private static final long serialVersionUID = 1L;
	}
	
	private boolean _cancelFlag ;
	private int     _currentStep;
	private int     _maxSteps   ;
	
	/**
	 * Constructor
	 */
	public Controller()
	{
		_cancelFlag  = false;
		_currentStep = 0;
		_maxSteps    = 0;
	}
	
	/**
	 * Maximum number of steps
	 */
	public int getMaxSteps()
	{
		return _maxSteps;
	}
	
	/**
	 * Set the maximum number of steps
	 */
	public void setMaxSteps(int value)
	{
		_maxSteps = value;
	}
	
	/**
	 * Return the current percentage of progress
	 */
	public double getProgress()
	{
		int currentStep = _currentStep;
		int maxSteps    = _maxSteps   ;
		if(maxSteps<=0) {
			return -1;
		}
		else {
			double progress = (double)currentStep / (double)maxSteps;
			if(progress>=1.0) {
				return 1.0;
			}
			else if(progress<=0.0) {
				return 0.0;
			}
			else {
				return progress;
			}
		}
	}
	
	/**
	 * Method to call to ask the convert function to stop the computation before it finishes
	 * @warning Calling this method does not mean the computation will actually be interrupted.
	 *          The computation is interrupted if a CanceledByUser exception is thrown.
	 */
	public void cancelComputation()
	{
		_cancelFlag = true;
	}
	
	/**
	 * Check-point function called by the computation function
	 */
	public void checkPoint() throws CanceledByUser
	{
		if(_cancelFlag) {
			throw new CanceledByUser();
		}
	}
	
	/**
	 * Increment the step counter by delta, and call the check-point function
	 */
	public void checkPointDelta(int delta) throws CanceledByUser
	{
		_currentStep += delta;
		checkPoint();
	}
	
	/**
	 * Increment the step counter to its maximum value, and call the check-point function
	 */
	public void checkPointFinnished() throws CanceledByUser
	{
		_currentStep = _maxSteps;
		checkPoint();
	}
}
