package plugins.ylemontag.noisegenerator.noisemodels;

import icy.sequence.Sequence;
import icy.sequence.SequenceBuilder;
import icy.type.collection.array.Array1DUtil;
import plugins.ylemontag.noisegenerator.Controller;
import plugins.ylemontag.noisegenerator.NoiseModel;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Base class for pixel separable noise models
 * 
 * A noise model is said to be pixel separable if the random processes that
 * degrades the data at a coordinate (x,y,z,t,c) depends only on the value
 * of the unnoisy sequence at (x,y,z,t,c) and on the coordinates (x,y,z,t,c).
 */
public abstract class PixelSeparableNoise extends NoiseModel
{
	/**
	 * Constructor
	 */
	protected PixelSeparableNoise() {}
	
	@Override
	protected void generateNoise(Sequence in, SequenceBuilder out, Controller controller)
		throws Controller.CanceledByUser
	{
		int sizeX = in.getSizeX();
		int sizeY = in.getSizeY();
		int sizeZ = in.getSizeZ();
		int sizeT = in.getSizeT();
		int sizeC = in.getSizeC();
		boolean inIsSigned = in.getDataType_().isSigned();
		controller.setMaxSteps(sizeT*sizeZ*sizeC);
		for(int t=0; t<sizeT; ++t) {
			for(int z=0; z<sizeZ; ++z) {
				for(int c=0; c<sizeC; ++c) {
					controller.checkPointDelta(1);
					double[] dataIn  = Array1DUtil.arrayToDoubleArray(in.getDataXY(t, z, c), inIsSigned);
					double[] dataOut = out.getDataAsDouble(t, z, c);
					int xy = 0;
					for(int y=0; y<sizeY; ++y) {
						for(int x=0; x<sizeX; ++x) {
							dataOut[xy] = generateNoise(dataIn[xy], x, y, z, t, c);
							++xy;
						}
					}
					out.validateData(t, z, c);
				}
			}
		}
		controller.checkPointFinnished();
	}
	
	/**
	 * Generate a degraded sample given the value of the clean sample 'in',
	 * and the current coordinates
	 */
	protected abstract double generateNoise(double in, int x, int y, int z, int t, int c);
}
