package plugins.ylemontag.ssim;

import icy.gui.frame.progress.AnnounceFrame;
import icy.sequence.Sequence;
import icy.system.IcyHandledException;
import plugins.adufour.ezplug.EzComponent;
import plugins.adufour.ezplug.EzGroup;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzStoppable;
import plugins.adufour.ezplug.EzVarSequence;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * GUI to compute the SSIM map between two sequences
 */
public class SSIMPlugin extends EzPlug implements EzStoppable
{
	private EzVarSequence          _seq1       = new EzVarSequence("Sequence 1");
	private EzVarSequence          _seq2       = new EzVarSequence("Sequence 2");
	private SSIMParameterComponent _parameters = new SSIMParameterComponent();
	private Controller _controller;
	
	@Override
	protected void initialize()
	{
		// Input sequences
		addEzComponent(_seq1);
		addEzComponent(_seq2);
		
		// Parameters
		EzGroup parameterGroup = new EzGroup("Parameters");
		for(EzComponent component : _parameters.getComponents()) {
			parameterGroup.addEzComponent(component);
		}
		addEzComponent(parameterGroup);
	}

	@Override
	protected void execute()
	{
		// Inputs
		SSIMCalculator calculator = retrieveSSIMCalculator();
		Sequence       seq1       = _seq1.getValue(true);
		Sequence       seq2       = _seq2.getValue(true);
		if(!SSIMCalculator.haveSameSize(seq1, seq2)) {
			throw new IcyHandledException(
				"All the sequences involved in a SSIM computation must have the same size."
			);
		}
		
		// Core computation
		try
		{
			_controller = new Controller();
			SSIMCalculator.Result result = calculator.compute(seq1, seq2, _controller);
			showSSIMCalculationResult(seq1, seq2, result.mean);
			addSequence(result.map);
		}
		catch(Controller.CanceledByUser err) {}
		finally {
			_controller = null;
		}
	}

	@Override
	public void clean() {}

	@Override
	public void stopExecution()
	{
		Controller controller = _controller;
		if(controller!=null) {
			controller.cancelComputation();
		}
	}
	
	/**
	 * Return a newly allocated SSIM calculator object corresponding to the parameters
	 * selected by the user
	 */
	private SSIMCalculator retrieveSSIMCalculator()
	{
		try {
			return _parameters.createCalculator();
		}
		catch(IllegalSSIMParameterException err) {
			throw new IcyHandledException("Invalid SSIM parameter: " + err.getMessage());
		}
	}
	
	/**
	 * Show an announce frame with result of a SSIM calculation
	 */
	private static void showSSIMCalculationResult(Sequence seq1, Sequence seq2, double ssimMean)
	{
		new AnnounceFrame(
			"The mean SSIM between " + seq1.getName() + " and " + seq2.getName() + " is: " + ssimMean + "."
		);
	}
}
