/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.snake2D;

import java.awt.geom.Point2D;

/**
 * This class implements a optimization scheme for parametric snakes based on
 * Powell's method. It handles objects that implement the <code>Snake2D</code>
 * interface.
 * 
 * @version May 3, 2014
 * 
 * @author Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 */
public class Snake2DPowellOptimizer extends Snake2DOptimizer {

	/** Square root of the <code>float</code> machine precision. */
	private static final double SQRT_TINY = Math.sqrt(Float
			.intBitsToFloat(0x33FFFFFF));

	/** <code>Float</code> machine precision. */
	private static final double TINY = Float.intBitsToFloat(0x33FFFFFF);

	/** Golden ratio (1.0 + sqrt(5.0))/2.0. */
	private static final double GOLDEN_RATIO = 0.5 + Math.sqrt(1.25);

	/** Golden ratio conjugate (1.0 - sqrt(5.0))/2.0. */
	private static final double COMPLEMENTARY_GOLDEN_RATIO = 1.5 - Math
			.sqrt(1.25);

	/**
	 * The absolute accuracy of the line minimizer. Initialized to the
	 * <code>float</code> machine precision. Determines convergence of the
	 * line-minimization process when the optimal parameter nears zero.
	 */
	private double absoluteAccuracy_ = TINY;

	/**
	 * The size of the first step taken when attempting to bracket the minimum
	 * along a new direction. Initialized to the square root of the
	 * <code>float</code> machine precision.
	 */
	private double initialBracketingStep_ = SQRT_TINY;

	/**
	 * The size of the maximum magnification allowed for a parabolic-fit step
	 * when attempting to bracket the minimum along a new direction. Initialized
	 * to the inverse of the square root of the <code>float</code> machine
	 * precision.
	 */
	private double maximalParabolicBracketingExcursion_ = 1.0 / SQRT_TINY;

	/**
	 * Determines when parabolic bracketing must be abandoned in favor of
	 * arbitrary extrapolation. Initialized to the <code>float</code> machine
	 * precision.
	 */
	private double nonQuadraticBracketingDegeneracy_ = TINY;

	/**
	 * The size of the steps taken when computing a numerical approximation of
	 * the gradient by a finite-difference mechanism. Initialized to the square
	 * root of the <code>float</code> machine precision. Mathematically, one
	 * component <i>g</i> of the gradient is computed as <i>g</i>(<i>x</i>) =
	 * &#189; (<i>g</i>(<i>x</i> + &#916;<i>x</i>) &#8722; <i>g</i>(<i>x</i>
	 * &#8722; &#916;<i>x</i>)) &#8260; &#916;<i>x</i>, where &#916;<i>x</i> is
	 * the quantity given by <code>numericalGradientStepSize</code>.
	 */
	private double numericalGradientStepSize_ = SQRT_TINY;

	/**
	 * The relative accuracy of the line minimizer. Initialized to the square
	 * root of the <code>float</code> machine precision.
	 */
	private double relativeAccuracy_ = SQRT_TINY;

	/**
	 * Determines convergence by virtue of a vanishing norm of the gradient.
	 * Initialized to the square root of the <code>float</code> machine
	 * precision.
	 */
	private double vanishingGradientConvergence_ = SQRT_TINY;

	/**
	 * Determines convergence by virtue of a vanishing line-minimization
	 * relative step. Initialized to the square root of the <code>float</code>
	 * machine precision.
	 */
	private double vanishingStepConvergence_ = SQRT_TINY;

	/**
	 * Determines convergence by virtue of a globally vanishing global update of
	 * the parameters after cycling through all its components. Initialized to
	 * the square root of the <code>float</code> machine precision.
	 */
	private double vanishingTotalDisplacementConvergence_ = SQRT_TINY;

	/** Snake energy. */
	private Double energy_ = null;

	/** Snake being optimized. */
	private Snake2D snake_ = null;

	/** If <code>true</code>, the snake evolution process converged. */
	private boolean optimalSnakeFound_ = false;

	/** If <code>true</code>, the snake evolution process is ongoing. */
	private boolean optimizing_ = false;

	/** If <code>true</code>, the snake died in the evolution process. */
	private boolean snakeDied_ = false;

	// ============================================================================
	// PUBLIC METHODS

	/** This method evolves the snake. */
	@Override
	public void optimize(final Snake2D snake, final Snake2DNode[] configuration) {
		snake_ = snake;
		optimizing_ = true;
		optimalSnakeFound_ = false;
		final int K = configuration.length;
		final Point2D.Double[] V = new Point2D.Double[K];
		for (int k = 0; k < K; k++) {
			V[k] = new Point2D.Double(0.0, 0.0);
		}

		Point2D.Double[] G0 = g(configuration);

		if (null == G0) {
			return;
		}

		double totalDisplacement;
		do {
			double g0 = 0.0;
			for (int k = 0; k < K; k++) {
				V[k].x = -G0[k].x;
				V[k].y = -G0[k].y;
				g0 += G0[k].x * G0[k].x + G0[k].y * G0[k].y;
			}
			if (g0 <= vanishingGradientConvergence_) {
				snake.setNodes(configuration);
				snakeDied_ = !snake.isAlive();
				if ((null == energy_) && !snakeDied_) {
					willProbe(snake);
					energy_ = new Double(snake.energy());
					wasSuccessfulProbing(snake);
				}
				optimalSnakeFound_ = true;
				break;
			}
			totalDisplacement = 0.0;
			for (int n = 0; n <= 2 * K; n++) {
				final double dx = lineMinimization(configuration, V);
				if (dx < 0.0) {
					snake.setNodes(configuration);
					return;
				}
				totalDisplacement += dx;
				Point2D.Double[] G1 = g(configuration);
				if (null == G1) {
					snake.setNodes(configuration);
					return;
				}
				double g1 = 0.0;
				double b = 0.0;
				for (int k = 0; k < K; k++) {
					b += G1[k].x * (G1[k].x - G0[k].x) + G1[k].y
							* (G1[k].y - G0[k].y);
					g1 += G1[k].x * G1[k].x + G1[k].y * G1[k].y;
				}
				if (g1 <= vanishingGradientConvergence_) {
					snake.setNodes(configuration);
					optimalSnakeFound_ = true;
					return;
				} else {
					b /= g0;
					double v = 0.0;
					for (int k = 0; k < K; k++) {
						V[k].x = b * V[k].x - G1[k].x;
						V[k].y = b * V[k].y - G1[k].y;
						v += V[k].x * V[k].x + V[k].y * V[k].y;
					}
					if (v <= vanishingStepConvergence_) {
						snake.setNodes(configuration);
						optimalSnakeFound_ = true;
						return;
					}
					g0 = g1;
					G0 = G1;
				}
			}
		} while (vanishingTotalDisplacementConvergence_ < totalDisplacement);
		snake.setNodes(configuration);
		optimalSnakeFound_ = true;
	}

	// ----------------------------------------------------------------------------

	/** Returns the best energy observed so far during the optimization process. */
	@Override
	public Double reportSnakeBestObservedEnergy() {
		return energy_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns <code>true</code> if the snake died, and <code>false</code>
	 * otherwise.
	 */
	@Override
	public boolean reportSnakeDeath() {
		return snakeDied_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns <code>true</code> if the optimal snake has been found, and
	 * <code>false</code> otherwise.
	 */
	@Override
	public boolean reportSnakeOptimality() {
		return optimalSnakeFound_;
	}

	// ----------------------------------------------------------------------------

	/** Requests stopping the optimization process. */
	@Override
	public void stopOptimizing() {
		optimizing_ = false;
	}

	// ============================================================================
	// PRIVATE METHODS

	/** F function in the Powell's method. */
	private Double f(final Snake2DNode[] X, final double u,
			final Point2D.Double[] V) {
		final int K = X.length;
		final Snake2DNode[] Y = new Snake2DNode[K];
		for (int k = 0; k < K; k++) {
			Y[k] = new Snake2DNode(X[k].x, X[k].y);
			Y[k].x += u * V[k].x;
			Y[k].y += u * V[k].y;
		}
		snake_.setNodes(Y);
		willProbe(snake_);
		snakeDied_ = !snake_.isAlive();
		if (snakeDied_ || !optimizing_) {
			optimizing_ = false;
			return null;
		}
		Double attempt = new Double(snake_.energy());
		if (null == energy_) {
			wasSuccessfulProbing(snake_);
			energy_ = attempt;
		}
		if (0 < energy_.compareTo(attempt)) {
			wasSuccessfulProbing(snake_);
			energy_ = attempt;
		}
		return attempt;
	}

	// ----------------------------------------------------------------------------

	/** G function in the Powell's method. */
	private Point2D.Double[] g(final Snake2DNode[] X) {
		final int K = X.length;
		snake_.setNodes(X);
		willProbe(snake_);
		snakeDied_ = !snake_.isAlive();
		if (snakeDied_ || !optimizing_) {
			optimizing_ = false;
			return null;
		}
		Point2D.Double[] G = snake_.getEnergyGradient();
		if (G != null) {
			final Point2D.Double[] G0 = new Point2D.Double[K];
			for (int k = 0; k < K; k++) {
				if (X[k].isFrozen()) {
					G0[k] = new Point2D.Double(0.0, 0.0);
				} else {
					G0[k] = new Point2D.Double(G[k].x, G[k].y);
				}
			}
			G = G0;
		} else {
			final Snake2DNode[] Y = new Snake2DNode[K];
			G = new Point2D.Double[K];
			for (int k = 0; k < K; k++) {
				Y[k] = new Snake2DNode(X[k].x, X[k].y);
				G[k] = new Point2D.Double(0.0, 0.0);
			}
			for (int k = 0; k < K; k++) {
				if (!X[k].isFrozen()) {
					Y[k].x = X[k].x - numericalGradientStepSize_;
					snake_.setNodes(Y);
					willProbe(snake_);
					snakeDied_ = !snake_.isAlive();
					if (snakeDied_ || !optimizing_) {
						optimizing_ = false;
						return null;
					}
					double f0 = snake_.energy();
					Y[k].x = X[k].x + numericalGradientStepSize_;
					snake_.setNodes(Y);
					willProbe(snake_);
					snakeDied_ = !snake_.isAlive();
					if (snakeDied_ || !optimizing_) {
						optimizing_ = false;
						return null;
					}
					double f1 = snake_.energy();
					G[k].x = 0.5 * (f1 - f0) / numericalGradientStepSize_;
					Y[k].x = X[k].x;
					Y[k].y = X[k].y - numericalGradientStepSize_;
					snake_.setNodes(Y);
					willProbe(snake_);
					snakeDied_ = !snake_.isAlive();
					if (snakeDied_ || !optimizing_) {
						optimizing_ = false;
						return null;
					}
					f0 = snake_.energy();
					Y[k].y = X[k].y + numericalGradientStepSize_;
					snake_.setNodes(Y);
					willProbe(snake_);
					snakeDied_ = !snake_.isAlive();
					if (snakeDied_ || !optimizing_) {
						optimizing_ = false;
						return null;
					}
					f1 = snake_.energy();
					G[k].y = 0.5 * (f1 - f0) / numericalGradientStepSize_;
					Y[k].y = X[k].y;
				}
			}
			snake_.setNodes(X);
			willProbe(snake_);
		}
		return G;
	}

	// ----------------------------------------------------------------------------

	/** Line minimization routine. */
	private double lineMinimization(final Snake2DNode[] X,
			final Point2D.Double[] V) {
		final int K = X.length;
		double a = 0.0;
		final Double Fa = f(X, a, V);
		if (null == Fa) {
			return -1.0;
		}

		double fa = Fa.doubleValue();
		snakeDied_ = !snake_.isAlive();
		if (snakeDied_ || !optimizing_) {
			optimizing_ = false;
			return -1.0;
		}
		double b = initialBracketingStep_;
		final Double Fb = f(X, b, V);
		if (null == Fb) {
			return -1.0;
		}

		double fb = Fb.doubleValue();
		snakeDied_ = !snake_.isAlive();
		if (snakeDied_ || !optimizing_) {
			optimizing_ = false;
			if (fb < fa) {
				for (int k = 0; k < K; k++) {
					X[k].x += b * V[k].x;
					X[k].y += b * V[k].y;
				}
			}
			return -1.0;
		}
		if (fa < fb) {
			final double z = a;
			a = b;
			b = z;
			double f = fa;
			fa = fb;
			fb = f;
		}
		double c = b + GOLDEN_RATIO * (b - a);
		Double Fc = f(X, c, V);
		if (null == Fc) {
			for (int k = 0; k < K; k++) {
				X[k].x += b * V[k].x;
				X[k].y += b * V[k].y;
			}
			return -1.0;
		}
		double fc = Fc.doubleValue();
		if (fc < fb) {
			snakeDied_ = !snake_.isAlive();
			if (snakeDied_ || !optimizing_) {
				optimizing_ = false;
				for (int k = 0; k < K; k++) {
					X[k].x += c * V[k].x;
					X[k].y += c * V[k].y;
				}
				return -1.0;
			}
		}
		double u = c;
		double fu = fc;
		while (fc <= fb) {
			final double r = (b - a) * (fb - fc);
			final double q = (b - c) * (fb - fa);
			u = 0.5 * (b - (b - c) * q + (b - a) * r);
			u = (nonQuadraticBracketingDegeneracy_ < Math.abs(q - r)) ? (u / (q - r))
					: ((r < q) ? (u / nonQuadraticBracketingDegeneracy_)
							: (-u / nonQuadraticBracketingDegeneracy_));
			final double ulim = b + (c - b)
					* maximalParabolicBracketingExcursion_;
			if (0.0 < ((b - u) * (u - c))) {
				Double Fu = f(X, u, V);
				if (null == Fu) {
					for (int k = 0; k < K; k++) {
						X[k].x += c * V[k].x;
						X[k].y += c * V[k].y;
					}
					return -1.0;
				}
				fu = Fu.doubleValue();
				if (fu < fc) {
					snakeDied_ = !snake_.isAlive();
					if (snakeDied_ || !optimizing_) {
						optimizing_ = false;
						for (int k = 0; k < K; k++) {
							X[k].x += u * V[k].x;
							X[k].y += u * V[k].y;
						}
						return -1.0;
					}
					a = b;
					fa = fb;
					b = u;
					fb = fu;
					break;
				} else {
					if (fb < fu) {
						c = u;
						fc = fu;
						break;
					}
				}
				u = c + GOLDEN_RATIO * (c - b);
				Fu = f(X, u, V);
				if (null == Fu) {
					for (int k = 0; k < K; k++) {
						X[k].x += c * V[k].x;
						X[k].y += c * V[k].y;
					}
					return -1.0;
				}
				fu = Fu.doubleValue();
				if (fu < fc) {
					snakeDied_ = !snake_.isAlive();
					if (snakeDied_ || !optimizing_) {
						optimizing_ = false;
						for (int k = 0; k < K; k++) {
							X[k].x += u * V[k].x;
							X[k].y += u * V[k].y;
						}
						return -1.0;
					}
				}
			} else {
				if (0.0 < ((c - u) * (u - ulim))) {
					Double Fu = f(X, u, V);
					if (null == Fu) {
						for (int k = 0; k < K; k++) {
							X[k].x += c * V[k].x;
							X[k].y += c * V[k].y;
						}
						return -1.0;
					}
					fu = Fu.doubleValue();
					if (fu < fc) {
						snakeDied_ = !snake_.isAlive();
						if (snakeDied_ || !optimizing_) {
							optimizing_ = false;
							for (int k = 0; k < K; k++) {
								X[k].x += u * V[k].x;
								X[k].y += u * V[k].y;
							}
							return -1.0;
						}
						b = c;
						c = u;
						u = c + GOLDEN_RATIO * (c - b);
						fb = fc;
						fc = fu;
						Fu = f(X, u, V);
						if (null == Fu) {
							for (int k = 0; k < K; k++) {
								X[k].x += c * V[k].x;
								X[k].y += c * V[k].y;
							}
							return -1.0;
						}
						fu = Fu.doubleValue();
						if (fu < fc) {
							snakeDied_ = !snake_.isAlive();
							if (snakeDied_ || !optimizing_) {
								optimizing_ = false;
								for (int k = 0; k < K; k++) {
									X[k].x += u * V[k].x;
									X[k].y += u * V[k].y;
								}
								return -1.0;
							}
						}
					}
				} else {
					if (0.0 <= ((u - ulim) * (ulim - c))) {
						u = ulim;
						Double Fu = f(X, u, V);
						if (null == Fu) {
							for (int k = 0; k < K; k++) {
								X[k].x += c * V[k].x;
								X[k].y += c * V[k].y;
							}
							return -1.0;
						}
						fu = Fu.doubleValue();
					} else {
						u = c + GOLDEN_RATIO * (c - b);
						Double Fu = f(X, u, V);
						if (null == Fu) {
							for (int k = 0; k < K; k++) {
								X[k].x += c * V[k].x;
								X[k].y += c * V[k].y;
							}
							return -1.0;
						}
						fu = Fu.doubleValue();
					}
					if (fu < fc) {
						snakeDied_ = !snake_.isAlive();
						if (snakeDied_ || !optimizing_) {
							optimizing_ = false;
							for (int k = 0; k < K; k++) {
								X[k].x += u * V[k].x;
								X[k].y += u * V[k].y;
							}
							return -1.0;
						}
					}
				}
			}
			a = b;
			b = c;
			c = u;
			fa = fb;
			fb = fc;
			fc = fu;
		}
		double d = 0.0;
		double e = 0.0;
		double x = b;
		double v = b;
		double w = b;
		double fx = fb;
		double fv = fb;
		double fw = fb;
		if (c < a) {
			b = a;
			a = c;
			fb = fa;
			fa = fc;
		} else {
			b = c;
			fb = fc;
		}

		while (true) {
			final double xm = 0.5 * (a + b);
			final double tol1 = absoluteAccuracy_ + Math.abs(x)
					* relativeAccuracy_;
			final double tol2 = 2.0 * tol1;
			if (Math.abs(x - xm) <= (tol2 - 0.5 * (b - a))) {
				double dx = 0.0;
				for (int k = 0; k < K; k++) {
					X[k].x += x * V[k].x;
					X[k].y += x * V[k].y;
					dx += V[k].x * V[k].x + V[k].y * V[k].y;
				}
				return Math.abs(x) * Math.sqrt(dx);
			}
			if (tol1 < Math.abs(e)) {
				final double r = (x - w) * (fx - fv);
				double q = (x - v) * (fx - fw);
				double p = (x - v) * q - (x - w) * r;
				q = 2.0 * (q - r);
				if (0.0 < q) {
					p = -p;
				}
				q = Math.abs(q);
				final double etemp = e;
				e = d;
				if ((Math.abs(0.5 * q * etemp) <= Math.abs(p))
						|| (p <= (q * (a - x))) || ((q * (b - x)) <= p)) {
					e = (xm <= x) ? (a - x) : (b - x);
					d = COMPLEMENTARY_GOLDEN_RATIO * e;
				} else {
					d = p / q;
					u = x + d;
					if (((u - a) < tol2) || ((b - u) < tol2)) {
						d = (x <= xm) ? (tol1) : (-tol1);
					}
				}
			} else {
				e = (xm <= x) ? (a - x) : (b - x);
				d = COMPLEMENTARY_GOLDEN_RATIO * e;
			}
			u = (tol1 <= Math.abs(d)) ? (x + d) : (x + ((0.0 <= d) ? (tol1)
					: (-tol1)));
			final Double Fu = f(X, u, V);
			if (null == Fu) {
				for (int k = 0; k < K; k++) {
					X[k].x += x * V[k].x;
					X[k].y += x * V[k].y;
				}
				return -1.0;
			}
			fu = Fu.doubleValue();
			if (fu <= fx) {
				snakeDied_ = !snake_.isAlive();
				if (snakeDied_ || !optimizing_) {
					optimizing_ = false;
					for (int k = 0; k < K; k++) {
						X[k].x += u * V[k].x;
						X[k].y += u * V[k].y;
					}
					return -1.0;
				}
				if (x <= u) {
					a = x;
				} else {
					b = x;
				}
				v = w;
				fv = fw;
				w = x;
				fw = fx;
				x = u;
				fx = fu;
			} else {
				if (u < x) {
					a = u;
				} else {
					b = u;
				}
				if ((fu <= fw) || (w == x)) {
					v = w;
					fv = fw;
					w = u;
					fw = fu;
				} else {
					if ((fu <= fv) || (v == x) || (v == w)) {
						v = u;
						fv = fu;
					}
				}
			}
		}
	}

	// ============================================================================
	// GETTERS AND SETTERS

	public double getAbsoluteAccuracy() {
		return absoluteAccuracy_;
	}

	// ----------------------------------------------------------------------------

	public void setAbsoluteAccuracy(double absoluteAccuracy) {
		absoluteAccuracy_ = absoluteAccuracy;
	}

	// ----------------------------------------------------------------------------

	public double getInitialBracketingStep() {
		return initialBracketingStep_;
	}

	// ----------------------------------------------------------------------------

	public void setInitialBracketingStep(double initialBracketingStep) {
		initialBracketingStep_ = initialBracketingStep;
	}

	// ----------------------------------------------------------------------------

	public double getMaximalParabolicBracketingExcursion() {
		return maximalParabolicBracketingExcursion_;
	}

	// ----------------------------------------------------------------------------

	public void setMaximalParabolicBracketingExcursion(
			double maximalParabolicBracketingExcursion) {
		maximalParabolicBracketingExcursion_ = maximalParabolicBracketingExcursion;
	}

	// ----------------------------------------------------------------------------

	public double getNonQuadraticBracketingDegeneracy() {
		return nonQuadraticBracketingDegeneracy_;
	}

	// ----------------------------------------------------------------------------

	public void setNonQuadraticBracketingDegeneracy(
			double nonQuadraticBracketingDegeneracy) {
		nonQuadraticBracketingDegeneracy_ = nonQuadraticBracketingDegeneracy;
	}

	// ----------------------------------------------------------------------------

	public double getNumericalGradientStepSize() {
		return numericalGradientStepSize_;
	}

	// ----------------------------------------------------------------------------

	public void setNumericalGradientStepSize(double numericalGradientStepSize) {
		numericalGradientStepSize_ = numericalGradientStepSize;
	}

	// ----------------------------------------------------------------------------

	public double getRelativeAccuracy() {
		return relativeAccuracy_;
	}

	// ----------------------------------------------------------------------------

	public void setRelativeAccuracy(double relativeAccuracy) {
		relativeAccuracy_ = relativeAccuracy;
	}

	// ----------------------------------------------------------------------------

	public double getVanishingGradientConvergence() {
		return vanishingGradientConvergence_;
	}

	// ----------------------------------------------------------------------------

	public void setVanishingGradientConvergence(
			double vanishingGradientConvergence) {
		vanishingGradientConvergence_ = vanishingGradientConvergence;
	}

	// ----------------------------------------------------------------------------

	public double getVanishingStepConvergence() {
		return vanishingStepConvergence_;
	}

	// ----------------------------------------------------------------------------

	public void setVanishingStepConvergence(double vanishingStepConvergence) {
		vanishingStepConvergence_ = vanishingStepConvergence;
	}

	// ----------------------------------------------------------------------------

	public double getVanishingTotalDisplacementConvergence() {
		return vanishingTotalDisplacementConvergence_;
	}

	// ----------------------------------------------------------------------------

	public void setVanishingTotalDisplacementConvergence(
			double vanishingTotalDisplacementConvergence) {
		vanishingTotalDisplacementConvergence_ = vanishingTotalDisplacementConvergence;
	}
}
