/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 *     Daniel Schmitter (daniel.schmitter@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.shape.priorshapes;

import plugins.big.bigsnakeutils.icy.snake2D.Snake2DNode;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.PriorShape;
import Jama.Matrix;

/**
 * Class that projects spline curves onto an affine invariant shape space.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class ShapeProjector {

	/** Type of prior-shape that is used. */
	private final PriorShape shape_;
	/**
	 * Matrix that encodes the orthogonal projector of an arbitrary shape to the
	 * shape-space.
	 */
	private final Matrix projector_;
	/**
	 * Matrix that encodes the orthogonal projector of an arbitrary shape to the
	 * subspace orthogonal to the shape-space.
	 */
	private final Matrix orthoProjector_;
	/**
	 * Matrix that encodes augmented autocorrelation matrix (B in the paper).
	 */
	private final Matrix B_;
	/** Number of control points of the shape. */
	private final int M_;

	// ============================================================================
	// PUBLIC METHODS

	/** Constructor affine transform (discrete). */
	public ShapeProjector(PriorShape shape, Matrix projector,
			Matrix orthoProjector) {
		shape_ = shape;
		projector_ = projector;
		orthoProjector_ = orthoProjector;
		M_ = projector.getRowDimension();
		B_ = null;
	}
	
	/** Constructor Similarity transform. */
	public ShapeProjector(PriorShape shape, Matrix projector, Matrix orthoProjector, Matrix B) {
		shape_ = shape;
		projector_ = projector;
		M_ = (int)projector.getRowDimension()/2;
		orthoProjector_ = null;
		B_ = B;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns the distance between the shape space and the projection of the
	 * input shape to the orthogonal complement.
	 */
	public double orthoProjectionDistanceAffine(Snake2DNode[] inputNodes) {
		if (inputNodes == null) {
			System.err.println("Input nodes not valid.");
			return Double.POSITIVE_INFINITY;
		}
		if (inputNodes.length != M_) {
			System.err.println("Incompatible length.");
			return Double.POSITIVE_INFINITY;
		}

		double VcnArray[][] = new double[2][inputNodes.length];
		for (int i = 0; i < M_; i++) {
			VcnArray[0][i] = inputNodes[i].x;
			VcnArray[1][i] = inputNodes[i].y;
		}
		Matrix Vcn = new Matrix(VcnArray);
		Vcn = Vcn.times(orthoProjector_);
		double e = Vcn.normF();
		return e * e;
	}
	
	// ----------------------------------------------------------------------------

	/**
	 * Returns the distance between the shape space and the projection of the
	 * input shape to the orthogonal complement.
	 */
	public double orthoProjectionDistanceSimilarity(Snake2DNode[] inputNodes) {
		if (inputNodes == null) {
			System.err.println("Input nodes not valid.");
			return Double.POSITIVE_INFINITY;
		}
		if (inputNodes.length != M_) {
			System.err.println("Incompatible length.");
			return Double.POSITIVE_INFINITY;
		}
		
		double[][] VcnArray = new double[1][2 * inputNodes.length];
		for (int i = 0; i < M_; i++) {
			VcnArray[0][i] = inputNodes[i].x;
			VcnArray[0][i + M_] = inputNodes[i].y;
		}
		Matrix VcnT = new Matrix(VcnArray);
		Matrix Vcn = VcnT.transpose();
		Matrix projectorT_ = projector_.transpose();
		
		// The following matrix corresponds to the continuous domain projector (I-P), c.f. ISBI paper
		Matrix orthoProj = B_.minus(projectorT_.times(B_.times(projector_)));
		
		Matrix energy = VcnT.times(orthoProj.times(Vcn));
		
		return energy.det();
	}
	

	// ----------------------------------------------------------------------------

	/** Projects a set of snake-defining nodes to the corresponding shape-space. */
	public Snake2DNode[] project(Snake2DNode[] inputNodes) {
		if (inputNodes == null) {
			System.err.println("Input nodes not valid.");
			return null;
		}
		if (inputNodes.length != M_) {
			System.err.println("Incompatible length.");
			return inputNodes;
		}

		double VcnArray[][] = new double[2][inputNodes.length];
		for (int i = 0; i < M_; i++) {
			VcnArray[0][i] = inputNodes[i].x;
			VcnArray[1][i] = inputNodes[i].y;
		}
		Matrix Vcn = new Matrix(VcnArray);
		Vcn = Vcn.times(projector_);

		double VcnArrayOut[][] = Vcn.getArray();

		Snake2DNode[] outputNodes = new Snake2DNode[M_];
		for (int i = 0; i < M_; i++) {
			outputNodes[i] = new Snake2DNode(VcnArrayOut[0][i],
					VcnArrayOut[1][i], inputNodes[i].isFrozen(),
					inputNodes[i].isHidden());
		}
		return outputNodes;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Projects a set of snake-defining nodes to the subspace orthogonal to the
	 * corresponding shape-space.
	 */
	public Snake2DNode[] orthoProject(Snake2DNode[] inputNodes) {
		if (inputNodes == null) {
			System.err.println("Input nodes not valid.");
			return null;
		}
		if (inputNodes.length != M_) {
			System.err.println("Incompatible length.");
			return inputNodes;
		}

		double VcnArray[][] = new double[2][inputNodes.length];
		for (int i = 0; i < M_; i++) {
			VcnArray[0][i] = inputNodes[i].x;
			VcnArray[1][i] = inputNodes[i].y;
		}
		Matrix Vcn = new Matrix(VcnArray);
		Vcn = Vcn.times(orthoProjector_);

		double VcnArrayOut[][] = Vcn.getArray();

		Snake2DNode[] outputNodes = new Snake2DNode[M_];
		for (int i = 0; i < M_; i++) {
			outputNodes[i] = new Snake2DNode(VcnArrayOut[0][i],
					VcnArrayOut[1][i], inputNodes[i].isFrozen(),
					inputNodes[i].isHidden());
		}
		return outputNodes;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns the matrix that encodes the orthogonal projector of an arbitrary
	 * shape to the shape-space.
	 */
	public Matrix getProjectionMatrix() {
		return projector_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns the matrix that encodes the orthogonal projector of an arbitrary
	 * shape to the subspace orthogonal to the shape-space.
	 */
	public Matrix getOrthoprojectionMatrix() {
		return orthoProjector_;
	}

	// ----------------------------------------------------------------------------

	/** Returns the type of prior-shape that is used. */
	public PriorShape getShape() {
		return shape_;
	}
}
