/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnake.core;

import icy.gui.frame.progress.ProgressFrame;
import icy.image.IcyBufferedImage;
import icy.image.IcyBufferedImageUtil;
import icy.sequence.Sequence;
import icy.sequence.SequenceUtil;
import icy.type.DataType;

import java.util.Arrays;

import plugins.big.bigsnakeutils.process.process2D.FastGaussianFilter;
import plugins.big.bigsnakeutils.process.process2D.Filters;

/**
 * Container for the LUTs.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class ImageLUTContainer {

	// ----------------------------------------------------------------------------
	// IMAGE

	/** <code>Sequence</code> of interest. */
	private Sequence sequence_ = null;
	/** Array containing the image data of the active channel. */
	private double[] imageDataArray_ = null;
	/**
	 * Array containing a filtered image data of the active channel using a
	 * smoothed Laplacian filter.
	 */
	private double[] filteredImageDataArray_ = null;
	/**
	 * Array containing the preintegrated image data along the vertical
	 * direction of the active channel.
	 */
	private double[] preintegratedImageDataArray_ = null;
	/**
	 * Array containing the preintegrated and filtered image data along the
	 * vertical direction of the active channel with a smoothed Laplacian
	 * filter.
	 */
	private double[] preintegratedFilteredImageDataArray_ = null;

	// ----------------------------------------------------------------------------
	// IMAGE PROPERTIES

	/** Width of the original image. */
	private int width_ = 0;
	/** Height of the original image. */
	private int height_ = 0;

	// ----------------------------------------------------------------------------
	// PARAMETERS

	/** Image smoothing factor. */
	private double sigma_ = 0;
	/** Active channel of the image. */
	private int channelNumber_ = 0;

	// ----------------------------------------------------------------------------
	// OTHER

	/** Smallest discretization of a Laplacian filter. */
	private static int LAPLACIAN_KERNEL[] = { 0, -1, 0, -1, 4, -1, 0, -1, 0 };
	/** If <code>true</code>, the LUTs are consistent with the input parameters. */
	private boolean isLUTUpToDate_ = false;

	// ----------------------------------------------------------------------------
	// XML LABELS

	/**
	 * Label of the XML tag containing the standard deviation of the Gaussian
	 * smoothing applied to the input image when computing the contour energy.
	 */
	public static final String ID_SIGMA = "sigma";
	/**
	 * Label of the XML tag containing the information of the channel number
	 * used in the snake.
	 */
	public static final String ID_ACTIVECHANNEL = "activeChannel";

	// ============================================================================
	// PUBLIC METHODS

	/** Default constructor. */
	public ImageLUTContainer() {
	}

	// ----------------------------------------------------------------------------

	/** Constructs the look-up-tables that the snake needs to function. */
	public void buildLUTs() throws Exception {
		// FIXME check the duplication of data here
		ProgressFrame pFrame = new ProgressFrame("Precomputing LUTs");
		pFrame.setLength(4 * width_);

		Sequence singleChannelSequence = SequenceUtil.extractChannel(sequence_,
				channelNumber_);
		IcyBufferedImage inputImage = singleChannelSequence.getFirstImage();
		IcyBufferedImage inputCopy = IcyBufferedImageUtil.convertToType(
				inputImage, DataType.DOUBLE, false);
		imageDataArray_ = inputCopy.getDataXYAsDouble(0);
		width_ = singleChannelSequence.getWidth();
		height_ = singleChannelSequence.getHeight();

		filteredImageDataArray_ = Arrays.copyOf(imageDataArray_,
				imageDataArray_.length);
		pFrame.setPosition(width_);

		FastGaussianFilter.smooth(filteredImageDataArray_, width_, height_,
				sigma_);
		pFrame.setPosition(2 * width_);

		Filters.filter3x3(filteredImageDataArray_, width_, height_,
				LAPLACIAN_KERNEL);
		pFrame.setPosition(3 * width_);

		preintegratedImageDataArray_ = new double[width_ * height_];
		preintegratedFilteredImageDataArray_ = new double[width_ * height_];

		double fuyLap_val, fuy_val;
		for (int x = 0; x < width_; x++) {
			fuy_val = 0.0;
			fuyLap_val = 0;
			for (int y = 0; y < height_; y++) {
				int index = x + width_ * y;
				fuy_val += imageDataArray_[index];
				preintegratedImageDataArray_[index] = fuy_val;
				fuyLap_val += filteredImageDataArray_[index];
				preintegratedFilteredImageDataArray_[index] = fuyLap_val;
			}
			pFrame.incPosition();
		}
		pFrame.close();
		isLUTUpToDate_ = true;
	}

	// ----------------------------------------------------------------------------

	/** If <code>true</code>, the LUTs are consistent with the input parameters. */
	public boolean isLUTUpToDate() {
		return isLUTUpToDate_;
	}

	// ----------------------------------------------------------------------------

	/** Returns a string with all the information of the LUT. */
	@Override
	public String toString() {
		return new String("[LUT container: sigma = " + sigma_
				+ ", sequence title = " + sequence_.getName()
				+ ", channel number = " + channelNumber_
				+ ", LUT up to date = " + isLUTUpToDate_ + "]");
	}

	// ============================================================================
	// SETTERS AND GETTERS

	/** Sets the active channel. */
	public void setChannelNumber(int channelNumber) {
		isLUTUpToDate_ = false;
		channelNumber_ = channelNumber;
	}

	// ----------------------------------------------------------------------------

	/** Returns the active channel. */
	public int getChannelNumber() {
		return channelNumber_;
	}

	// ----------------------------------------------------------------------------

	/** Sets the standard deviation of the Gaussian kernel used for smoothing. */
	public void setSigma(double sigma) {
		isLUTUpToDate_ = false;
		if (sigma < 0) {
			sigma_ = 0;
		} else {
			sigma_ = sigma;
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns the standard deviation of the Gaussian kernel used for smoothing.
	 */
	public double getSigma() {
		return sigma_;
	}

	// ----------------------------------------------------------------------------

	/** Sets the active image from where the data is extracted. */
	public void setSequence(Sequence sequence) {
		isLUTUpToDate_ = false;
		sequence_ = sequence;
	}

	// ----------------------------------------------------------------------------

	/** Returns the image from where the image data is extracted. */
	public Sequence getOriginalSequence() {
		return sequence_;
	}

	// ----------------------------------------------------------------------------

	/** Returns the with in pixels of the image. */
	public int getImageWidth() {
		return width_;
	}

	// ----------------------------------------------------------------------------

	/** Returns the height in pixels of the image. */
	public int getImageHeight() {
		return height_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns a one dimensional array containing the pixel values of the
	 * original image of the active channel.
	 */
	public double[] getImageDataArray() {
		return imageDataArray_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns a one dimensional array containing the pixel values of the image
	 * after applying a smoothed Laplacian filter of the active channel.
	 */
	public double[] getFilteredImageDataArray() {
		return filteredImageDataArray_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns a one dimensional array containing the pixel values of the
	 * original image after preintegration of the active channel.
	 */
	public double[] getPreintegratedImageDataArray() {
		return preintegratedImageDataArray_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns a one dimensional array containing the pixel values of the image
	 * after applying a smoothed Laplacian filter after preintegration of the
	 * active channel.
	 */
	public double[] getPreintegratedFilteredImageDataArray() {
		return preintegratedFilteredImageDataArray_;
	}
}