package plugins.fab.internetconnectivitymonitor;

import icy.network.NetworkUtil;
import icy.system.IcyHandledException;
import icy.system.thread.ThreadUtil;
import icy.util.ClassUtil;

import java.io.IOException;
import java.util.Calendar;
import java.util.regex.Pattern;

import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzStoppable;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarEnum;
import plugins.adufour.ezplug.EzVarText;

public class InternetConnectivityMonitor extends EzPlug implements EzStoppable
{
	/**
	 * Checking mode
	 */
	private static enum Mode
	{
		INTERNET("Internet"),
		PING    ("Ping"    );
		
		private String _name;
		
		private Mode(String name)
		{
			_name = name;
		}
		
		@Override
		public String toString()
		{
			return _name;
		}
	}
	
	EzVarEnum<Mode> _mode   = new EzVarEnum<Mode>("Mode", Mode.values());
	EzVarText       _target = new EzVarText("Target");
	EzVarBoolean    _load   = new EzVarBoolean("Load archives", false);
	
	Pattern _targetValidator = null;
	boolean _interruptFlag = false;
	
	@Override
	protected void initialize()
	{
		addEzComponent(_mode  );
		addEzComponent(_target);
		addEzComponent(_load  );
		_mode.addVisibilityTriggerTo(_target, Mode.PING);
	}
	
	@Override
	protected void execute()
	{
		_interruptFlag = false;
		Mode    mode = _mode.getValue();
		boolean load = _load.getValue();
		
		// Load archives mode
		if(load) {
			ConnectivityMap map = null;
			switch(mode) {
				case INTERNET: map = getInternetConnectivityMap(); break;
				case PING: map = getPingConnectivityMap(getPingTarget()); break;
			}
			loadData(map);
		}
		
		// Live mode
		else {
			switch(mode)
			{
				case INTERNET:
					ThreadUtil.bgRun(new Runnable()
					{
						@Override
						public void run()
						{
							InternetMonitoringDaemon daemon = new InternetMonitoringDaemon();
							daemon.run();
						}
					});
					break;
				
				case PING:
					final String target = getPingTarget();
					ThreadUtil.bgRun(new Runnable()
					{
						@Override
						public void run()
						{
							PingMonitoringDaemon daemon = new PingMonitoringDaemon(target);
							daemon.run();
						}
					});
					break;
			}
		}
	}
	
	@Override
	public void clean() {}
	
	@Override
	public void stopExecution()
	{
		_interruptFlag = true;
	}
	
	/**
	 * Internet connectivity map
	 */
	private ConnectivityMap getInternetConnectivityMap()
	{
		return new ConnectivityMap(rootDirectory() + "/internet", "Internet connectivity monitor", 5);
	}
	
	/**
	 * Ping connectivity map
	 */
	private ConnectivityMap getPingConnectivityMap(String target)
	{
		return new ConnectivityMap(rootDirectory() + "/ping/" + target, "Ping: " + target, 10);
	}
	
	/**
	 * Monitoring daemon for Internet
	 */
	private class InternetMonitoringDaemon extends MonitoringDeamon
	{
		public InternetMonitoringDaemon()
		{
			super(getInternetConnectivityMap());
		}

		@Override
		protected boolean checkConnectivity()
		{
			return NetworkUtil.hasInternetAccess();
		}
	}
	
	/**
	 * Monitoring daemon for ping
	 */
	private class PingMonitoringDaemon extends MonitoringDeamon
	{
		private String _command;
		
		public PingMonitoringDaemon(String target)
		{
			super(getPingConnectivityMap(target));
			_command = "ping -q -c 1 -t 1 " + target;
		}

		@Override
		protected boolean checkConnectivity()
		{
			try {
				Process p = Runtime.getRuntime().exec(_command);
				return p.waitFor()==0;
			}
			catch(IOException e) {
				return false;
			}
			catch (InterruptedException e) {
				return false;
			}
		}
	}
	
	/**
	 * Return the ping target and ensure that it is well-formated
	 */
	private String getPingTarget()
	{
		// Trim the string entered by the user, and check that it is not empty
		String retVal = _target.getValue();
		if(retVal!=null) {
			retVal = retVal.trim();
		}
		if(retVal==null || retVal.isEmpty()) {
			throw new IcyHandledException("'Target' is empty.");
		}
		
		// URL are not case-sensitive => force lower case
		retVal = retVal.toLowerCase();
		
		// Only a subset of characters is allowed
		if(_targetValidator==null) {
			_targetValidator = Pattern.compile("[a-z0-9\\-_\\.]*");
		}
		if(!_targetValidator.matcher(retVal).matches()) {
			throw new IcyHandledException("'" + retVal + "' is not a valid URL/IP.");
		}
		
		// OK now!
		return retVal;
	}
	
	/**
	 * Load data between the selected date bounds into the given map
	 */
	private void loadData(ConnectivityMap map)
	{
		Calendar begin = convert(map.getFirstArchivedDay());
		Calendar end   = convert(map.getLastArchivedDay ());
		if(begin==null || end==null) {
			throw new IcyHandledException("No existing archives");
		}
		for(Calendar pos=begin; pos.compareTo(end)<=0; pos.add(Calendar.DAY_OF_MONTH, 1))
		{
			// Check the interrupt flag
			if(_interruptFlag) {
				return;
			}
			
			// Load the image corresponding to the current day
			int year  = pos.get(Calendar.YEAR        );
			int month = pos.get(Calendar.MONTH       )+1; // Java returns month valued between 0 and 11 by default :-( 
			int day   = pos.get(Calendar.DAY_OF_MONTH);
			map.load(year, month, day);
		}
	}
	
	/**
	 * Convert a ConnectivityMap.Info into a CalendarObject
	 */
	public static Calendar convert(ConnectivityMap.Info info)
	{
		if(info==null) {
			return null;
		}
		Calendar retVal = Calendar.getInstance();
		retVal.set(info.year, info.month-1, info.day);
		return retVal;
	}
	
	/**
	 * Plugin root directory
	 */
	private String rootDirectory()
	{
		return ClassUtil.getPathFromQualifiedName(getDescriptor().getPackageName());
	}
}
