package plugins.adufour.filtering;

import icy.plugin.abstract_.Plugin;
import icy.plugin.interface_.PluginBundled;
import icy.sequence.Sequence;
import icy.sequence.SequenceUtil;
import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.vars.gui.model.DoubleRangeModel;
import plugins.adufour.vars.gui.model.RangeModel;
import plugins.adufour.vars.lang.VarDouble;
import plugins.adufour.vars.lang.VarSequence;

/**
 * Block version of the filter tool box for the specific case of Gaussian filtering
 * 
 * @author Alexandre Dufour
 */
public class GaussianFilter extends Plugin implements Block, PluginBundled
{
    VarSequence input  = new VarSequence("input", null);
    
    VarDouble   gX     = new VarDouble("sigma (x)", 0.0);
    VarDouble   gY     = new VarDouble("sigma (y)", 0.0);
    VarDouble   gZ     = new VarDouble("sigma (z)", 0.0);
    
    VarSequence output = new VarSequence("output", null);
    
    @Override
    public void run()
    {
        output.setValue(filter(input.getValue(true), gX.getValue(), gY.getValue(), gZ.getValue()));
    }
    
    @Override
    public void declareInput(VarList inputMap)
    {
        RangeModel<Double> constraint = new DoubleRangeModel(0.0, 0.0, 1000.0, 0.1);
        gX.setDefaultEditorModel(constraint);
        gY.setDefaultEditorModel(constraint);
        gZ.setDefaultEditorModel(constraint);
        inputMap.add("input", input);
        inputMap.add("filterX", gX);
        inputMap.add("filterY", gY);
        inputMap.add("filterZ", gZ);
    }
    
    @Override
    public void declareOutput(VarList outputMap)
    {
        outputMap.add("output", output);
    }
    
    @Override
    public String getMainPluginClassName()
    {
        return FilterToolbox.class.getCanonicalName();
    }
    
    /**
     * Applies a gaussian filter (also known as "blur") on the specified sequence
     * 
     * @param in
     *            the sequence to filter
     * @param sigmaX
     *            the standard deviation of the gaussian along the X axis (set to 0 if not needed)
     * @param sigmaY
     *            the standard deviation of the gaussian along the Y axis (set to 0 if not needed)
     * @param sigmaZ
     *            the standard deviation of the gaussian along the Z axis (set to 0 if not needed)
     * @return a filtered version of the input sequence
     */
    public static Sequence filter(Sequence in, double sigmaX, double sigmaY, double sigmaZ)
    {
        double[] gaussianX = Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(sigmaX).getData();
        double[] gaussianY = Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(sigmaY).getData();
        double[] gaussianZ = Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(sigmaZ).getData();
        
        Sequence filtered = SequenceUtil.getCopy(in);
        filtered.setName(in.getName() + "_filtered");
        
        try
        {
            Convolution1D.convolve(filtered, gaussianX == null ? null : gaussianX, gaussianY == null ? null : gaussianY,
                    gaussianZ == null ? null : filtered.getSizeZ() > 1 ? gaussianZ : null);
        }
        catch (Exception e)
        {
            throw new RuntimeException("GaussianFilter: " + e.getMessage());
        }
        
        return filtered;
    }
}
