package plugins.fab.challengescoring;

import icy.system.profile.Chronometer;
import icy.system.thread.ThreadUtil;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Date;

import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarFolder;
import plugins.fab.trackmanager.TrackSegment;
import plugins.nchenouard.trackprocessorperformance.DistanceTypes;
import plugins.nchenouard.trackprocessorperformance.PerformanceAnalyzer;
import plugins.nchenouard.trackprocessorperformance.TrackProcessorPerformance;

public class ChallengeScoring extends EzPlug {

	EzVarFolder folderGroundTruth = new EzVarFolder( "Ground Truth Folder" , "" );
	EzVarFolder folderCandidate = new EzVarFolder( "Canditate Folder" , "" );
	EzVarBoolean useImprovedMethod = new EzVarBoolean( "Use improved method" , true );
	EzVarDouble minimumDistanceBetweenDetections = new EzVarDouble( "Minimum distance between detections" , 5 , 0, 100, 0.001 );
		
	@Override
	protected void initialize() {

		addEzComponent( folderGroundTruth );
		addEzComponent( folderCandidate );
		addEzComponent( useImprovedMethod );
		addEzComponent( minimumDistanceBetweenDetections );

	}

	@Override
	protected void execute() {
				
		System.out.println("Starting...");
		boolean useImprovedMethodCopy = useImprovedMethod.getValue(); // avoid the user changes in GUI while running.
		double minimumDistance = minimumDistanceBetweenDetections.getValue(); // avoid the user changes in GUI while running.
		
		ArrayList<File> groundTruthFiles = new ArrayList<File>();
		ArrayList<File> candidateFiles = new ArrayList<File>();
		
		ArrayList<ChallengeDocument> groundTruthDocumentList = new ArrayList<ChallengeDocument>();
		ArrayList<ChallengeDocument> candidateDocumentList = new ArrayList<ChallengeDocument>();
		
		System.out.println("Loading Ground Truth XML Files...");
		
		groundTruthFiles = new RecursiveFileListBuilder( folderGroundTruth.getValue() , "xml" ).getFileList();			
		
		for ( File file : groundTruthFiles )
		{
			System.out.println( "Loading " + file + "...");
			ChallengeDocument cd = new ChallengeDocument( file );
			groundTruthDocumentList.add( cd );
		}
		
		System.out.println("Loading Candidate XML Files...");
		
		candidateFiles = new RecursiveFileListBuilder( folderCandidate.getValue() , "xml" ).getFileList();			
		
		for ( File file : candidateFiles )
		{
			ChallengeDocument cd = new ChallengeDocument( file );
			candidateDocumentList.add( cd );
		}
		
		System.out.println("Processing scoring...");
		
		for ( ChallengeDocument candidateDocument : candidateDocumentList )
		{
			System.out.println("---------------------------");
			System.out.println("Scoring " + candidateDocument );
			
			ChallengeDocument groundTruthDocument = findGroundTruth( candidateDocument , groundTruthDocumentList );
			System.out.println( "Ground truth: " + groundTruthDocument );
			
			
			Date date = new Date();
			System.out.println( "Starting time: "+ date );
			Chronometer c = new Chronometer("Score computation time ");
						
			ChronoFrame cf = new ChronoFrame( "Scoring " + candidateDocument.getSourceFile().getAbsolutePath() );
			
			scoreCandidate( candidateDocument , groundTruthDocument , useImprovedMethodCopy , minimumDistance );
			
			cf.close();
			
			c.displayInSeconds();
			
		}
		
		System.out.println("----------------------------");
		System.out.println("Score computation finished !");		

	}

	
	TrackProcessorPerformance tpp ;

	private void scoreCandidate(
			ChallengeDocument candidateDocument, ChallengeDocument groundTruthDocument , boolean useImprovedMethod, 
			double minimumDistanceBetweenDetections  
			)
	{			
		
		ThreadUtil.invokeNow( new Runnable() {
			
			@Override
			public void run() {
				tpp = new TrackProcessorPerformance();				
			}
			
		});
		
		System.out.println("Loading ground truth...");
		ArrayList<TrackSegment> groundTruthTrackSegmentList = groundTruthDocument.getTrackSegmentList();
		System.out.println("Loading candidate...");
		ArrayList<TrackSegment> candidateTrackSegmentList = candidateDocument.getTrackSegmentList();
		
		System.out.println("Scoring...");
		PerformanceAnalyzer performanceAnalyser = tpp.pairTracks( groundTruthTrackSegmentList, candidateTrackSegmentList , useImprovedMethod );
		
		//double maxDist = 5;
		double maxDist = minimumDistanceBetweenDetections;
		DistanceTypes distType = DistanceTypes.DISTANCE_EUCLIDIAN;
				
		int numRefTracks = performanceAnalyser.getNumRefTracks();
		int numCandidateTracks = performanceAnalyser.getNumCandidateTracks();
		int numRefDetections = performanceAnalyser.getNumRefDetections();
		int numCandidateDetections = performanceAnalyser.getNumCandidateDetections();
		double pairsDistance = performanceAnalyser.getPairedTracksDistance(distType, maxDist);
		double pairsNormalizedDistance = performanceAnalyser.getPairedTracksNormalizedDistance(distType, maxDist);
		double pairsFullDistance = performanceAnalyser.getFullTrackingScore(distType, maxDist);
		int numSpuriousTracks = performanceAnalyser.getNumSpuriousTracks();
		int numMissedTracks = performanceAnalyser.getNumMissedTracks();
		int numCorrectTracks = performanceAnalyser.getNumPairedTracks();
		int numRecoveredDetections = performanceAnalyser.getNumPairedDetections(maxDist);
		int numMissedDetections = performanceAnalyser.getNumMissedDetections(maxDist);
		int numWrongDetections = performanceAnalyser.getNumWrongDetections(maxDist);
		
		double distance[] = performanceAnalyser.getDistanceDetectionData(maxDist);
		double rmseDetection = distance[0];
		double minDistanceDetection = distance[1];
		double maxDistanceDetection = distance[2];
		double stdDistanceDetection = distance[3];
		
		double detectionsSimilarity = (double)numRecoveredDetections/((double)numRecoveredDetections + (double)numMissedDetections + (double)numWrongDetections);
		double tracksSimilarity = (double)numCorrectTracks/((double)numCorrectTracks + (double) numMissedTracks + (double) numSpuriousTracks);

		{
			FileWriter outFile;
			try {
				outFile = new FileWriter( candidateDocument.getSourceFile().getAbsoluteFile()+".score.txt" );
				PrintWriter out = new PrintWriter(outFile);

				out.println("original file:\t" + candidateDocument.getSourceFile().getAbsoluteFile() );
				out.println("scenario:\t" + candidateDocument.scenario );
				out.println("density:\t" + candidateDocument.density );
				out.println("snr:\t" + candidateDocument.snr );
				out.println(pairsDistance+"\t : pairing distance");
				out.println(pairsNormalizedDistance+"\t : normalized pairing score (alpha)");
				out.println(pairsFullDistance+"\t : full normalized score (beta)");
				out.println(numRefTracks+"\t : number of reference tracks");
				out.println(numCandidateTracks+"\t : number of candidate tracks");
				out.println(tracksSimilarity+"\t : Similarity between tracks (Jaccard)");
				out.println(numCorrectTracks+"\t : number of paired tracks (out of "+numRefTracks+")");
				out.println(numMissedTracks+"\t : number of missed tracks (out of "+numRefTracks+")");
				out.println(numSpuriousTracks+"\t : number of spurious tracks)");
				out.println(numRefDetections+"\t : number of reference detections");
				out.println(numCandidateDetections+"\t : number of candidate detections");
				out.println(detectionsSimilarity+"\t : Similarity between detections (Jaccard)");
				out.println(numRecoveredDetections+"\t : number of paired detections (out of "+numRefDetections+")");
				out.println(numMissedDetections+"\t : number of missed detections (out of "+numRefDetections+")");
				out.println(numWrongDetections+"\t : number of spurious detections");

				out.println(rmseDetection+"\t : RMSE Detection");
				out.println(minDistanceDetection+"\t : min distance Detection");
				out.println(maxDistanceDetection+"\t : max distance Detection");
				out.println(stdDistanceDetection+"\t : Std distance Detection");


				out.close();

			} catch (IOException e) {
				System.err.println("failed to save score !");
				e.printStackTrace();
			}
		}
			
		// output 
		{
			FileWriter outFile;
			try {
				outFile = new FileWriter( candidateDocument.getSourceFile().getAbsoluteFile()+".extrascore.txt" );
				PrintWriter out = new PrintWriter(outFile);

				out.println("original file:\t" + candidateDocument.getSourceFile().getAbsoluteFile() );
				out.println("scenario:\t" + candidateDocument.scenario );
				out.println("density:\t" + candidateDocument.density );
				out.println("snr:\t" + candidateDocument.snr );
								
				// instantaneous velocity for reference tracks
		        ArrayList<Double> jumpList = performanceAnalyser.getReferenceTracksJumpLengthList();
		        out.println("Ref jump list");
		        for (Double d:jumpList)
		            out.print(d+"\t");
		        out.println("");
		        // instantaneous velocity for candidate tracks
		        jumpList = performanceAnalyser.getCandidateTracksJumpLengthList();
		        out.println("Candidate jump list");
		        for (Double d:jumpList)
		            out.print(d+"\t");
		        out.println("");
		        // distance between detections
		        ArrayList<Double> distList = performanceAnalyser.getAllPairsDetectionEuclidianDistances();
		        out.println("Detection distance list");
		        for (Double d:distList)
		            out.print(d+"\t");
		        out.println("");
		       
		       
		        // reference tracks MSDS       
		        //double[] msds = performanceAnalyser.getMSDs(referenceTracks);
		        //double[] msds = performanceAnalyser.getMSDs( groundTruthTrackSegmentList );	        
		        double[] msds = performanceAnalyser.getReferenceTracksMSDs();
		        out.println("Reference MSDs");
		        for (Double d:msds)
		            out.print(d+"\t");
		        out.println("");
		       
		        // candidate tracks MSDS
		        msds = performanceAnalyser.getCandidateTracksMSDs();
		        out.println("Candidate MSDs");
		        for (Double d:msds)
		            out.print(d+"\t");
		        out.println("");

				out.close();

			} catch (IOException e) {
				System.err.println("failed to create extra score information !");
				e.printStackTrace();
			}
		}
		
			
		
			
			
			

		
				
	}

	private ChallengeDocument findGroundTruth(ChallengeDocument candidateDocument, ArrayList<ChallengeDocument> groundTruthDocumentList) {

		for ( ChallengeDocument groundTruthDocument : groundTruthDocumentList )
		{
			if ( groundTruthDocument.asSameParametersAs( candidateDocument ) )
			{
				return groundTruthDocument;
			}
		}
		
		System.err.println("No matching ground truth found for " + candidateDocument );

		return null;
		
	}

	@Override
	public void clean() {
		

	}
	
	
	
	
	
}
