package plugins.tprovoost.painting;

import icy.canvas.IcyCanvas;
import icy.canvas.IcyCanvas2D;
import icy.image.IcyBufferedImage;
import icy.image.IcyBufferedImageUtil;
import icy.painter.Overlay;
import icy.sequence.Sequence;
import icy.type.point.Point5D;
import icy.util.EventUtil;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;
import java.util.LinkedList;

import plugins.tprovoost.painting.PaintingTools.PaintingTool;
import plugins.tprovoost.painting.shapes.ArrowShape;
import plugins.tprovoost.painting.shapes.Cloud;
import plugins.tprovoost.painting.shapes.CurveShape;
import plugins.tprovoost.painting.shapes.LineShape;
import plugins.tprovoost.painting.shapes.Oval;
import plugins.tprovoost.painting.shapes.PaintingShape;
import plugins.tprovoost.painting.shapes.RectangleShape;

public class PaintingOverlay extends Overlay
{
    private final LinkedList<PaintingShape> shapes;
    private final LinkedList<PaintingShape> undo;
    private Point startPoint;
    private Point currentPoint;
    private Point mouseMovePoint;
    private PaintingShape currentShape;

    public PaintingOverlay(String name)
    {
        super(name);

        shapes = new LinkedList<PaintingShape>();
        undo = new LinkedList<PaintingShape>();
        startPoint = null;
        currentPoint = null;
        mouseMovePoint = null;
        currentShape = null;
    }

    @Override
    public void mousePressed(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
    {
        if (e.isConsumed())
            return;
        if (imagePoint == null)
            return;

        final PaintingTools paintingTools = PaintingTools.instance;
        if ((paintingTools == null) || !paintingTools.isVisible())
            return;

        final PaintingTool currentTool = paintingTools.getTool();
        if (currentTool == PaintingTool.NONE)
            return;

        int x = (int) imagePoint.x;
        int y = (int) imagePoint.y;

        if (currentShape != null)
        {
            if ((currentShape instanceof CurveShape) && (currentTool != PaintingTool.CURVE))
            {
                // if current shape is not a curve anymore
                // the current shape is not editable and becomes null.
                currentShape.setEditable(false);
                currentShape = null;
            }
            else
            {
                e.consume();
                return; // do nothing
            }
        }
        startPoint = new Point(x, y);
        if (currentTool == PaintingTool.PEN)
        {
            shapes.add(new Cloud(paintingTools, startPoint));
            undo.clear();
            currentShape = shapes.getLast();
            painterChanged();
        }

        e.consume();
    }

    @Override
    public void mouseMove(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
    {
        if (imagePoint == null)
            return;

        final PaintingTools paintingTools = PaintingTools.instance;
        if ((paintingTools == null) || !paintingTools.isVisible())
            return;

        mouseMovePoint = new Point((int) imagePoint.x, (int) imagePoint.y);
        painterChanged();
    }

    @Override
    public void mouseDrag(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
    {
        if (e.isConsumed())
            return;
        if (imagePoint == null)
            return;

        final PaintingTools paintingTools = PaintingTools.instance;
        if ((paintingTools == null) || !paintingTools.isVisible())
            return;

        if (startPoint == null)
        {
            mousePressed(e, imagePoint, canvas);
            return;
        }

        int x = (int) imagePoint.x;
        int y = (int) imagePoint.y;

        currentPoint = new Point(x, y);
        if (currentShape == null)
        {
            PaintingTool currentTool = paintingTools.getTool();
            shapes.add(createTool(paintingTools, currentTool));
            undo.clear();
            currentShape = shapes.getLast();
        }
        else
        {
            currentShape.update(currentPoint);
        }

        painterChanged();
    }

    @Override
    public void mouseReleased(MouseEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
    {
        final PaintingTools paintingTools = PaintingTools.instance;
        if (paintingTools == null || !paintingTools.isVisible())
            return;

        if (currentShape != null)
        {
            if (currentShape instanceof CurveShape && !((CurveShape) currentShape).isPointBLocked())
            {
                ((CurveShape) currentShape).lockPointB();
            }
            else
            {
                currentShape.setEditable(false);
                currentShape = null;
                startPoint = null;
                currentPoint = null;

                if (!paintingTools.getDrawOnPainter())
                {
                    IcyBufferedImage img;
                    final int channelAffected = paintingTools.getChannelAffected();

                    if (channelAffected != -1)
                        img = IcyBufferedImageUtil.extractChannel(canvas.getCurrentImage(), channelAffected);
                    else
                        img = canvas.getCurrentImage();

                    final BufferedImage imgBuff = IcyBufferedImageUtil.toBufferedImage(img,
                            new BufferedImage(img.getWidth(), img.getHeight(), BufferedImage.TYPE_BYTE_GRAY));
                    final Graphics2D g2d;

                    if (img.getSizeC() == 1)
                        g2d = imgBuff.createGraphics();
                    else
                        g2d = img.createGraphics();

                    final PaintingShape shape = shapes.pollLast();

                    if (shape != null)
                    {
                        if (shape instanceof Cloud)
                            g2d.setStroke(new BasicStroke(paintingTools.getThickness(), BasicStroke.CAP_ROUND,
                                    BasicStroke.JOIN_ROUND));
                        else
                            g2d.setStroke(new BasicStroke(paintingTools.getThickness()));
                        shape.drawShape(g2d);
                    }

                    g2d.dispose();

                    if (channelAffected != -1)
                    {
                        img = IcyBufferedImage.createFrom(imgBuff);
                        canvas.getCurrentImage().setDataXY(channelAffected, img.getDataXY(0));
                    }

                    canvas.getSequence().dataChanged();
                }
            }
        }

        painterChanged();
    }

    @Override
    public void keyPressed(KeyEvent e, Point5D.Double imagePoint, IcyCanvas canvas)
    {
        if (EventUtil.isControlDown(e))
        {
            if (e.getKeyCode() == KeyEvent.VK_Z)
            {
                if (EventUtil.isShiftDown(e))
                {
                    PaintingShape s = undo.pollFirst();
                    if (s != null)
                    {
                        shapes.add(s);
                        painterChanged();
                    }
                }
                else
                {
                    PaintingShape s = shapes.pollLast();
                    if (s != null)
                    {
                        undo.addFirst(s);
                        painterChanged();
                    }
                }
            }
        }
        else
        {
            if (e.getKeyCode() == KeyEvent.VK_ESCAPE)
            {
                startPoint = null;
                currentPoint = null;
                if (currentShape != null)
                {
                    currentShape.setEditable(false);
                    currentShape = null;
                }
            }
        }
    }

    @Override
    public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas)
    {
        if ((g == null) || !(canvas instanceof IcyCanvas2D))
            return;

        final Graphics2D g2 = (Graphics2D) g.create();

        for (PaintingShape shape : shapes)
        {
            g2.setStroke(shape.getStroke());
            g2.setColor(shape.getShapeColor());
            shape.drawShape(g2);
        }

        final PaintingTools paintingTools = PaintingTools.instance;
        if ((paintingTools != null) && paintingTools.isVisible())
        {
            final PaintingTool currentTool = paintingTools.getTool();

            if ((mouseMovePoint != null) && (currentShape == null) && (currentTool != PaintingTool.NONE))
            {
                // set stroke
                g2.setStroke(new BasicStroke(paintingTools.getThickness(), BasicStroke.CAP_ROUND,
                        BasicStroke.JOIN_ROUND));

                // set color
                int fgColor = paintingTools.getForegroundColor().getRGB();
                fgColor &= 0xFFFFFF;
                fgColor |= 80 << 24;
                g2.setColor(new Color(fgColor, true));
                g2.drawLine(mouseMovePoint.x, mouseMovePoint.y, mouseMovePoint.x, mouseMovePoint.y);
            }
        }

        g2.dispose();
    }

    private PaintingShape createTool(PaintingTools tools, PaintingTool currentTool)
    {
        PaintingShape toReturn = null;

        switch (currentTool)
        {
            case ARROW:
                toReturn = new ArrowShape(tools, startPoint.x, startPoint.y, currentPoint.x, currentPoint.y);
                break;
            case LINE:
                toReturn = new LineShape(tools, startPoint.x, startPoint.y, currentPoint.x, currentPoint.y);
                break;
            case PLAIN_OVAL:
                toReturn = new Oval(tools, startPoint.x, startPoint.y, currentPoint.x, currentPoint.y, true);
                break;
            case OVAL:
            {
                toReturn = new Oval(tools, startPoint.x, startPoint.y, currentPoint.x, currentPoint.y, false);
                break;
            }
            case PLAIN_RECTANGLE:
                toReturn = new RectangleShape(tools, startPoint.x, startPoint.y, currentPoint.x, currentPoint.y, true);
                break;
            case RECTANGLE:
            {
                toReturn = new RectangleShape(tools, startPoint.x, startPoint.y, currentPoint.x, currentPoint.y, false);
                break;
            }
            case CURVE:
                toReturn = new CurveShape(tools, startPoint.x, startPoint.y, currentPoint.x, currentPoint.y);
                break;
            default:
                return null;
        }

        return toReturn;
    }
}
