package plugins.fab.mycosislungquantifier;

import icy.canvas.IcyCanvas;
import icy.file.FileUtil;
import icy.file.xls.XlsManager;
import icy.gui.frame.progress.AnnounceFrame;
import icy.gui.frame.progress.FailedAnnounceFrame;
import icy.painter.Overlay;
import icy.roi.ROI;
import icy.roi.ROI2D;
import icy.sequence.Sequence;
import ij.gui.Roi;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.geom.Line2D;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.vecmath.Point3d;

import com.sun.media.CircularBuffer;

import plugins.adufour.connectedcomponents.ConnectedComponent;
import plugins.adufour.connectedcomponents.ConnectedComponents;
import plugins.adufour.connectedcomponents.ConnectedComponents.ExtractionType;
import plugins.adufour.ezplug.EzGroup;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.adufour.thresholder.Thresholder;
import plugins.kernel.roi.roi2d.ROI2DArea;
import plugins.kernel.roi.roi5d.ROI5DArea;

public class MycosisLungQuantifier extends EzPlug {

	
	
	EzVarSequence inputSequenceVar = new EzVarSequence("Input Sequence");
	EzVarDouble thresholdComponent = new EzVarDouble("Threshold", 30, 0, Double.MAX_VALUE, 0.1 );
	EzVarInteger minSizeComponent = new EzVarInteger("Min size", 0, 0, Integer.MAX_VALUE, 1 );
	EzVarInteger maxSizeComponent = new EzVarInteger("Max size", 500, 0, Integer.MAX_VALUE, 1 );
	
	EzVarInteger maxDistanceInPixel = new EzVarInteger("Max distance in px", 320, 0, Integer.MAX_VALUE, 1 );
	EzVarInteger epitheliumWidthInPixel = new EzVarInteger("Epithelium width in pix", 16, 0, Integer.MAX_VALUE, 1 );

	EzVarBoolean thresholdCompactnessTest = new EzVarBoolean( "Enable thresholds", false );
	EzVarDouble thresholdCompactnessInput = new EzVarDouble("Threshold Min Compactness", 0.4, 0, 2, 0.01 );

	EzVarDouble surfaceMinInput = new EzVarDouble("Surface Min (px)", 5000, 0, Integer.MAX_VALUE, 1 );
	EzVarDouble surfaceMaxInput = new EzVarDouble("Surface Max (px)", 100000, 0, Integer.MAX_VALUE, 1 );
	EzVarDouble perimeterMinInput = new EzVarDouble("Perimeter Min (px)", 300, 0, Integer.MAX_VALUE, 1 );
	EzVarDouble perimeterMaxInput = new EzVarDouble("Perimeter Max (px)", 10000, 0, Integer.MAX_VALUE, 1 );

	
	OverlayResult overlayResult = null;
	
	@Override
	public void clean() {
		
	}

	@Override
	protected void execute() {
		
		System.gc();
		System.gc();
		System.gc();
		
		if (inputSequenceVar.getValue().getROI2Ds().size() == 0 )
		{			
			new FailedAnnounceFrame("You need at least 1 2D ROI to use this plugin", 5 );
			return;
		}
		
		// perform the threshold
		double thresholdArray[] = { thresholdComponent.getValue() };
		Sequence thresholdedSequence = Thresholder.threshold( inputSequenceVar.getValue() , 2, thresholdArray, false );
		
		// apply connected components
		
		Sequence labeledSequence = new Sequence(); 
		
		Map<Integer, List<ConnectedComponent>> ccAll = ConnectedComponents.extractConnectedComponents(
				thresholdedSequence, 
				0, 
				ExtractionType.VALUE, 
				false, 
				false, 
				false, 
				minSizeComponent.getValue(), 
				maxSizeComponent.getValue(),
				labeledSequence);
		
//		addSequence( labeledSequence );
	
		// grab the RoiArea available in sequence
		
		ArrayList<ROI > roiList = new ArrayList<ROI >();
		HashMap< ROI , Point[] > roi2edge = new HashMap<ROI , Point[]>();
		
 
		 // build the input roi list
		
		for ( ROI roi : inputSequenceVar.getValue().getROIs() )
		{
//			ROI5DArea r;
//			r.getBooleanMask2D(z, t, c, inclusive)
			
			// If the ROI is black, it's not considered.
			if ( roi.getColor().getGreen() == 0 && roi.getColor().getRed() == 0 && roi.getColor().getBlue() == 0 )
			{
				continue;
			}
			
			if ( roi instanceof ROI2DArea )
			{
				ROI2DArea roi2DArea = (ROI2DArea) roi;
				
				roiList.add( roi2DArea );				
				roi2edge.put( roi2DArea , roi2DArea.getBooleanMask(false).getContourPoints() );
				
			}
			
			
			if ( roi instanceof ROI5DArea )
			{
				ROI5DArea roi5DArea = (ROI5DArea) roi;
				
				roiList.add( roi5DArea );				
				roi2edge.put( roi5DArea , roi5DArea.getBooleanMask2D( 0 , 0 , 0, false).getContourPoints() );
				
			}
		}
		
		// set ROi names
		
		int indexROI = 0;
		for ( ROI  roi2DArea : roiList )
		{
			indexROI++;
//			if( !(roi2DArea.getName().substring( 0 , 6 ).equals( "Area #" ) ) ) // check if the area was already renamed before.
			{
//				System.out.println("renaming ROI " + indexROI );
				roi2DArea.setName( "Area #" + indexROI );
			}
//			else
//			{
//				System.out.println("not renaming ROI " + indexROI );
//			}
		}
		
		// Remove detection in ROI if they are black.
		
		{
			List<ConnectedComponent> detectionList = ccAll.get( 0 );
			
			for ( ROI2D roi : inputSequenceVar.getValue().getROI2Ds() )
			{
				if ( roi.getColor().getGreen() == 0 && roi.getColor().getRed() == 0 && roi.getColor().getBlue() == 0 )
				{
					List<ConnectedComponent> detectionListCopy = new ArrayList<ConnectedComponent>( detectionList );
					
					for ( ConnectedComponent detection : detectionListCopy )
					{
						Point3d point = detection.getMassCenter();
						
						if ( roi.contains( point.x , point.y ) ) 
						{
							detectionList.remove( detection );
						}
					}

				}
			}
		}
		
		int maxDistanceInPix = maxDistanceInPixel.getValue();
				
		System.out.println( roiList.size() + " ROI Area 2D found.");
		
		// sort the Connected Component with the closest ROIArea and store results
		
		HashMap< ConnectedComponent , BestResult > cc2Roi = new HashMap<ConnectedComponent, BestResult>();
		
		List<ConnectedComponent> ccList = ccAll.get( 0 ); // get time point at 0
		for ( ConnectedComponent cc: ccList )
		{
			Point3d massCenter = cc.getMassCenter();			
			double bestDistance = Double.MAX_VALUE;
			ROI bestROI = null;
			Point bestPoint = null;
			
			for ( ROI  roi2DArea : roiList )
			{
				Point[] edgePoints = roi2edge.get( roi2DArea );
				for ( int i = 0 ; i < edgePoints.length ; i++ )
				{
					Point p = edgePoints[i];
					double distance = p.distance( massCenter.x, massCenter.y );
					if ( distance < bestDistance )
					{
						bestROI = roi2DArea;
						bestDistance = distance;
						bestPoint = p;
					}
				}
			}
			boolean isInsideROI = bestROI.contains( cc.getMassCenter().x , cc.getMassCenter().y , 0 , 0 , 0 );
			if ( isInsideROI ) bestDistance = -bestDistance;
			
			if ( bestDistance <=  maxDistanceInPix ) // only keep if closer than maxDistance.
			{
				cc2Roi.put( cc , new BestResult( bestROI, bestDistance , bestPoint , isInsideROI ) );
			}
		}
		
		// output
		
		inputSequenceVar.getValue().removePainter( overlayResult );
		overlayResult = new OverlayResult("Mycosis results" , cc2Roi );
		inputSequenceVar.getValue().addPainter( overlayResult );

		// output xls data
		
		try {
			
			File xlsFile = new File( FileUtil.setExtension( inputSequenceVar.getValue().getFilename(), ".xls") );
			XlsManager xls = new XlsManager( xlsFile );
			xls.createNewPage("res " + inputSequenceVar.getValue().getName() );

			xls.setLabel( 0, 0, "roi id" );
			xls.setLabel( 1, 0, "roi name" );
			xls.setLabel( 2, 0, "myc distance" );
			xls.setLabel( 3, 0, "is inside ROI" );
			xls.setLabel( 4, 0, "myc size" );
			int row = 1;

			for ( ConnectedComponent cc : cc2Roi.keySet() )
			{
				BestResult bestResult = cc2Roi.get( cc );
				cc.getMassCenter();

				xls.setNumber( 0, row, bestResult.roi.getId() );	
				xls.setLabel( 1, row, bestResult.roi.getName() );
				xls.setNumber( 2, row, bestResult.distance );
				xls.setLabel( 3, row, ""+bestResult.isInsideROI );
				xls.setNumber( 4, row, cc.getSize() );

				row++;
			}
			
			// save ROI parameters
			
			xls.createNewPage("Results by roi");
			row = 1;
			xls.setLabel( 0, 0, "ROI id" );
			xls.setLabel( 1, 0, "ROI name" );
			xls.setLabel( 2, 0, "ROI surface in px" );
			xls.setLabel( 3, 0, "nb of value<=0" );
			xls.setLabel( 4, 0, "nb of value>0 and value<epithelium width" );
			xls.setLabel( 5, 0, "nb of value>=epithelium width" );
			xls.setLabel( 6, 0, "ROI perimeter in px" );
			xls.setLabel( 7, 0, "compactness" );

			//return 4 * Math.PI * ( Area - perimater)  / Math.pow(nbPixelPerimeter, 2); // c est la compactness,
			
			
			int epitheliumWith = epitheliumWidthInPixel.getValue() ;	
			
			double totalSurface = 0;
			double totalPerimeter = 0;
			double totalNbValueBelowZero = 0;
			double totalNbValueInEpithelium = 0;
			double totalNbValueOverEpithelium = 0;
			
			ArrayList<String> listError = new ArrayList<String>();
			
			for ( ROI roi2DArea : roiList )
			{
	
				double surface=roi2DArea.getNumberOfPoints();
				double perimeter = roi2DArea.getNumberOfContourPoints();
				double compactness = 4 * Math.PI * ( surface - perimeter ) / Math.pow( perimeter , 2); // circularity computation
				double thresholdCompactness = thresholdCompactnessInput.getValue();
				
				if ( thresholdCompactnessTest.getValue() ) // check if threshold on compactness is enabled.
				{
					boolean shouldNotPass = false;
					if ( compactness < thresholdCompactness ) // if compactness is not ok, skip.
					{
						String error = roi2DArea.getName() + " compactness test not ok.";
						listError.add( error );
						shouldNotPass = true;
					}

					if ( surface < surfaceMinInput.getValue() )
					{
						String error = roi2DArea.getName() + " surface min test not ok.";
						listError.add( error );
						shouldNotPass = true;
					}
					if ( surface > surfaceMaxInput.getValue() )
					{
						String error = roi2DArea.getName() + " surface max test not ok.";
						listError.add( error );
						shouldNotPass = true;
					}

					if ( perimeter < perimeterMinInput.getValue() )
					{
						String error = roi2DArea.getName() + " perimeter min test not ok.";
						listError.add( error );
						shouldNotPass = true;
					}

					if ( perimeter > perimeterMaxInput.getValue() )
					{
						String error = roi2DArea.getName() + " perimeter max test not ok.";
						listError.add( error );
						shouldNotPass = true;
					}

					if ( shouldNotPass )
					{
						roi2DArea.setColor( Color.red );
						continue;
					}
					
				}
				
				roi2DArea.setColor( Color.green );
				
				xls.setLabel( 0 , row, ""+roi2DArea.getId() );
				xls.setLabel( 1 , row, ""+roi2DArea.getName() );
				//double surface=roi2DArea.getNumberOfPoints();
				totalSurface+=surface;
				
				//double perimeter = roi2DArea.getNumberOfContourPoints();
				totalPerimeter+=perimeter;
				xls.setNumber( 2 , row, surface );
				xls.setNumber( 6 , row, perimeter );				
//				double compactness = 4 * Math.PI * ( surface - perimeter ) / Math.pow( perimeter , 2); // circularity computation
				xls.setNumber( 7 , row, compactness );
				
				int nbValueBelowZero = 0;
				int nbValueInEpithelium = 0;
				int nbValueOverEpithelium = 0;
				
				for ( ConnectedComponent cc : cc2Roi.keySet() )
				{
					BestResult bestResult = cc2Roi.get( cc );
					if ( bestResult.roi != roi2DArea ) continue;
					
					if ( bestResult.distance <= 0 ) nbValueBelowZero++;				
					if ( bestResult.distance > 0 && bestResult.distance < epitheliumWith ) nbValueInEpithelium++;
					if ( bestResult.distance >= epitheliumWith ) nbValueOverEpithelium++;
				}
				
				xls.setNumber( 3 , row, nbValueBelowZero );
				xls.setNumber( 4 , row, nbValueInEpithelium );
				xls.setNumber( 5 , row, nbValueOverEpithelium );
				
				totalNbValueBelowZero+=nbValueBelowZero;
				totalNbValueInEpithelium+=nbValueInEpithelium;
				totalNbValueOverEpithelium+=nbValueOverEpithelium;
				
				row++;
			}
			
			xls.setLabel( 0 , row, "Total:" );
			
			xls.setNumber( 2 , row, totalSurface );
			xls.setNumber( 3 , row, totalNbValueBelowZero );
			xls.setNumber( 4 , row, totalNbValueInEpithelium );
			xls.setNumber( 5 , row, totalNbValueOverEpithelium );
			xls.setNumber( 6 , row, totalPerimeter );
			
			// save parameters
			
			xls.createNewPage("Parameters");
			
			xls.setLabel( 0 , 0 , "threshold" );
			xls.setNumber( 1 , 0 , thresholdComponent.getValue() );

			xls.setLabel( 0 , 1 , "min size" );
			xls.setNumber( 1 , 1 , minSizeComponent.getValue() );
			
			xls.setLabel( 0 , 2 , "max size" );
			xls.setNumber( 1 , 2 , maxSizeComponent.getValue() );
			
			xls.setLabel( 0 , 3 , "max distance" );
			xls.setNumber( 1 , 3 , maxDistanceInPixel.getValue() );
			
			xls.setLabel( 0 , 4 , "epithelium Width" );
			xls.setNumber( 1 , 4 , epitheliumWidthInPixel.getValue() );				

			if ( thresholdCompactnessTest.getValue() )
			{
				xls.setLabel( 0 , 5 , "min compactness" );
				xls.setNumber( 1 , 5 , thresholdCompactnessInput.getValue() );				
				
				xls.setLabel( 0 , 6 , "min surface" );
				xls.setNumber( 1 , 6 , surfaceMinInput.getValue() );				

				xls.setLabel( 0 , 7 , "max surface" );
				xls.setNumber( 1 , 7 , surfaceMaxInput.getValue() );				
				
				xls.setLabel( 0 , 8 , "min perimeter" );
				xls.setNumber( 1 , 8 , perimeterMinInput.getValue() );				
				
				xls.setLabel( 0 , 9 , "max perimeter" );
				xls.setNumber( 1 , 9 , perimeterMaxInput.getValue() );				
				
			}
			
			// save roi discarded
			
			xls.createNewPage("ROI filtered");
			row = 0;
			for ( String error : listError )
			{
				xls.setLabel( 0 , row , error );
				row++;	
			}
			
			xls.SaveAndClose();

			new AnnounceFrame( "Results saved in "+xlsFile.getAbsolutePath() , 5 );			

		} catch (IOException e) {
			e.printStackTrace();
		}

	}

	@Override
	protected void initialize() {

		addEzComponent( inputSequenceVar );
		addEzComponent( thresholdComponent );
		addEzComponent( minSizeComponent );
		addEzComponent( maxSizeComponent );
		
		addEzComponent( maxDistanceInPixel );
		addEzComponent( epitheliumWidthInPixel );
		
		EzGroup group = new EzGroup("Thresholds", thresholdCompactnessTest, thresholdCompactnessInput , 
				surfaceMinInput,
				surfaceMaxInput,
				perimeterMinInput,
				perimeterMaxInput
				);
		
		addEzComponent( group );
		
		
//		EzVarBoolean thresholdCompactnessTest = new EzVarBoolean( "Enable thresholds", false );
//		EzVarDouble thresholdCompactnessInput = new EzVarDouble("Threshold Min Compactness", 0.4, 0, 2, 0.01 );
//
//		EzVarDouble surfaceMinInput = new EzVarDouble("Surface Min (px)", 5000, 0, Integer.MAX_VALUE, 1 );
//		EzVarDouble surfaceMaxInput = new EzVarDouble("Surface Max (px)", 100000, 0, Integer.MAX_VALUE, 1 );
//		EzVarDouble perimeterMinInput = new EzVarDouble("Perimeter Min (px)", 300, 0, Integer.MAX_VALUE, 1 );
//		EzVarDouble perimeterMaxInput = new EzVarDouble("Perimeter Max (px)", 10000, 0, Integer.MAX_VALUE, 1 );

		
	}

	class BestResult
	{
		ROI roi;
		double distance;
		Point bestPoint;
		boolean isInsideROI;
		
		public BestResult( ROI roi , double distance , Point bestPoint , boolean isInsideROI ) {
			this.roi = roi;
			this.distance = distance;
			this.bestPoint = bestPoint;
			this.isInsideROI = isInsideROI;			
		}
	}
	
	class OverlayResult extends Overlay
	{
		HashMap<ConnectedComponent, BestResult> cc2Roi;

		public OverlayResult(String name, HashMap<ConnectedComponent, BestResult> cc2Roi) {
			super(name);
			this.cc2Roi = cc2Roi;
		}
		
		@Override
		public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas) {
			
			g.setStroke( new BasicStroke( 2 ) ); 
			
			for ( ConnectedComponent cc : cc2Roi.keySet() )
			{
				BestResult bestResult = cc2Roi.get( cc );
				cc.getMassCenter();
			
				Line2D line = new Line2D.Double(
						cc.getMassCenter().x ,
						cc.getMassCenter().y ,
						bestResult.bestPoint.x,
						bestResult.bestPoint.y
						);

				if ( bestResult.isInsideROI )
				{
					g.setColor( Color.red );
				}else
				{
					g.setColor( Color.yellow );
				}
				
				g.draw( line );
				
				
			}
			
			
			
		}
		
	}

	

	
}
