package plugins.fab.internetconnectivitymonitor;

import icy.canvas.Canvas2D;
import icy.canvas.IcyCanvas;
import icy.file.Loader;
import icy.file.Saver;
import icy.gui.viewer.Viewer;
import icy.image.IcyBufferedImage;
import icy.main.Icy;
import icy.painter.Overlay;
import icy.sequence.Sequence;
import icy.sequence.SequenceListener;
import icy.type.DataType;
import icy.util.GraphicsUtil;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import loci.formats.FormatException;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Wrap a sequence that is used to report a connectivity state over time and
 * to load/save it from a folder
 */
public class ConnectivityMap
{
	private String             _folder     ;
	private Pattern            _filePattern;
	private int                _interval   ;
	private Sequence           _sequence   ;
	private int                _width      ;
	private int                _height     ;
	private Map<Integer, Info> _infos      ;
	private int                _currentT   ;
	
	/**
	 * Constructor
	 */
	public ConnectivityMap(String folder, String name, int interval)
	{
		// Check the arguments
		if(interval<=0 || (60 % interval)!=0) {
			throw new IllegalArgumentException("Interval is expected to be a positive divider of 60.");
		}
		
		// Initialization
		_folder   = folder  ;
		_interval = interval;
		_sequence = new Sequence();
		_width    = 60*15/_interval;
		_height   = 24*4;
		_infos    = new HashMap<Integer, Info>();
		_currentT = -1;
		
		// Initialize the sequence
		_sequence.setName(name);
		new ConnectivityMapOverlay();
	}
	
	/**
	 * Return the interval
	 */
	public int getInterval()
	{
		return _interval;
	}
	
	/**
	 * Add a listener to the underlying sequence
	 */
	public void addListener(SequenceListener l)
	{
		_sequence.addListener(l);
	}
	
	/**
	 * Remove a listener from the underlying sequence
	 */
	public void removeListener(SequenceListener l)
	{
		_sequence.removeListener(l);
	}
	
	/**
	 * Load the existing results for the given day
	 */
	public void load(int year, int month, int day)
	{
		File file = getFile(year, month, day);
		IcyBufferedImage image = null;
		
		// Load the file if it exists
		if(file.exists()) {
			try {
				image = Loader.loadImage(file);
			}
			catch (FormatException e) {}
			catch (IOException     e) {}
			
			// Check the dimensions of the image
			if(image.getSizeC()!=3 || image.getDataType_()!=DataType.UBYTE) {
				throw new RuntimeException(
					"The connectivity map is expected to be uint8-valued and to have 3 channels."
				);
			}
			if(image.getSizeX()!=_width || image.getSizeY()!=_height) {
				throw new RuntimeException(
					"Wrong dimensions for the given connectivity map."
				);
			}
		}
		
		
		// Create the image if the file does not exist or cannot be loaded
		if(image==null) {
			image = createNewImage();
		}
		
		// Append it to the sequence
		_currentT = _sequence.getSizeT();
		_sequence.addImage(_currentT, image);
		_infos.put(_currentT, new Info(year, month, day));
		
		// Hack: addSequence is buggy with empty sequences 
		if(_currentT==0) {
			Icy.getMainInterface().addSequence(_sequence);
		}
		
		// Move to the last frame
		Viewer viewer = _sequence.getFirstViewer();
		if(viewer!=null) {
			viewer.getCanvas().setPositionT(_currentT);
		}
	}
	
	/**
	 * Save the last image
	 */
	public void save()
	{
		Info info = _infos.get(_currentT);
		if(info==null) {
			return;
		}
		File file = getFile(info.year, info.month, info.day);
		IcyBufferedImage image = _sequence.getImage(_currentT, 0);
		try {
			Saver.saveImage(image, file, true);
		}
		catch(FormatException e) {
			e.printStackTrace();
		}
		catch(IOException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Update the connected state for the given time point
	 */
	public void updateMap(boolean connected, int year, int month, int day, int hr, int min, int sec)
	{
		// Create a new image if necessary
		Info info = _infos.get(_currentT);
		if(info==null || info.year!=year || info.month!=month || info.day!=day) {
			save();
			load(year, month, day);
		}
		
		// Pixel to set
		int x = ((min % 15) * 60 + sec) / _interval;
		int y = hr*4 + min/15;
		_sequence.getImage(_currentT, 0).setData(x, y, connected ? 1 : 0, 255);
		
		// Save the image every 15 minutes
		if(x==_width-1) {
			save();
		}
	}
	
	/**
	 * Return the first day for which an archived file exists
	 */
	public Info getFirstArchivedDay()
	{
		return getFirstOrLastArchivedDay(true);
	}
	
	/**
	 * Return the last day for which an archived file exists
	 */
	public Info getLastArchivedDay()
	{
		return getFirstOrLastArchivedDay(false);
	}
	
	/**
	 * Auxiliary function for getFirst/LastArchivedDay
	 */
	private Info getFirstOrLastArchivedDay(boolean first)
	{
		String[] rawFiles = (new File(_folder)).list();
		if(rawFiles==null) {
			rawFiles = new String[0];
		}
		List<String> files = Arrays.asList(rawFiles);
		Collections.sort(files);
		if(!first) {
			Collections.reverse(files);
		}
		for(String file : files) {
			Info retVal = parseInfoFromFileName(file);
			if(retVal!=null) {
				return retVal;
			}
		}
		return null;
	}
	
	/**
	 * Parse a file name and return the corresponding time point if it matches
	 * the template used to archive connectivity maps
	 */
	private Info parseInfoFromFileName(String fileName)
	{
		if(_filePattern==null) {
			_filePattern = Pattern.compile(
				"log-([0-9]{4})-([0-9]{2})-([0-9]{2})\\.png"
			);
		}
		Matcher res = _filePattern.matcher(fileName);
		if(!res.matches() || res.groupCount()!=3) {
			return null;
		}
		try {
			int y = Integer.decode(res.group(1));
			int m = Integer.decode(res.group(2));
			int d = Integer.decode(res.group(3));
			return new Info(y, m, d);
		}
		catch(NumberFormatException e) {}
		return null;
	}
	
	/**
	 * File corresponding to the given day
	 */
	private File getFile(int year, int month, int day)
	{
		String fileName = _folder + String.format("/log-%04d-%02d-%02d.png", year, month, day);
		return new File(fileName);
	}
	
	/**
	 * Create an empty image with the appropriate size
	 */
	private IcyBufferedImage createNewImage()
	{
		return new IcyBufferedImage(_width, _height, 3, DataType.UBYTE);
	}
	
	/**
	 * Information associated to frames in the sequence
	 */
	public static class Info
	{
		int year ;
		int month;
		int day  ;
		Info(int y, int m, int d) { year=y; month=m; day=d; } 
	}
	
	/**
	 * Painter
	 */
	private class ConnectivityMapOverlay extends Overlay
	{
		private int _x;
		private int _y;
		private boolean _clipped;
		private DateFormat _dateFormat;
		
		ConnectivityMapOverlay()
		{
			super("Date/time", OverlayPriority.TEXT_NORMAL);
			_x = 0;
			_y = 0;
			_clipped = false;
			_dateFormat = new SimpleDateFormat("EEEE d MMMM yyyy");
			_sequence.addPainter(this);
		}
		
		@Override
		public void mouseMove(MouseEvent e, Point2D imagePoint, IcyCanvas canvas)
		{
			if(_clipped) {
				return;
			}
			refreshXY(imagePoint);
		}
		
		
		@Override
		public void mouseClick(MouseEvent e, Point2D imagePoint, IcyCanvas canvas)
		{
			_clipped = !_clipped;
			refreshXY(imagePoint);
		}
		
		private void refreshXY(Point2D imagePoint)
		{
			_x = (int)imagePoint.getX();
			_y = (int)imagePoint.getY();
			_x = Math.min(Math.max(_x, 0), _width -1);
			_y = Math.min(Math.max(_y, 0), _height-1);
			_sequence.painterChanged(this);
		}
		
		@Override
		public void paint(Graphics2D g0, Sequence sequence, IcyCanvas canvas)
		{
			if(g0==null) return;
			if(!(canvas instanceof Canvas2D)) return;
			Graphics2D g = null;
			try {
				g = (Graphics2D)g0.create();
				
				// Color
				Color foreground = new Color(0, 128, 255);
				g.setColor(foreground);
				
				// Draw the mire
				g.translate(0.5, 0.5);
				int mireRadius   = 3;
				int squareRadius = _clipped ? 6 : 8;
				if(!_clipped) {
					if(_y-mireRadius>=0        ) g.drawLine( _x, 0        , _x, _y-mireRadius);
					if(_y+mireRadius<=_height-1) g.drawLine( _x, _height-1, _x, _y+mireRadius);
					if(_x-mireRadius>=0        ) g.drawLine(0       , _y, _x-mireRadius, _y);
					if(_x+mireRadius<=_width -1) g.drawLine(_width-1, _y, _x+mireRadius, _y);
				}
				g.drawLine(_x-squareRadius, _y-squareRadius, _x-squareRadius, _y+squareRadius);
				g.drawLine(_x+squareRadius, _y-squareRadius, _x+squareRadius, _y+squareRadius);
				g.drawLine(_x-squareRadius, _y-squareRadius, _x+squareRadius, _y-squareRadius);
				g.drawLine(_x-squareRadius, _y+squareRadius, _x+squareRadius, _y+squareRadius);
				g.translate(-0.5, -0.5);
				if(_clipped) {
					g.drawOval(_x-1, _y-1, 3, 3);
					g.drawLine(_x-squareRadius+1, _y-squareRadius+1, _x-1, _y-1);
					g.drawLine(_x-squareRadius+1, _y+squareRadius  , _x-1, _y+2);
					g.drawLine(_x+squareRadius  , _y-squareRadius+1, _x+2, _y-1);
					g.drawLine(_x+squareRadius  , _y+squareRadius  , _x+2, _y+2);
				}
				
				
				// Day corresponding to the current frame
				Info info = _infos.get(canvas.getPositionT());
				if(info==null) return;
				Calendar calendar = Calendar.getInstance();
				calendar.set(info.year, info.month-1, info.day);
				String day = _dateFormat.format(calendar.getTime());
				
				// Time corresponding to the current coordinate (x,y)
				int hr  = _y / 4;
				int min = (_y % 4)*15 + _x/(60 / _interval);
				int sec = (_x % (60/_interval)) * _interval;
				String time = String.format("%02d:%02d:%02d", hr, min, sec);
				
				// Draw the text
				double fontScale = 1.3;
				g.transform(((Canvas2D)canvas).getInverseTransform());
				g.setFont(g.getFont().deriveFont(AffineTransform.getScaleInstance(fontScale, fontScale)));
				GraphicsUtil.drawHint(g, day + "\n" + time, 3, 3, foreground, Color.WHITE);
			}
			
			// Release the graphics object
			finally {
				if(g!=null) {
					g.dispose();
				}
			}
		}
	}
}
