/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.ellipsoid;

import icy.roi.ROI;

import java.util.HashMap;

/**
 * Abstract class describing an ellipsoid. Gives access to some general methods.
 * It's also possible to add custom properties to the ellipsoid, indexing them
 * by name.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public abstract class AbstractEllipsoid {
	/** Time associated with this ellipsoid. */
	protected int t_;

	protected HashMap<String, Double> properties_ = new HashMap<String, Double>();

	/** Return another instance of this descriptor sharing the same properties. */
	@Override
	public abstract AbstractEllipsoid clone();

	/** Return true if the descriptor represents a valid ellipsoid. */
	public abstract boolean isValid();

	/** Minimum radius of the ellipsoid. */
	public abstract double getMinimalRadius();

	/** Maximum radius of the ellipsoid. */
	public abstract double getMaximalRadius();

	/** Perimeter of the ellipsoid (surface in 3D). */
	public abstract double getPerimeter();

	/** Volume of the ellipsoid (surface in 2D). */
	public abstract double getVolume();

	/**
	 * Return the matrix of rotation corresponding to the ellipsoid orientation.
	 */
	public abstract double[] getRotationMatrix();

	/** Time frame of the ellipsoid. */
	public int getT() {
		return t_;
	}

	/** Set the time associated with this ellipsoid. */
	public void setT(int t) {
		t_ = t;
	}

	/**
	 * Set the value of a property. If the property doesn't exist yet, it is
	 * added to the list of ellipsoid properties.
	 * 
	 * @param propertyName
	 *            Name of the property to set.
	 * @param value
	 *            New value of the property.
	 */
	public void setProperty(final String propertyName, double value) {
		properties_.put(propertyName, value);
	}

	/**
	 * Return the value of a property.
	 * 
	 * @throws IllegalArgumentException
	 *             An exception is raised if the property doesn't exist
	 */
	public double getProperty(final String propertyName) {
		if (!properties_.containsKey(propertyName)) {
			throw new IllegalArgumentException("The property '" + propertyName
					+ "' doesn't exist");
		}
		return properties_.get(propertyName);
	}

	/** Return a new ROI corresponding to this descriptor. */
	public abstract ROI toROI();

	/**
	 * Apply a rotation to a vector.
	 * 
	 * @param R
	 *            Rotation matrix
	 * @param V
	 *            Vector to rotate
	 * @return Return the rotated vector
	 */
	public static double[] applyRotation(final double R[], final double V[]) {
		return multiplyMatrices(R, V);
	}

	/**
	 * Multiply to matrices. The matrices have to be of size 3x3, 4x4, 3x1 or
	 * 4x1.
	 */
	public static double[] multiplyMatrices(final double A[], final double B[]) {
		int m, n, p;
		if (A.length == 4) {
			m = n = 2;
		} else {
			m = n = 3;
		}
		p = B.length / n;

		double C[] = new double[m * p];
		for (int i = 0; i < m; ++i) {
			for (int j = 0; j < p; ++j) {
				double c = 0;
				for (int k = 0; k < n; ++k) {
					c += A[i + m * k] * B[k + n * j];
				}
				C[i + m * j] = c;
			}
		}

		return C;
	}

	/**
	 * Transpose the given matrix. The matrix has to be of size 3x3 or 4x4.
	 */
	public static double[] transposeMatrix(final double A[]) {
		int n = (A.length == 4) ? 2 : 3;
		double B[] = new double[A.length];

		for (int i = 0; i < n; ++i) {
			for (int j = 0; j < n; ++j) {
				B[i + n * j] = A[j + n * i];
			}
		}

		return B;
	}
}
