/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.snake2D;

import icy.util.XMLUtil;

import java.awt.geom.Point2D;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * This class is used to store the snake-defining parameters. It extends the
 * capabilities of the class <code>Point2D.Double</code> by additional state
 * variables.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @author Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 */
public class Snake2DNode extends Point2D.Double {

	/**
	 * Determines if a de-serialized file is compatible with this class.
	 * 
	 * Maintainers must change this value if and only if the new version of this
	 * class is not compatible with old versions. See Sun docs for <a
	 * href=http://java.sun.com/products/jdk/1.1/docs/guide
	 * /serialization/spec/version.doc.html> details. </a>
	 * 
	 * Not necessary to include in first version of the class, but included here
	 * as a reminder of its importance.
	 */
	private static final long serialVersionUID = -3850694037976061520L;

	// ----------------------------------------------------------------------------
	// STATE FIELDS

	/**
	 * Methods of the class <code>Snake3DKeeper</code> are allowed to modify
	 * <code>Snake2DNode</code> objects only if their <code>frozen</code> flag
	 * is set to <code>false</code>.
	 */
	private boolean frozen_ = false;
	/**
	 * Methods of the class <code>Snake2DKeeper</code> are allowed to
	 * interactively display (as a cross) the nodes stored in
	 * <code>Snake3DNode</code> objects only if their <code>hidden</code> flag
	 * is set to <code>false</code>.
	 */
	private boolean hidden_ = false;

	// ----------------------------------------------------------------------------
	// XML TAGS

	public static final String ID_X = "x";
	public static final String ID_Y = "y";
	public static final String ID_FROZEN = "frozen";
	public static final String ID_HIDDEN = "hidden";

	// ============================================================================
	// PUBLIC METHODS

	/**
	 * This constructor builds a point that is initially neither frozen nor
	 * hidden.
	 */
	public Snake2DNode(final double x, final double y) {
		super(x, y);
	}

	// ----------------------------------------------------------------------------

	/** This constructor builds a point with the given initial values. */
	public Snake2DNode(final double x, final double y, final boolean frozen,
			final boolean hidden) {
		super(x, y);
		frozen_ = frozen;
		hidden_ = hidden;
	}

	// ----------------------------------------------------------------------------

	public void loadFromXML(Node node) {
		x = XMLUtil.getElementDoubleValue(node, ID_X, 0d);
		y = XMLUtil.getElementDoubleValue(node, ID_Y, 0d);
		frozen_ = XMLUtil.getElementBooleanValue(node, ID_FROZEN, false);
		hidden_ = XMLUtil.getElementBooleanValue(node, ID_HIDDEN, false);
	}

	// ----------------------------------------------------------------------------

	public void saveToXML(Element node) {
		XMLUtil.setElementDoubleValue(node, ID_X, x);
		XMLUtil.setElementDoubleValue(node, ID_Y, y);
		XMLUtil.setElementBooleanValue(node, ID_FROZEN, frozen_);
		XMLUtil.setElementBooleanValue(node, ID_HIDDEN, hidden_);
	}

	// ----------------------------------------------------------------------------

	/** This method returns text-based information about this object. */
	@Override
	public String toString() {
		return "[" + super.toString() + ", frozen: " + frozen_ + ", hidden: "
				+ hidden_ + "]";
	}

	// ----------------------------------------------------------------------------

	/** Sets the hidden state of the node to <true>. */
	public void hide() {
		hidden_ = true;
	}

	// ----------------------------------------------------------------------------

	/** Sets the hidden state of the node to <false>. */
	public void unhide() {
		hidden_ = false;
	}

	// ----------------------------------------------------------------------------

	/** Returns the hidden state of the node. */
	public boolean isHidden() {
		return hidden_;
	}

	// ----------------------------------------------------------------------------

	/** Sets the frozen state of the node to <true>. */
	public void freeze() {
		frozen_ = true;
	}

	// ----------------------------------------------------------------------------

	/** Sets the frozen state of the node to <false>. */
	public void unfreeze() {
		frozen_ = false;
	}

	// ----------------------------------------------------------------------------

	/** Returns the frozen state of the node. */
	public boolean isFrozen() {
		return frozen_;
	}
}
