/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.shape.utils;

import java.awt.Point;
import java.awt.Polygon;
import java.util.Vector;

/**
 * Class that encapsulates a contour tracing algorithm.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class ContourTracing {
	private final boolean[] T;
	private int nPoints;
	private final int width;
	private final int height;
	private final Vector<Point> path;
	private static final int UP = 0;
	private static final int DOWN = 1;
	private static final int LEFT = 2;
	private static final int RIGHT = 3;

	// ============================================================================
	// PUBLIC METHODS

	public ContourTracing(boolean[] T, int width, int height) {
		this.T = T;
		this.width = width;
		this.height = height;
		path = new Vector<Point>();
		nPoints = 0;
	}

	// ----------------------------------------------------------------------------

	public void trace() {
		int sx = 0;
		int sy = 0;
		int px = 0;
		int py = 0;
		int direction = UP;

		boolean found = false;
		for (int y = height - 1; y >= 0 && !found; y--) {
			for (int x = 0; x < width && !found; x++) {
				if (T[x + y * width]) {
					found = true;
					sx = x;
					sy = y;
					px = x;
					py = y;
					path.addElement(new Point(x, y));
				}
			}
		}

		direction = RIGHT;
		px--;
		boolean cond;
		while (!(sx == px && sy == py)) {

			if (px < 0 || px >= width || py < 0 || py >= height) {
				cond = false;
			} else {
				cond = T[px + py * width];
			}
			if (cond) {
				path.addElement(new Point(px, py));
				switch (direction) {
				case UP:
					px++;
					direction = LEFT;
					break;
				case DOWN:
					px--;
					direction = RIGHT;
					break;
				case LEFT:
					py--;
					direction = DOWN;
					break;
				case RIGHT:
					py++;
					direction = UP;
					break;
				default:
					System.err.println("Orientation unknown");
					break;
				}
			} else {
				switch (direction) {
				case UP:
					px--;
					direction = RIGHT;
					break;
				case DOWN:
					px++;
					direction = LEFT;
					break;
				case LEFT:
					py++;
					direction = UP;
					break;
				case RIGHT:
					py--;
					direction = DOWN;
					break;
				default:
					System.err.println("Orientation unknown");
					break;
				}
			}
		}
		nPoints = path.size();
	}

	// ============================================================================
	// GETTERS

	public Polygon getTrace() {
		return new Polygon(getXCoordinates(), getYCoordinates(), nPoints);
	}

	// ----------------------------------------------------------------------------

	public int[] getXCoordinates() {
		int[] xCoordinates = new int[nPoints];
		for (int i = 0; i < nPoints; i++) {
			xCoordinates[i] = path.elementAt(i).x;
		}
		return xCoordinates;
	}

	// ----------------------------------------------------------------------------

	public int[] getYCoordinates() {
		int[] yCoordinates = new int[nPoints];
		for (int i = 0; i < nPoints; i++) {
			yCoordinates[i] = path.elementAt(i).y;
		}
		return yCoordinates;
	}

	// ----------------------------------------------------------------------------

	public int getNPoints() {
		return nPoints;
	}
}
