/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.gui.pair;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import plugins.adufour.vars.gui.VarEditor;

/**
 * VarEditor for Pair objects. The editor is compose of tow JSpinner separated
 * by a linking word (the default is '-'). This linking word cannot be changed
 * in Protocol.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public class PairEditor extends VarEditor<Pair> {
	private JSpinner firstSpinner_;
	private JSpinner secondSpinner_;
	private JLabel textLabel_;
	private SpinnerNumberModel firstSpinnerModel_;
	private SpinnerNumberModel secondSpinnerModel_;

	private ChangeListener changeListener;
	private FocusListener focusListener;
	private KeyListener keyListener;

	/** default constructor */
	public PairEditor(VarPair variable) {
		super(variable);
	}

	/** Set the minimum for the first value of the pair. */
	public void setFirstMinimum(double min) {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		cons.setFirstMinimum(min);
		updateSpinnersBounds();
	}

	/** Returns the minimum for the first value of the pair. */
	public double getFirstMinimum() {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		return cons.getFirstMinimum();
	}

	/** Set the maximum for the first value of the pair. */
	public void setFirstMaximum(double max) {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		cons.setFirstMaximum(max);
		updateSpinnersBounds();
	}

	/** Returns the maximum for the first value of the pair. */
	public double getFirstMaximum() {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		return cons.getFirstMaximum();
	}

	/** Set the bounds for the first value of the pair. */
	public void setFirstBounds(double min, double max) {
		setFirstMinimum(min);
		setFirstMaximum(max);
	}

	/** Set the minimum for the second value of the pair. */
	public void setSecondMinimum(double min) {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		cons.setSecondMinimum(min);
		updateSpinnersBounds();
	}

	/** Returns the minimum for the second value of the pair. */
	public double getSecondMinimum() {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		return cons.getSecondMinimum();
	}

	/** Set the maximum for the second value of the pair. */
	public void setSecondMaximum(double max) {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		cons.setSecondMaximum(max);
		updateSpinnersBounds();
	}

	/** Returns the maximum for the second value of the pair. */
	public double getSecondMaximum() {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		return cons.getSecondMaximum();
	}

	/** Set the bounds for the second value of the pair. */
	public void setSecondBounds(double min, double max) {
		setSecondMinimum(min);
		setSecondMaximum(max);
	}

	/** Set the minimum for the both values of the pair. */
	public void setMinimum(double min) {
		setFirstMinimum(min);
		setSecondMinimum(min);
	}

	/** Set the maximum for the both values of the pair. */
	public void setMaximum(double max) {
		setFirstMaximum(max);
		setSecondMaximum(max);
	}

	/** Set the bounds for the both values of the pair. */
	public void setBounds(double min, double max) {
		setFirstBounds(min, max);
		setSecondBounds(min, max);
	}

	/** Set the step size for the first value of the pair. */
	public void setFirstStepSize(double step) {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		cons.setFirstStepSize(step);
		firstSpinnerModel_.setStepSize(step);
	}

	/** Returns the step size for the first value of the pair. */
	public double getFirstStepSize() {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		return cons.getFirstStepSize();
	}

	/** Set the step size for the second value of the pair. */
	public void setSecondStepSize(double step) {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		cons.setSecondStepSize(step);
		secondSpinnerModel_.setStepSize(step);
	}

	/** Returns the step size for the second value of the pair. */
	public double getSecondStepSize() {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		return cons.getSecondStepSize();
	}

	/** Set the step size for the both values of the pair. */
	public void setStepSize(double step) {
		setFirstStepSize(step);
		setSecondStepSize(step);
	}

	/**
	 * Set the linking word displayed between the first and the second value of
	 * the pair.
	 */
	public void setLinkingWord(final String word) {
		textLabel_.setText(word);
		getEditorComponent().repaint();
	}

	/**
	 * Returns the linking word displayed between the first and the second value
	 * of the pair.
	 */
	public String getLinkingWord() {
		return textLabel_.getText();
	}

	@Override
	public JComponent getEditorComponent() {
		return (JComponent) super.getEditorComponent();
	}

	@Override
	public Dimension getPreferredSize() {
		return getEditorComponent().getPreferredSize();
	}

	@Override
	public void setComponentToolTipText(String s) {
		getEditorComponent().setToolTipText(s);
	}

	@Override
	public void dispose() {
		super.dispose();
	}

	/** Update the variable when an editor is modified. */
	protected void updateVariableValue() {
		try {
			PairRangeModel cons = ((PairRangeModel) getVariable()
					.getDefaultEditorModel());

			Object firstValue = firstSpinner_.getValue();
			Object secondValue = secondSpinner_.getValue();
			Pair value = getVariable().getValue().clone();

			if (firstValue instanceof Number && secondValue instanceof Number) {
				value.setPair(((Number) firstValue).doubleValue(),
						((Number) secondValue).doubleValue());
			}

			cons.updateBounds(value);
			updateSpinnersBounds();

			variable.setValue(value);

			firstSpinner_.setForeground(Color.black);
			secondSpinner_.setForeground(Color.black);
			firstSpinner_.setToolTipText(null);
			secondSpinner_.setToolTipText(null);
		} catch (RuntimeException e) {
			firstSpinner_.setForeground(Color.red);
			secondSpinner_.setForeground(Color.red);
			firstSpinner_.setToolTipText("Cannot convert input into a "
					+ getVariable().getTypeAsString());
			secondSpinner_.setToolTipText("Cannot convert input into a "
					+ getVariable().getTypeAsString());
		}
	}

	@Override
	protected void updateInterfaceValue() {
		Pair value = getVariable().getValue();
		firstSpinner_.setValue(value.getFirst());
		secondSpinner_.setValue(value.getSecond());
	}

	@Override
	protected void setEditorEnabled(boolean enabled) {
		getEditorComponent().setEnabled(enabled);
	}

	@Override
	protected JComponent createEditorComponent() {
		final JPanel fieldsPanel = new JPanel(new GridBagLayout());

		GridBagConstraints fieldGBC = new GridBagConstraints();
		fieldGBC.weightx = 1.0;
		fieldGBC.fill = GridBagConstraints.HORIZONTAL;

		GridBagConstraints textGBC = new GridBagConstraints();
		textGBC.insets = new Insets(0, 5, 0, 5);

		fieldsPanel.add(firstSpinner_ = new JSpinner(), fieldGBC);
		fieldsPanel.add(textLabel_ = new JLabel("-"), textGBC);
		fieldsPanel.add(secondSpinner_ = new JSpinner(), fieldGBC);

		firstSpinner_.setPreferredSize(new Dimension(75, 24));
		secondSpinner_.setPreferredSize(new Dimension(75, 24));

		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());

		firstSpinner_.setModel(firstSpinnerModel_ = new SpinnerNumberModel(cons
				.getDefaultValue().getFirst(), cons.getFirstMinimum(), cons
				.getFirstMaximum(), cons.getFirstStepSize()));
		secondSpinner_.setModel(secondSpinnerModel_ = new SpinnerNumberModel(
				cons.getDefaultValue().getSecond(), cons.getSecondMinimum(),
				cons.getSecondMaximum(), cons.getSecondStepSize()));

		changeListener = new ChangeListener() {
			@Override
			public void stateChanged(ChangeEvent arg0) {
				updateVariableValue();
			}
		};

		focusListener = new FocusAdapter() {
			@Override
			public void focusLost(FocusEvent e) {
				updateVariableValue();
			}
		};

		keyListener = new KeyListener() {
			@Override
			public void keyTyped(KeyEvent e) {
			}

			@Override
			public void keyReleased(KeyEvent e) {
				try {
					updateVariableValue();
					firstSpinner_.setForeground(Color.black);
					secondSpinner_.setForeground(Color.black);
				} catch (Exception ex) {
					firstSpinner_.setForeground(Color.red);
					secondSpinner_.setForeground(Color.red);
				}
			}

			@Override
			public void keyPressed(KeyEvent e) {
			}
		};

		return fieldsPanel;
	}

	@Override
	protected void activateListeners() {
		firstSpinner_.addChangeListener(changeListener);
		firstSpinner_.addFocusListener(focusListener);
		firstSpinner_.addKeyListener(keyListener);

		secondSpinner_.addChangeListener(changeListener);
		secondSpinner_.addFocusListener(focusListener);
		secondSpinner_.addKeyListener(keyListener);
	}

	@Override
	protected void deactivateListeners() {
		firstSpinner_.removeChangeListener(changeListener);
		firstSpinner_.removeFocusListener(focusListener);
		firstSpinner_.removeKeyListener(keyListener);

		secondSpinner_.removeChangeListener(changeListener);
		secondSpinner_.removeFocusListener(focusListener);
		secondSpinner_.removeKeyListener(keyListener);
	}

	/** Update the bounds of the spinners. */
	private void updateSpinnersBounds() {
		PairRangeModel cons = ((PairRangeModel) getVariable()
				.getDefaultEditorModel());
		firstSpinnerModel_.setMinimum(cons.getFirstMinimum());
		firstSpinnerModel_.setMaximum(cons.getFirstMaximum());
		secondSpinnerModel_.setMinimum(cons.getSecondMinimum());
		secondSpinnerModel_.setMaximum(cons.getSecondMaximum());
	}

	@Override
	public boolean isComponentEnabled() {
		return true;
	}

	@Override
	public boolean isComponentOpaque() {
		return true;
	}
}
