package plugins.nchenouard.particletracking.MHTracker;

import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.HashMap;

public class HeuristicBinaryOptimizer
{
	// algorithm parameters
	int Q;
	double[] weights;
	int[][] constrWeights;
	int[] constrBounds;

	// exploration variables
	// best current solution
	boolean[] z;
	double zScore;
	int[] zConstraintValues;
	boolean[] zBooleanTrace;
	// current solution
	boolean[] y;
	double yScore;
	int[] yConstraintValues;
	boolean[] yBooleanTrace;

	// hashmap between traces and scores
	HashMap<Integer, Double> traceBox;
	// interesting solutions to test
	ArrayDeque<boolean[]> interestingSolutions;

	public HeuristicBinaryOptimizer(int Q, double[] weights, int[][] constrWeights, int[] constrBounds)
	{
		this.Q = Q;
		this.weights = weights;
		this.constrWeights = constrWeights;
		this.constrBounds = constrBounds;
	}

	public boolean[] heuristicSearch(boolean[] initSolution)
	{
		z = initSolution.clone(); // best current solution
		computeZScore();
		zConstraintValues = computeConstraintValues(z, constrWeights);
		zBooleanTrace = computeBooleanTrace(z, zConstraintValues, constrBounds);
		traceBox = new HashMap<Integer, Double>(100);

		yConstraintValues = new int[constrBounds.length];
		yBooleanTrace = new boolean[constrBounds.length*2];

		interestingSolutions = new ArrayDeque<boolean[]>(100);
		interestingSolutions.addLast(z);
		while (!interestingSolutions.isEmpty())
		{
			y = interestingSolutions.pollFirst();
			for (int i = 0; i  < y.length; i++) // explore adjacent solutions
			{
				y[i] = !y[i]; // switch value at location i in the current solution vector
				computeYConstraintValues();
				computeYScore();
				computeYBooleanTrace();
				if (isFeasible(yConstraintValues, constrBounds) && yScore > zScore)
				{
					z = y.clone();
					zScore = yScore;
					zConstraintValues = yConstraintValues.clone();
					zBooleanTrace = yBooleanTrace.clone();
					interestingSolutions.clear();
					interestingSolutions.add(y);
					traceBox.clear();
					break;
				}
				else
					testYInteresting();
				y[i] = !y[i];
			}
		}
		return z;
	}

	private void computeYBooleanTrace() {
		for (int i = 0; i < yConstraintValues.length; i++)
		{		
			if (yConstraintValues[i] > constrBounds[i]) // constraint violation
			{
				yBooleanTrace[i] = true;
				yBooleanTrace[i + constrBounds.length] = false;
			}
			else
			{
				yBooleanTrace[i] = false;
				if (yConstraintValues[i] < constrBounds[i]) // constraint looseness
					yBooleanTrace[i + constrBounds.length] = true;
				else
					yBooleanTrace[i + constrBounds.length] = false;
			}
		}
	}

	private void computeYConstraintValues() {
		for (int i = 0; i < constrWeights.length; i++)
		{
			yConstraintValues[i] = 0;
			for (int j = 0; j < z.length; j++)
				if (y[j])
					yConstraintValues[i] += constrWeights[i][j];					
		}
	}

	private int[] computeConstraintValues(boolean[] z, int[][] constrWeights)
	{
		int[] c = new int[constrWeights.length];
		for (int i = 0; i < constrWeights.length; i++)
			for (int j = 0; j < z.length; j++)
				if (z[j])
					c[i] += constrWeights[i][j];
		return c;
	}

	private void testYInteresting()
	{
		// A1: test whether a constraint is violated by more than one unit
		for (int i = 0; i < yConstraintValues.length; i++)
		{
			if (yConstraintValues[i] > constrBounds[i] + 1)
				return;
		}

		// A2: the total amount of violation in y plus the number of different loose constraints (as compared to z) is at most Q
		// use booleans since by virtue of A1 interesting solutions have trace values in {0, 1}		
		int sumDiffTrace = 0;
		for (int i = 0; i < yBooleanTrace.length; i++)
			if (yBooleanTrace[i] != zBooleanTrace[i])
				sumDiffTrace++;
		if (sumDiffTrace > Q)
			return;

		// A3: the score of this solution is better than other solutions already examined and having the same hash for the trace
		int key = computeKey(yBooleanTrace);
		Double prevScore = traceBox.get(new Integer(key));
		if (prevScore != null && prevScore.doubleValue() >= yScore)
			return;
		// add y to the interesting solution list
		interestingSolutions.add(y.clone());
		traceBox.put(key, new Double(yScore));
	}

	private boolean[] computeBooleanTrace(boolean[] s, int[] sConstraintValues, int[] constrBounds) {
		boolean[] trace = new boolean[constrBounds.length*2];
		for (int i = 0; i < sConstraintValues.length; i++)
		{
			if (sConstraintValues[i] > constrBounds[i]) // constraint violation
				trace[i] = true;
			else if (sConstraintValues[i] < constrBounds[i]) // constraint looseness
				trace[i + constrBounds.length] = true;
		}
		return trace;
	}

	private int computeKey(boolean[] traceTab) {
		return Arrays.hashCode(traceTab);
	}

	private boolean isFeasible(int[] sConstraintValues, int[] constrBounds) {
		for (int i = 0; i < constrBounds.length; i++)
			if (sConstraintValues[i] > constrBounds[i])
				return false;
		return true;
	}

	private void computeYScore() {
		yScore = 0;
		for (int i = 0; i < y.length; i++)
			if (y[i])
				yScore += weights[i];
	}

	private void computeZScore() {
		zScore = 0;
		for (int i = 0; i < z.length; i++)
			if (z[i])
				zScore += weights[i];
	}
}
