package plugins.adufour.viewers;

import icy.canvas.IcyCanvas;
import icy.canvas.IcyCanvas3D;
import icy.common.listener.ProgressListener;
import icy.gui.frame.progress.ToolTipFrame;
import icy.gui.viewer.Viewer;
import icy.image.colormap.IcyColorMap;
import icy.image.lut.LUT;
import icy.image.lut.LUT.LUTChannel;
import icy.image.lut.LUTEvent;
import icy.image.lut.LUTListener;
import icy.plugin.abstract_.Plugin;
import icy.plugin.interface_.PluginCanvas;
import icy.sequence.DimensionId;
import icy.sequence.Sequence;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceEvent.SequenceEventSourceType;
import icy.sequence.SequenceListener;
import icy.system.IcyHandledException;
import icy.system.thread.ThreadUtil;
import icy.type.TypeUtil;
import icy.vtk.IcyVtkPanel;
import icy.vtk.VtkUtil;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.InputEvent;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;

import javax.swing.SwingUtilities;

import vtk.vtkCubeAxesActor2D;
import vtk.vtkElevationFilter;
import vtk.vtkImageData;
import vtk.vtkImageDataGeometryFilter;
import vtk.vtkLODActor;
import vtk.vtkLookupTable;
import vtk.vtkPanel;
import vtk.vtkPolyDataMapper;
import vtk.vtkRenderWindow;
import vtk.vtkRenderer;
import vtk.vtkSliderRepresentation2D;
import vtk.vtkUnsignedCharArray;
import vtk.vtkWarpScalar;

public class Elevation extends Plugin implements PluginCanvas
{
    @Override
    public String getCanvasClassName()
    {
        return ElevationCanvas.class.getName();
    }
    
    @Override
    public IcyCanvas createCanvas(Viewer viewer)
    {
        return new ElevationCanvas(viewer);
    }
    
    @SuppressWarnings("serial")
    public class ElevationCanvas extends IcyCanvas3D
    {
        private class MyVtkPanel extends IcyVtkPanel implements MouseWheelListener
        {
            public MyVtkPanel()
            {
                addMouseWheelListener(this);
            }
            
            @Override
            public void Delete()
            {
                removeMouseWheelListener(this);
                super.Delete();
                if (getSequence() != null)
                {
                    getSequence().removeListener(dataListener);
                    userLUT.removeListener(lutListener);
                }
            }
            
            public void keyTyped(java.awt.event.KeyEvent e)
            {
                if (e.getKeyChar() == 'c' || e.getKeyChar() == 'C')
                {
                    setPositionC((getPositionC() + 1) % getSequence().getSizeC());
                    e.consume();
                }
                else if (e.getKeyChar() == '-')
                {
                    warp.SetScaleFactor(warp.GetScaleFactor() * 9 / 10);
                    e.consume();
                    refresh();
                }
                else if (e.getKeyChar() == '+')
                {
                    warp.SetScaleFactor(warp.GetScaleFactor() * 10 / 9);
                    e.consume();
                    refresh();
                }
            };
            
            public void mousePressed(java.awt.event.MouseEvent e)
            {
                if (ren.VisibleActorCount() == 0) return;
                rw.SetDesiredUpdateRate(5.0);
                lastX = e.getX();
                lastY = e.getY();
                if ((e.getModifiers() == InputEvent.BUTTON2_MASK) || (e.getModifiers() == (InputEvent.BUTTON1_MASK | InputEvent.SHIFT_MASK)))
                {
                    InteractionModeZoom();
                }
                else if (e.getModifiers() == InputEvent.BUTTON3_MASK)
                {
                    InteractionModeRotate();
                }
                else
                {
                    InteractionModeTranslate();
                }
            }
            
            @Override
            public void mouseWheelMoved(MouseWheelEvent e)
            {
                double zoomFactor;
                zoomFactor = Math.pow(1.02, (e.getWheelRotation()));
                if (cam.GetParallelProjection() == 1)
                {
                    cam.SetParallelScale(cam.GetParallelScale() / zoomFactor);
                }
                else
                {
                    cam.Dolly(zoomFactor);
                    resetCameraClippingRange();
                }
                Render();
            }
        }
        
        final vtkPanel                   panel;
        
        final vtkRenderer                renderer;
        final vtkImageDataGeometryFilter geom            = new vtkImageDataGeometryFilter();
        final vtkWarpScalar              warp            = new vtkWarpScalar();
        final vtkElevationFilter         elevationFilter = new vtkElevationFilter();
        final vtkCubeAxesActor2D         axes            = new vtkCubeAxesActor2D();
        final vtkLookupTable             vtkLUT          = new vtkLookupTable();
        private vtkImageData             imageData       = new vtkImageData();
        final vtkSliderRepresentation2D  slider          = new vtkSliderRepresentation2D();
        
        private final LUT                userLUT;
        
        private final LUTListener                      lutListener     = new LUTListener()
                                                         {
                                                             @Override
                                                             public void lutChanged(LUTEvent event)
                                                             {
                                                                 updateVTKLUT();
                                                                 refresh();
                                                             }
                                                         };
        
        private final SequenceListener                 dataListener    = new SequenceListener()
                                                         {
                                                             
                                                             @Override
                                                             public void sequenceClosed(Sequence sequence)
                                                             {
                                                                 sequence.removeListener(this);
                                                             }
                                                             
                                                             @Override
                                                             public void sequenceChanged(SequenceEvent sequenceEvent)
                                                             {
                                                                 if (sequenceEvent.getSourceType() == SequenceEventSourceType.SEQUENCE_DATA)
                                                                 {
                                                                     updateVtkImageData();
                                                                     refresh();
                                                                 }
                                                             }
                                                         };
        
        final ToolTipFrame               helpBox;
        
        public ElevationCanvas(Viewer viewer)
        {
            super(viewer);
            
            MyVtkPanel myPanel;
            
            try
            {
                myPanel = new MyVtkPanel();
            }
            catch (UnsatisfiedLinkError error)
            {
                throw new IcyHandledException("Cannot load the elevation map: VTK library not found.");
            }
            
            panel = myPanel;
            renderer = panel.GetRenderer();
            
            add(panel, BorderLayout.CENTER);
            
            userLUT = getLut();
            
            setPositionC(0);
            setPositionZ(viewer.getPositionZ());
            
            updateZNav();
            updateTNav();
            setPositionX(-1);
            setPositionY(-1);
            
            getMouseImageInfosPanel().setVisible(false);
            
            int sizeX = getSequence().getSizeX();
            int sizeY = getSequence().getSizeY();
            
            int minDim = Math.min(sizeX, sizeY);
            
            double max = getSequence().getChannelMax(getPositionC());
            
            // 1) Construct the elevation map from the image data
            
            // create the geometry from the image data (not the actual data!)
            geom.SetOutputTriangles(1);
            
            // warp image values to height data using a given scale
            warp.SetInputData(geom.GetOutput());
            // invert the scale to extrude the image toward the camera
            warp.SetScaleFactor(-0.25 / max * minDim);
            
            elevationFilter.SetInputConnection(warp.GetOutputPort());
            
            vtkPolyDataMapper mapper = new vtkPolyDataMapper();
            mapper.SetInputConnection(elevationFilter.GetOutputPort());
            
            mapper.SetLookupTable(vtkLUT);
            mapper.SetImmediateModeRendering(1);
            
            vtkLODActor actor = new vtkLODActor();
            // optimize mouse dragging by showing only 1/100 of the pixels
            actor.SetNumberOfCloudPoints(sizeX * sizeY / 100);
            actor.SetScale(1, 1, 1);
            actor.SetMapper(mapper);
            renderer.AddActor(actor);
            
            // 2) Draw axes around the volume
            
            axes.SetInputData(elevationFilter.GetPolyDataOutput());
            axes.SetCamera(renderer.GetActiveCamera());
            axes.SetLabelFormat("%7.0f");
            axes.SetXLabel("X");
            axes.SetYLabel("Y");
            axes.SetZLabel("Intensity");
            axes.SetFlyModeToOuterEdges();// None();// ClosestTriad();
            axes.SetFontFactor(0.9);
            axes.GetProperty().SetColor(1.0, 1.0, 1.0);
            axes.UseBoundsOff();
            axes.UseRangesOn();
            // don't offset the axes => bounds will be aligned with the data
            axes.SetCornerOffset(0);
            
            renderer.AddActor(axes);
            
            renderer.SetBackground(0, 0, 0);
            
            // update the LUT one first time...
            
            LUTChannel currentLUT = userLUT.getLutChannel(getPositionC());
            IcyColorMap map = currentLUT.getColorMap();
            vtkLUT.SetNumberOfColors(IcyColorMap.SIZE);
            for (int i = 0; i < IcyColorMap.SIZE; i++)
                vtkLUT.SetTableValue(i, map.red.mapf[i], map.green.mapf[i], map.blue.mapf[i], 1.0);
            updateVTKLUT();
            
            // ... and every time the user LUT changes
            userLUT.addListener(lutListener);
            
            renderer.GetActiveCamera().SetViewUp(0, -1, 0);
            renderer.ResetCamera();
            renderer.GetActiveCamera().Elevation(180);
            renderer.ResetCameraClippingRange();
            
            String helpText = "<b>How to use the elevation map:</b><br><ul>";
            helpText += " <li>interact with the mouse and colormap just like in the 2D viewer<br>";
            helpText += " <li>press 'c' to toggle between channels<br>";
            helpText += " <li>press '+' or '-' to increase or decrease the depth";
            helpText += "</ul><i>(click on this message box to dismiss it)</i>";
            
            helpBox = new ToolTipFrame(helpText);
            
            SwingUtilities.invokeLater(refresher);
            
            // Update every time the sequence data changes (for real-time visualization)
            getSequence().addListener(dataListener);
        }
        
        private Runnable refresher = new Runnable()
                                   {
                                       @Override
                                       public void run()
                                       {
                                           
                                           ThreadUtil.invokeLater(new Runnable()
                                           {
                                               public void run()
                                               {
                                                   try
                                                   {
                                                       panel.Render();
                                                   }
                                                   catch (Exception e)
                                                   {
                                                   }
                                               }
                                           });
                                       }
                                   };
        
        @Override
        public void refresh()
        {
            ThreadUtil.bgRunSingle(refresher);
        }
        
        @Override
        protected void positionChanged(DimensionId dim)
        {
            updateVtkImageData();
            
            if (dim == DimensionId.C) updateVTKLUT();
            
            refresh();
        }
        
        @Override
        public Component getViewComponent()
        {
            return panel;
        }
        
        @Override
        public BufferedImage getRenderedImage(int t, int z, int c, boolean canvasView)
        {
            int oldT = getPositionT();
            int oldZ = getPositionZ();
            int oldC = getPositionC();
            
            setPositionT(t);
            setPositionZ(z);
            setPositionC(c);
            
            try
            {
                ThreadUtil.sleep(1000);
                return getRenderedImage();
            }
            finally
            {
                setPositionT(oldT);
                setPositionZ(oldZ);
                setPositionC(oldC);
            }
        }
        
        @Override
        public Sequence getRenderedSequence(boolean canvasView, ProgressListener progressListener)
        {
            Sequence screenshot = new Sequence("Screenshot of " + getSequence().getName() + " (channel " + getSequence().getChannelName(getPositionC()) + ")");
            
            if (getSequence().getSizeZ() == 1 && getSequence().getSizeT() == 1)
            {
                screenshot.setImage(0, 0, getRenderedImage());
                return screenshot;
            }
            
            int oldT = getPositionT();
            int oldZ = getPositionZ();
            
            try
            {
                for (int t = 0; t < getSequence().getSizeT(); t++)
                    for (int z = 0; z < getSequence().getSizeZ(); z++)
                    {
                        setPositionTInternal(t);
                        setPositionZInternal(z);
                        positionChanged(DimensionId.NULL);
                        ThreadUtil.sleep(1000);
                        screenshot.setImage(t, z, getRenderedImage());
                    }
                return screenshot;
            }
            finally
            {
                setPositionT(oldT);
                setPositionZ(oldZ);
            }
        }
        
        public BufferedImage getRenderedImage()
        {
            final vtkRenderWindow renderWindow = renderer.GetRenderWindow();
            final int[] size = renderWindow.GetSize();
            final int w = size[0];
            final int h = size[1];
            final vtkUnsignedCharArray array = new vtkUnsignedCharArray();
            
            // VTK need this to be called in the EDT :-(
            ThreadUtil.invokeNow(new Runnable()
            {
                @Override
                public void run()
                {
                    renderWindow.GetRGBACharPixelData(0, 0, w - 1, h - 1, 1, array);
                }
            });
            
            // convert the vtk array into a IcyBufferedImage
            final byte[] inData = array.GetJavaArray();
            final BufferedImage image = new BufferedImage(w, h, BufferedImage.TYPE_INT_ARGB);
            final int[] outData = ((DataBufferInt) image.getRaster().getDataBuffer()).getData();
            
            int inOffset = 0;
            for (int y = h - 1; y >= 0; y--)
            {
                int outOffset = y * w;
                
                for (int x = 0; x < w; x++)
                {
                    final int r = TypeUtil.unsign(inData[inOffset++]);
                    final int g = TypeUtil.unsign(inData[inOffset++]);
                    final int b = TypeUtil.unsign(inData[inOffset++]);
                    final int a = -1;
                    inOffset++;// TypeUtil.unsign(inData[inOffset++]);
                    
                    outData[outOffset++] = (a << 24) | (r << 16) | (g << 8) | (b << 0);
                }
            }
            
            return image;
        }
        
        /**
         * @see VtkUtil#getImageData(Object, icy.type.DataType, int, int, int, int)
         * @param plane
         * @return
         */
        private void updateVtkImageData()
        {
            Sequence seq = getSequence();
            if (getPositionC() == -1 || getPositionZ() == -1 || getPositionT() == -1) return;
            
            imageData = VtkUtil.getImageData(seq.getDataXY(getPositionT(), getPositionZ(), getPositionC()), seq.getDataType_(), seq.getSizeX(), seq.getSizeY(), seq.getSizeZ(), 1);
            geom.SetInputData(imageData);
            geom.Update();
            
            // update range axes
            axes.SetRanges(0, getSequence().getSizeX(), 0, getSequence().getSizeY(), getCurrentImage().getChannelMax(0), getSequence().getChannelMin(0));
        }
        
        private void updateVTKLUT()
        {
            LUTChannel currentLUT = userLUT.getLutChannel(getPositionC());
            
            IcyColorMap map = currentLUT.getColorMap();
            
            double newMin = currentLUT.getMin() * warp.GetScaleFactor(), newMax = currentLUT.getMax() * warp.GetScaleFactor();
            
            double[] rgba = new double[4];
            for (int i = 0; i < IcyColorMap.SIZE; i++)
            {
                rgba[0] = map.red.mapf[i];
                rgba[1] = map.green.mapf[i];
                rgba[2] = map.blue.mapf[i];
                rgba[3] = map.alpha.mapf[i];
                vtkLUT.SetTableValue(i, rgba);
            }
            elevationFilter.SetLowPoint(0, 0, newMin);
            elevationFilter.SetHighPoint(0, 0, newMax);
        }
        
        @Override
        public void shutDown()
        {
            userLUT.removeListener(lutListener);
            
            panel.Delete();
            
            helpBox.close();
            
            super.shutDown();
        }
    }
}