/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.ellipsoid;

import icy.roi.ROI;

import java.util.HashMap;

/**
 * Class describing a 3D ellipsoid.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class Ellipsoid3D extends AbstractEllipsoid {
	/** One of the axis of the ellipsoid. */
	public double a = 0;

	/** One of the axis of the ellipsoid. */
	public double b = 0;

	/** One of the axis of the ellipsoid. */
	public double c = 0;

	/** x coordinate of the center of the ellipsoid. */
	public double x0 = 0;

	/** y coordinate of the center of the ellipsoid. */
	public double y0 = 0;

	/** z coordinate of the center of the ellipsoid. */
	public double z0 = 0;

	/** One of the angle of the ellipsoid. */
	public double alpha = 0;

	/** One of the angle of the ellipsoid. */
	public double beta = 0;

	/** One of the angle of the ellipsoid. */
	public double gamma = 0;

	/** Default constructor. */
	public Ellipsoid3D() {
		t_ = 0;
	}

	/**
	 * Default constructor.
	 * 
	 * @param t
	 *            Time corresponding to this cell.
	 */
	public Ellipsoid3D(int t) {
		t_ = t;
	}

	/**
	 * Copy constructor.
	 * 
	 * @param other
	 *            Cell to copy.
	 */
	public Ellipsoid3D(final Ellipsoid3D other) {
		a = other.a;
		b = other.b;
		c = other.c;
		x0 = other.x0;
		y0 = other.y0;
		z0 = other.z0;
		alpha = other.alpha;
		beta = other.beta;
		gamma = other.gamma;
		t_ = other.t_;
		properties_ = new HashMap<String, Double>(other.properties_);
	}

	@Override
	public AbstractEllipsoid clone() {
		return new Ellipsoid3D(this);
	}

	@Override
	public boolean isValid() {
		return (!((Double) a).isNaN() && !((Double) b).isNaN()
				&& !((Double) c).isNaN() && !((Double) x0).isNaN()
				&& !((Double) y0).isNaN() && !((Double) z0).isNaN()
				&& !((Double) alpha).isNaN() && !((Double) beta).isNaN()
				&& !((Double) gamma).isNaN() && c > 0 && b > 0 && a > 0);
	}

	@Override
	public double getMinimalRadius() {
		return Math.min(Math.min(a, b), c);
	}

	@Override
	public double getMaximalRadius() {
		return Math.max(Math.max(a, b), c);
	}

	@Override
	public double getPerimeter() {
		double p = 1.6075;
		double ap = Math.pow(a, p);
		double bp = Math.pow(b, p);
		double cp = Math.pow(c, p);

		return 4.0 * Math.PI
				* Math.pow((ap * bp + bp * cp + cp * ap) / 3.0, 1.0 / p);
	}

	@Override
	public double getVolume() {
		return 4.0 / 3.0 * Math.PI * a * b * c;
	}

	@Override
	public double[] getRotationMatrix() {
		double cz = Math.cos(alpha);
		double sz = Math.sin(alpha);
		double cx = Math.cos(beta);
		double sx = Math.sin(beta);
		double cy = Math.cos(gamma);
		double sy = Math.sin(gamma);

		double[] R = { cy * cz, sx * sy * cz - cx * sz, sx * sz + cx * sy * cz,
				cy * sz, cx * cz + sx * sy * sz, cx * sy * sz - sx * cz, -sy,
				sx * cy, cx * cy };

		return R;
	}

	@Override
	public ROI toROI() {
		return new EllipsoidROI3D(this);
	}

	@Override
	public String toString() {
		return "Ellipsoid (t=" + t_ + ")";
	}

	/**
	 * Returns the intersection of this ellipsoid with a plan z = layer.
	 * 
	 * @return If the plane doesn't intersect the ellispoid, returns null.
	 */
	public Ellipsoid2D intersectionAtZ(int layer) {
		if (Math.abs(z0 - layer) > Math.max(Math.max(a, b), c)) {
			return null;
		}

		// Ellipsoid parameters
		final double[] R = getRotationMatrix();
		final double[] E = { 1 / (a * a), 0, 0, 0, 1 / (b * b), 0, 0, 0,
				1 / (c * c) };
		final double[] M = multiplyMatrices(transposeMatrix(R),
				multiplyMatrices(E, R));

		double den = M[0] * M[4] - M[1] * M[1];
		if (den == 0) {
			return null;
		}

		// Ellipse parameters
		Ellipsoid2D ellipse = new Ellipsoid2D();
		ellipse.x0 = (x0 * M[0] * M[4] - M[1] * M[1] * x0 - M[4] * M[2] * layer
				+ M[4] * M[2] * z0 + M[5] * layer * M[1] - M[5] * z0 * M[1])
				/ den;
		ellipse.y0 = (-M[1] * M[1] * y0 - M[0] * M[5] * layer + M[1] * M[2]
				* layer - M[1] * M[2] * z0 + M[0] * M[5] * z0 + M[0] * M[4]
				* y0)
				/ den;

		ellipse.alpha = Math.atan2(2 * M[1], M[0] - M[4]) / 2;

		double ca = Math.cos(ellipse.alpha);
		double sa = Math.sin(ellipse.alpha);
		double ca2 = ca * ca;
		double sa2 = sa * sa;
		den = ca2 * ca2 - sa2 * sa2;
		double A = 0, B = 0;
		if (den == 0) {
			A = 0.5 * M[0] + M[1];
			B = 0.5 * M[0] - M[1];
		} else {
			A = (M[0] * ca2 - M[4] * sa2) / den;
			B = (M[4] * ca2 - M[0] * sa2) / den;
		}
		if (A <= 0 || B <= 0) {
			return null;
		}

		ellipse.a = 1 / Math.sqrt(A);
		ellipse.b = 1 / Math.sqrt(B);

		return ellipse;
	}

	/** Returns true if the given point is inside the ellipsoid. */
	public boolean contains(double x, double y, double z) {
		return contains(x, y, z, getRotationMatrix());
	}

	/**
	 * Returns true if the give point is insid the ellipsoid.
	 * 
	 * @param R
	 *            Rotation matrix of the ellipse.
	 * */
	public boolean contains(double x, double y, double z, final double R[]) {
		return normalizedPoint(x, y, z, R) <= 1;
	}

	/** Return true if the given point is on the surface of the ellipsoid. */
	public boolean isOver(double x, double y, double z) {
		return Math.abs(1 - normalizedPoint(x, y, z, getRotationMatrix())) < 0.01;
	}

	/**
	 * Return a value indicating the distance between the given point and the
	 * ellipsoid.
	 * 
	 * @return If the value is < 1, the point is inside the ellipsoid. If the
	 *         value is 1, the point is on the surface. Otherwise, the point is
	 *         outside the ellipsoid.
	 */
	private double normalizedPoint(double x, double y, double z,
			final double R[]) {
		double V[] = new double[] { x - x0, y - y0, z - z0 };
		V = applyRotation(R, V);

		V[0] /= a;
		V[1] /= b;
		V[2] /= c;

		return V[0] * V[0] + V[1] * V[1] + V[2] * V[2];
	}
}
