package plugins.adufour.viewers;

import icy.canvas.Canvas2D;
import icy.canvas.IcyCanvas;
import icy.gui.viewer.Viewer;
import icy.image.IcyBufferedImage;
import icy.image.IcyBufferedImageUtil;
import icy.plugin.abstract_.Plugin;
import icy.plugin.interface_.PluginCanvas;
import icy.type.DataType;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * 2D canvas displaying a log-view of the image without changing the actual data. Useful to observe
 * images with large dynamics, e.g. Fourier transforms or 16-bits images
 * 
 * @author Alexandre Dufour
 * 
 */
public class LogCanvas2D extends Plugin implements PluginCanvas
{
    private final static ExecutorService service = Executors.newCachedThreadPool();
    
    @Override
    public String getCanvasClassName()
    {
        return LogCanvas2D.class.getName();
    }
    
    @Override
    public IcyCanvas createCanvas(Viewer viewer)
    {
        return new Canvas2D(viewer)
        {
            private static final long serialVersionUID = 1L;
            
            @Override
            public IcyBufferedImage getCurrentImage()
            {
                final IcyBufferedImage image = super.getCurrentImage();
                
                if (image == null) return null;
                
                final IcyBufferedImage log = (image.getDataType_() == DataType.DOUBLE) ? IcyBufferedImageUtil.getCopy(image) : IcyBufferedImageUtil.convertToType(image, DataType.DOUBLE, false);
                
                final double[][] data_C_XY = log.getDataXYCAsDouble();
                
                final Future<?>[] tasks = new Future[data_C_XY.length];
                
                for (int c = 0; c < data_C_XY.length; c++)
                {
                    final int channel = c;
                    tasks[c] = service.submit(new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            double max = log.getChannelTypeMax(channel);
                            double ratio = max / Math.log1p(max);
                            
                            double[] array = data_C_XY[channel];
                            for (int i = 0; i < array.length; i++)
                                array[i] = Math.log1p(array[i]) * ratio;
                        }
                    });
                }
                
                try
                {
                    for (Future<?> task : tasks)
                        task.get();
                    return log;
                }
                catch (Exception e)
                {
                    return image;
                }
            }
        };
    }
}
