package plugins.adufour.roi;

import icy.canvas.IcyCanvas;
import icy.painter.VtkPainter;
import icy.roi.ROI;
import icy.roi.ROIEvent;
import icy.roi.ROIListener;
import icy.sequence.Sequence;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceListener;
import icy.type.rectangle.Rectangle3D;
import icy.util.XMLUtil;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Iterator;
import java.util.TreeMap;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * Abstract class defining a generic 3D ROI as a stack of individual 2D ROI slices.
 * 
 * @author Alexandre Dufour
 * 
 * @param <R>
 *            the type of 2D ROI for each slice of this 3D ROI
 */
public abstract class ROI3DStack<R extends icy.roi.ROI2D> extends icy.roi.ROI3D implements ROIListener, Iterable<R>
{
    private final TreeMap<Integer, R> slices = new TreeMap<Integer, R>();
    
    protected final String            innerROI2DClassName;
    
    /**
     * Creates a new 3D ROI based on the given 2D ROI type
     * 
     * @param className
     *            the class name corresponding to the inner 2D ROI type
     */
    public ROI3DStack(String className)
    {
        this.innerROI2DClassName = className;
    }
    
    /**
     * Adds the specified slice to this 3D ROI
     * 
     * @param z
     *            the position where the slice must be added
     * @param slice
     *            the 2D ROI to insert
     * @param merge
     *            <code>true</code> if the given slice should be merged with the existing slice, or
     *            <code>false</code> to replace the existing slice
     */
    public abstract void addROI2D(int z, icy.roi.ROI2D slice, boolean merge);
    
    /**
     * Sets the slice for the given z position
     * 
     * @param z
     * @param slice
     */
    protected void setSlice(int z, R slice)
    {
        // remove the current slice (if any)
        R currentSlice = slices.get(z);
        if (currentSlice != null) currentSlice.removeListener(this);
        
        // add the new slice
        slice.beginUpdate();
        
        slice.setZ(z);
        slice.setColor(getColor());
        slice.setSelectedColor(getSelectedColor());
        slice.setStroke(getStroke());
        slice.setC(getC());
        slice.setT(getT());
        
        slice.endUpdate();
        
        slice.addListener(this);
        slices.put(z, slice);
    }
    
    @Override
    public Rectangle3D getBounds3D()
    {
        Rectangle2D xyBounds = slices.firstEntry().getValue().getBounds2D();
        
        int minZ = slices.firstKey();
        int sizeZ = getSizeZ();
        
        for (R slice : slices.subMap(1, slices.size()).values())
            xyBounds = xyBounds.createUnion(slice.getBounds2D());
        
        return new Rectangle3D.Double(xyBounds.getMinX(), xyBounds.getMinY(), minZ, xyBounds.getWidth(), xyBounds.getHeight(), sizeZ);
    }
    
    @Override
    public boolean contains(double x, double y, double z, double t, double c)
    {
        int slice = (int) Math.round(z);
        
        return slices.containsKey(slice) && slices.get(slice).contains(x, y, z, t, c);
    }
    
    @Override
    public void delete()
    {
//        for (R slice : slices.values())
//        {
//            slice.getPainter().remove();
//            slice.remove();
//        }
//        slices.clear();
//        super.remove();
    }
    
    @Override
    public double getPerimeter()
    {
        // perimeter = sum of perimeters over all slices except first and last,
        // for which the volume is added
        
        double perimeter = 0;
        
        if (slices.size() <= 2)
        {
            for (R slice : slices.values())
                perimeter += slice.getVolume();
        }
        else
        {
            
            perimeter = slices.firstEntry().getValue().getVolume();
            
            for (R slice : slices.subMap(1, slices.lastKey()).values())
                perimeter += slice.getPerimeter();
            
            perimeter += slices.lastEntry().getValue().getVolume();
        }
        return perimeter;
        
    }
    
    /**
     * @return The size of this ROI stack along Z. Note that the returned value indicates the
     *         difference between upper and lower bounds of this ROI, but doesn't guarantee that all
     *         slices in-between exist ({@link #getSlice(int, boolean)} may still return
     *         <code>null</code>.
     */
    public int getSizeZ()
    {
        int minZ = slices.firstKey();
        int maxZ = slices.lastKey();
        
        return maxZ - minZ + 1;
    }
    
    @SuppressWarnings("unchecked")
    public R getSlice(int z, boolean createIfNull)
    {
        R slice = slices.get(z);
        
        if (slice == null && createIfNull)
        {
            slice = (R) ROI.create(innerROI2DClassName);
            slice.setZ(z);
            slice.addListener(this);
            slices.put(z, slice);
        }
        
        return slice;
    }
    
    @Override
    public double getVolume()
    {
        double volume = 0;
        
        for (R slice : slices.values())
            volume += slice.getVolume();
        
        return volume;
    }
    
    @Override
    public boolean intersects(double x, double y, double z, double t, double c, double sizeX, double sizeY, double sizeZ, double sizeT, double sizeC)
    {
        // FIXME
        return false;
    }
    
    @Override
    public Iterator<R> iterator()
    {
        return slices.values().iterator();
    }
    
    @Override
    public boolean loadFromXML(Node node)
    {
        if (!super.loadFromXML(node)) return false;
        
        for (Element e : XMLUtil.getElements(node, "slice"))
        {
            @SuppressWarnings("unchecked")
            R slice = (R) ROI.create(innerROI2DClassName);
            
            if (!slice.loadFromXML(e)) return false;
            
            setSlice(slice.getZ(), slice);
        }
        
        return true;
    }
    
    @Override
    public void propertyChanged(String propertyName)
    {
        super.propertyChanged(propertyName);
        
        for (R slice : slices.values())
            slice.propertyChanged(propertyName);
    }
    
    /**
     * Internal listener that monitors events occurring on the inner slices of this 3D ROI
     */
    @Override
    public void roiChanged(ROIEvent event)
    {
        if (!slices.containsValue(event.getSource())) return;
        
        switch (event.getType())
        {
            case SELECTION_CHANGED:
                
                boolean isSelected = event.getSource().isSelected();
                if (isSelected != this.selected) setSelected(isSelected, true);
            
            break;
            
            case FOCUS_CHANGED:
            
            break;
            
            case ROI_CHANGED:
                roiChanged();
            break;
            
            case PROPERTY_CHANGED:
            // this causes stack overflow => is it actually necessary?
            // propertyChanged(event.getPropertyName());
            break;
            
            default:
        }
    }
    
    @Override
    public boolean saveToXML(Node node)
    {
        if (!super.saveToXML(node)) return false;
        
        for (R slice : slices.values())
        {
            Element sliceNode = XMLUtil.addElement(node, "slice");
            
            if (!slice.saveToXML(sliceNode)) return false;
        }
        
        return true;
    }
    
    @Override
    public void selectionChanged()
    {
        super.selectionChanged();
        
        for (R slice : slices.values())
            slice.setSelected(selected, false);
        // NB: the last parameter doesn't matter (the slices aren't attached to the sequence)
    }
    
    @Override
    public void setC(int value)
    {
        super.setC(value);
        
        for (R slice : slices.values())
            slice.setC(value);
    }
    
    @Override
    public void setColor(Color value)
    {
        super.setColor(value);
        
        for (R slice : slices.values())
            slice.setColor(value);
    }
    
    @Override
    public void setT(int value)
    {
        super.setT(value);
        
        for (R slice : slices.values())
            slice.setT(value);
    }
    
    @Override
    public void setSelectedColor(Color value)
    {
        super.setSelectedColor(value);
        
        for (R slice : slices.values())
            slice.setSelectedColor(value);
    }
    
    @Override
    public void setStroke(double value)
    {
        super.setStroke(value);
        
        for (R slice : slices.values())
            slice.setStroke(value);
    }
    
    public abstract class ROI3DStackPainter extends ROIPainter implements VtkPainter, SequenceListener
    {
        
        @Override
        public void mouseMove(MouseEvent e, Point2D imagePoint, IcyCanvas canvas)
        {
            // TODO Auto-generated method stub
            super.mouseMove(e, imagePoint, canvas);
        }
        
        
        @Override
        public void keyReleased(KeyEvent e, Point2D imagePoint, IcyCanvas canvas)
        {
            if (!isActiveFor(canvas)) return;
            
            if (!isReadOnly())
            {
                if (selected) switch (e.getKeyCode())
                {
                    case KeyEvent.VK_BACK_SPACE:
                    case KeyEvent.VK_DELETE:
                        ROI3DStack.this.delete();
                    default:
                        return;
                }
            }
        }
        
        @Override
        public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas)
        {
            int z = canvas.getPositionZ();
            R slice = getSlice(z, false);
            
            if (z >= 0 && slice != null) slice.getPainter().paint(g, sequence, canvas);
//            {
//                ROIPainter painter = slice.getPainter();
//                if (!painter.isAttached(sequence))
//                {
//                    sequence.addPainter(painter);
//                    sequence.addListener(this);
//                }
//            }
        }
        
        @Override
        public void sequenceChanged(SequenceEvent sequenceEvent)
        {
//            if (sequenceEvent.getSourceType() == SequenceEventSourceType.SEQUENCE_ROI && sequenceEvent.getType() == SequenceEventType.REMOVED && sequenceEvent.getSource() == ROI3DStack.this)
//            {
//                sequenceEvent.getSequence().removeListener(this);
//                ROI3DStack.this.delete();
//            }
        }
        
        @Override
        public void sequenceClosed(Sequence sequence)
        {
            //sequence.removeListener(this);
        }
    }
}
