package plugins.tprovoost.Microscopy.MicroManager;

import icy.gui.dialog.MessageDialog;
import icy.gui.frame.progress.FailedAnnounceFrame;
import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;
import icy.system.IcyExceptionHandler;
import icy.system.thread.ThreadUtil;
import icy.util.ClassUtil;
import icy.util.ReflectionUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.prefs.Preferences;

import mmcorej.CMMCore;
import mmcorej.TaggedImage;

import org.json.JSONObject;
import org.micromanager.MMOptions;
import org.micromanager.MMStudioMainFrame;
import org.micromanager.acquisition.AcquisitionEngine;
import org.micromanager.acquisition.AcquisitionWrapperEngine;
import org.micromanager.acquisition.TaggedImageQueue;
import org.micromanager.api.IAcquisitionEngine2010;
import org.micromanager.api.SequenceSettings;
import org.micromanager.api.TaggedImageAnalyzer;
import org.micromanager.utils.MDUtils;
import org.micromanager.utils.ReportingUtils;

import plugins.tprovoost.Microscopy.MicroManager.core.AcquisitionResult;
import plugins.tprovoost.Microscopy.MicroManager.event.AcquisitionListener;
import plugins.tprovoost.Microscopy.MicroManager.event.LiveListener;
import plugins.tprovoost.Microscopy.MicroManager.gui.LoadFrame;
import plugins.tprovoost.Microscopy.MicroManager.gui.LoadingFrame;
import plugins.tprovoost.Microscopy.MicroManager.gui.MMMainFrame;
import plugins.tprovoost.Microscopy.MicroManager.tools.MMUtils;
import plugins.tprovoost.Microscopy.MicroManager.tools.StageMover;
import plugins.tprovoost.Microscopy.MicroManagerForIcy.MicromanagerPlugin;
import plugins.tprovoost.Microscopy.MicroManagerForIcy.MicroscopePlugin;

public class MicroManager
{
    /**
     * Metadata associated to last retrieved image.
     */
    private static final Map<Integer, JSONObject> metadatas = new HashMap<Integer, JSONObject>(4);

    static final List<AcquisitionListener> acqListeners = new ArrayList<AcquisitionListener>();
    static final List<LiveListener> liveListeners = new ArrayList<LiveListener>();

    static AcquisitionResult sequenceManager = null;
    static MMMainFrame instance = null;
    static Thread liveManager = null;
    static MMOptions options = null;
    static volatile boolean initializing = false;

    /**
     * Retrieve the MicroManager main frame instance.
     */
    public static MMMainFrame getInstance()
    {
        // initialize micro manager if needed
        if (instance == null)
            MicromanagerPlugin.init();

        return instance;
    }

    /**
     * @return the acquisition listener list.
     */
    public static List<AcquisitionListener> getAcquisitionListeners()
    {
        final List<AcquisitionListener> result;

        // create safe copy
        synchronized (acqListeners)
        {
            result = new ArrayList<AcquisitionListener>(acqListeners);
        }

        return result;
    };

    /**
     * Every {@link MicroscopePlugin} shares the same core, so you will receive every acquisition
     * even if it was not asked by you.<br/>
     * You should register your listener only when you need image and remove it when your done.
     * 
     * @param listener
     *        Your listener
     */
    public static void addAcquisitionListener(AcquisitionListener listener)
    {
        synchronized (acqListeners)
        {
            if (!acqListeners.contains(listener))
                acqListeners.add(listener);
        }
    }

    public static void removeAcquisitionListener(AcquisitionListener listener)
    {
        synchronized (acqListeners)
        {
            acqListeners.remove(listener);
        }
    }

    /**
     * @return the live listener list.
     */
    public static List<LiveListener> getLiveListeners()
    {
        final List<LiveListener> result;

        // create safe copy
        synchronized (liveListeners)
        {
            result = new ArrayList<LiveListener>(liveListeners);
        }

        return result;
    }

    /**
     * Every {@link MicroscopePlugin} shares the same core, so the same live,<br/>
     * when your plugin start, a live may have been already started and your {@link LiveListener} <br/>
     * could receive images at the moment you call this method. <br />
     * You should register your listener only when you need image and remove it when your done.
     */
    public static void addLiveListener(LiveListener listener)
    {
        synchronized (liveListeners)
        {
            if (!liveListeners.contains(listener))
                liveListeners.add(listener);
        }
    }

    public static void removeLiveListener(LiveListener listener)
    {
        synchronized (liveListeners)
        {
            liveListeners.remove(listener);
        }
    }

    /**
     * @deprecated Use {@link #addAcquisitionListener(AcquisitionListener)} instead.
     */
    @Deprecated
    public static void registerListener(AcquisitionListener listener)
    {
        addAcquisitionListener(listener);
    }

    /**
     * @deprecated Use {@link #removeAcquisitionListener(AcquisitionListener)} instead.
     */
    @Deprecated
    public static void removeListener(AcquisitionListener listener)
    {
        removeAcquisitionListener(listener);
    }

    /**
     * @deprecated Use {@link #addLiveListener(LiveListener)} instead.
     */
    @Deprecated
    public static void registerListener(LiveListener listener)
    {
        addLiveListener(listener);
    }

    /**
     * @deprecated Use {@link #removeLiveListener(LiveListener)} instead.
     */
    @Deprecated
    public static void removeListener(LiveListener listener)
    {
        removeLiveListener(listener);
    }

    /**
     * Use this to access micro-manager main GUI and to be able to use function that
     * are not been implemented in Icy's micro-Manager.
     * 
     * @return The micro-manager main frame (hidden in the Icy GUI).
     */
    public static MMStudioMainFrame getMicroManagerMainInterface()
    {
        final MMMainFrame inst = getInstance();

        if (inst != null)
            return inst.MMFRAME;

        return null;
    }

    /**
     * Use this to access micro-manager core and to be able to use function that
     * are not been implemented in Micro-Manger for Icy. </br>
     * </br>
     * Be careful, if you use core functions instead of the integrated {@link MicroManager} methods
     * or utility class as {@link StageMover} you will have to handle synchronization of the
     * core and catch all exception. </br>
     * In most of case you don't need to use it, but only if you want to make fancy things.
     */
    public static CMMCore getCore()
    {
        final MMStudioMainFrame mainInterface = getMicroManagerMainInterface();

        if (mainInterface != null)
            return mainInterface.getCore();

        return null;
    }

    /**
     * Get exclusive access to micro manager.</br>
     * 
     * @param wait
     *        number of milli second to wait to retrieve exclusive access if micro manager is
     *        already locked by another thread. If set to 0 then it returns immediately if already
     *        locked.
     * @return <code>true</code> if you obtained exclusive access and <code>false</code> if micro
     *         manager is already locked by another thread and wait time elapsed.
     * @throws InterruptedException
     * @see #lock()
     * @see #unlock()
     */
    public static boolean lock(long wait) throws InterruptedException
    {
        return getInstance().lock(wait);
    }

    /**
     * Get exclusive access to micro manager.</br>
     * If another thread already has exclusive access then it will wait until it release it.
     * 
     * @see #lock(long)
     * @see #unlock()
     */
    public static void lock()
    {
        getInstance().lock();
    }

    /**
     * Release exclusive access to micro manager.
     * 
     * @see #lock()
     * @see #lock(long)
     */
    public static void unlock()
    {
        getInstance().unlock();
    }

    /**
     * @return The acquisition engine wrapper from MicroManager.
     */
    public static AcquisitionWrapperEngine getAcquisitionEngine()
    {
        return getMicroManagerMainInterface().getAcquisitionEngine();
    }

    /**
     * @return The internal new acquisition engine from MicroManager.
     */
    public static IAcquisitionEngine2010 getAcquisitionEngine2010()
    {
        return getMicroManagerMainInterface().getAcquisitionEngine2010();
    }

    /**
     * @return The engine settings for the most recently started acquisition sequence, or return
     *         null if you never started an acquisition.
     */
    public static SequenceSettings getAcquisitionSettings()
    {
        return getAcquisitionEngine().getSequenceSettings();
    }

    /**
     * @return The summaryMetadata for the most recently started acquisition sequence, or return
     *         null if you never started an acquisition.
     */
    public static JSONObject getAcquisitionMetaData()
    {
        return getAcquisitionEngine().getSummaryMetadata();
    }

    /**
     * @return Returns the number of channel of the camera device (usually 3 for color camera and 1
     *         in other case).</br>
     *         Just a shortcut for <code>getCore().getNumberOfCameraChannels()</code>
     */
    public static long getCameraChannelCount()
    {
        final CMMCore core = MicroManager.getCore();

        if (core == null)
            return 0L;

        return core.getNumberOfCameraChannels();
    }

    /**
     * Returns the metadata object associated to the last image retrieved with {@link #getLastImage()} or
     * {@link #snapImage()}.</br>
     * Returns <code>null</code> if there is no metadata associated to the specified channel.
     * 
     * @param channel
     *        channel index for multi channel camera
     * @see #getLastImage()
     * @see #snapImage()
     * @see #getMetadata()
     * @see MDUtils
     */
    public static JSONObject getMetadata(int channel)
    {
        return metadatas.get(Integer.valueOf(channel));
    }

    /**
     * Returns the metadata object associated to the last image retrieved with {@link #getLastImage()} or
     * {@link #snapImage()}.</br>
     * Returns <code>null</code> if there is no image has been retrieved yet.
     * 
     * @see #getLastImage()
     * @see #snapImage()
     * @see #getMetadata(int)
     * @see MDUtils
     */
    public static JSONObject getMetadata()
    {
        return getMetadata(0);
    }

    /**
     * Returns the last image captured from the micro manager continuous acquisition.</br>
     * Returns <code>null</code> if the continuous acquisition is not running.</br>
     * </br>
     * You can listen new image event from acquisition or live mode by using these methods:</br>
     * {@link #addLiveListener(LiveListener)}</br> {@link #addAcquisitionListener(AcquisitionListener)}</br>
     * </br>
     * You can retrieve the associated metadata by using {@link #getMetadata(int)} method.
     * 
     * @throws Exception
     */
    public static IcyBufferedImage getLastImage() throws Exception
    {
        return MMUtils.convertToIcyImage(getLastTaggedImage());
    }

    /**
     * Returns a list of {@link TaggedImage} representing the last image captured from the micro
     * manager continuous acquisition. The list contains as many image than camera channel (see
     * {@link #getCameraChannelCount()}).</br>
     * Returns an empty list if the continuous acquisition is not running.</br>
     * </br>
     * You can listen new image event from acquisition or live mode by using these methods:</br>
     * {@link #addLiveListener(LiveListener)}</br> {@link #addAcquisitionListener(AcquisitionListener)}</br>
     * 
     * @throws Exception
     */
    public static List<TaggedImage> getLastTaggedImage() throws Exception
    {
        final CMMCore core = MicroManager.getCore();

        if (core == null || !core.isSequenceRunning())
            return new ArrayList<TaggedImage>();

        final long numChannel = getCameraChannelCount();
        final List<TaggedImage> result = new ArrayList<TaggedImage>();

        lock();
        try
        {
            for (int c = 0; c < numChannel; c++)
            {
                final TaggedImage img = core.getLastTaggedImage(c);

                if (img != null)
                {
                    result.add(img);
                    // store metadata
                    metadatas.put(Integer.valueOf(c), img.tags);
                }
            }
        }
        finally
        {
            unlock();
        }

        return result;
    }

    /**
     * Capture and return an image (or <code>null</code> if an error occurred).</br>
     * If an acquisition is currently in process (live mode or standard acquisition) the method will
     * return the last image from the image acquisition buffer.<br/>
     * In other case it just snaps a new image from the camera device and returns it.</br>
     * You can retrieve the associated metadata by using {@link #getMetadata(int)} method.
     * 
     * @throws Exception
     */
    public static IcyBufferedImage snapImage() throws Exception
    {
        return MMUtils.convertToIcyImage(snapTaggedImage());
    }

    /**
     * Capture and return an image (or an empty list if an error occurred).</br>
     * If an acquisition is currently in process (live mode or standard acquisition) the method will
     * return the last image from the image acquisition buffer.<br/>
     * In other case it just snaps a new image from the camera device and returns it.</br>
     * This function return a list of image as the camera device can have several channel (see
     * {@link #getCameraChannelCount()} in which case we have one image per channel.</br>
     * </br>
     * You can retrieve the associated metadata by using {@link #getMetadata(int)} method.
     * 
     * @throws Exception
     */
    public static List<TaggedImage> snapTaggedImage() throws Exception
    {
        final CMMCore core = MicroManager.getCore();

        if (core == null)
            return new ArrayList<TaggedImage>();

        // continuous acquisition --> retrieve next image acquired
        if (core.isSequenceRunning())
        {
            // wait for exposure as we are snapping
            ThreadUtil.sleep((long) core.getExposure());
            return getLastTaggedImage();
        }

        final long numChannel = getCameraChannelCount();
        final List<TaggedImage> result = new ArrayList<TaggedImage>();

        lock();
        try
        {
            // wait for camera to be ready
            core.waitForDevice(core.getCameraDevice());
            // manual snap
            core.snapImage();

            // get result
            for (int c = 0; c < numChannel; c++)
            {
                final TaggedImage img = core.getTaggedImage(c);

                if (img != null)
                {
                    result.add(img);
                    // store metadata
                    metadatas.put(Integer.valueOf(c), img.tags);
                }
            }
        }
        finally
        {
            unlock();
        }

        return result;
    }

    /**
     * Use this method to know if the continuous acquisition (live mode) is running.
     */
    public static boolean isLiveRunning()
    {
        return getCore().isSequenceRunning();
    }

    /**
     * Use this method to start the continuous acquisition mode (live mode) and retrieve images with
     * {@link LiveListener}.</br>
     * This command does not block the calling thread for the duration of the acquisition.
     * The GUI will draw the indeterminate progress bar representing a live acquisition running,
     * in the Running Acquisition tab panel.<br/>
     * <b>There is only one "live" progress bar for all plugin<b/>.<br/>
     * 
     * @throws Exception
     *         If a sequence acquisition is running (see {@link #startAcquisition(int, double, boolean)}) or if the core
     *         doesn't
     *         respond.
     * @see #stopLiveMode()
     * @see #addLiveListener(LiveListener)
     * @see #isLiveRunning()
     * @return <code>true</code> if the method actually started the live mode and <code>false</code> if it was already
     *         running.
     */
    public static boolean startLiveMode() throws Exception
    {
        if (!isLiveRunning())
        {
            lock();
            try
            {
                // start continuous acquisition (need to clear circular buffer first)
                getCore().clearCircularBuffer();
                getCore().startContinuousSequenceAcquisition(0d);
            }
            finally
            {
                unlock();
            }

            // notify about it
            instance.notifyLiveRunning();
            for (LiveListener l : getLiveListeners())
                l.liveStarted();

            return true;
        }

        return false;
    }

    /**
     * Use this method to stop the continuous acquisition mode (live mode).<br/>
     * The GUI will remove indeterminate progress bar representing a live acquisition running,
     * in the Running Acquisition tab panel, if live have been stopped. <br/>
     * 
     * @throws Exception
     *         If the core can't stop the continuous acquisition mode.
     */
    public static void stopLiveMode() throws Exception
    {
        if (isLiveRunning())
        {
            lock();
            try
            {
                // stop continuous acquisition
                getCore().stopSequenceAcquisition();
            }
            finally
            {
                unlock();
            }

            // notify about it
            instance.notifyLiveStopped();
            for (LiveListener l : getLiveListeners())
                l.liveStopped();
        }
    }

    /**
     * Returns <code>true</code> if Micro-Manager is initialized / loaded.<br>
     * 
     * @see #isInitializing()
     * @see #init()
     */
    public static boolean isInitialized()
    {
        return instance != null;
    }

    /**
     * Returns <code>true</code> if currently initializing / loading the Micro-Manager library
     * 
     * @see #isInitialized()
     * @see #init()
     */
    public static boolean isInitializing()
    {
        return initializing;
    }

    /**
     * Use this method to know if an acquisition is currently in process.
     * 
     * @see #startAcquisition(int, double, boolean)
     * @see #stopAcquisition()
     */
    public static boolean isAcquisitionRunning()
    {
        final AcquisitionEngine eng = getAcquisitionEngine();
        return (eng != null) && eng.isAcquisitionRunning();
    }

    /**
     * Use this method to start an acquisition on the current camera device and retrieve images with
     * {@link AcquisitionListener}.</br>
     * This command does not block the calling thread for the duration of the acquisition.</br>
     * Note that you have to stop the live mode (see {@link #stopLiveMode()}) before calling this
     * method.
     * 
     * @param numImages
     *        Number of images requested from the camera
     * @param intervalMs
     *        The interval between images
     * @param pauseLive
     *        Whether or not the live is paused while acquisition is running, <b> {@link LiveListener} will not been
     *        notify by this pause.</b>
     * @throws Exception
     *         if live is running or if a sequence acquisition have been started and is not finished
     *         yet.
     * @see #isAcquisitionRunning()
     * @see #stopAcquisition()
     * @see StageMover#moveZAbsolute(double)
     */
    public static void startAcquisition(int numImages, double intervalMs, boolean pauseLive) throws Exception
    {
        /*
         * stopOnOverflow is manually set to false, because we don't care if the circular buffer
         * is rewritten, because we capture each frame at the moment they have been put on the
         * buffer and we don't need them anymore.
         */
        getCore().startSequenceAcquisition(numImages, intervalMs, false);
    }

    /**
     * Use this method to interruption the current acquisition.
     * 
     * @see #isAcquisitionRunning()
     * @see #startAcquisition(int, double, boolean)
     */
    public static void stopAcquisition()
    {
        if (isAcquisitionRunning())
        {
            final AcquisitionEngine eng = getAcquisitionEngine();

            if (eng != null)
                eng.stop(true);
        }
    }

    /**
     * @return The list of sequence corresponding to the last sequence acquisition or <code>null</code> if no
     *         acquisition was done.</br>
     *         Note you can have severals sequence for acquisition using different XY position.
     * @see #startAcquisition(int, double, boolean)
     */
    public static List<Sequence> getAcquisitionResult()
    {
        if (sequenceManager == null)
            return null;

        return sequenceManager.getSequences();
    }

    /**
     * Set the exposure of the camera
     * 
     * @param exposure
     * @throws Exception
     */
    public static void setExposure(double exposure) throws Exception
    {
        lock();
        try
        {
            // stop acquisition if needed
            if (isAcquisitionRunning())
                stopAcquisition();

            // save continuous acquisition state
            final boolean liveRunning = isLiveRunning();
            // stop live
            if (liveRunning)
                stopLiveMode();

            // exposure changed ?
            if (getCore().getExposure() != exposure)
                getCore().setExposure(exposure);

            // restore continuous acquisition
            if (liveRunning)
                startLiveMode();
        }
        finally
        {
            unlock();
        }
    }

    /**
     * Use this method to enable logging (disabled by default).<br>
     * Log file are in Icy folder and are named this way : CoreLog20140515.txt ; 20140515 is the
     * date of debugging (2014/05/15)<br/>
     * <b>Be careful, log files can be very huge and take easily 1Gb.</b>
     * 
     * @param enable
     */
    public static void enableDebugLogging(boolean enable)
    {
        getCore().enableDebugLog(enable);
        getCore().enableStderrLog(enable);
    }

    /**
     * For internal use only (this method should never be called directly).
     */
    public static synchronized void init()
    {
        // not initialized ?
        if (instance == null)
        {
            // start initialization
            initializing = true;

            try
            {
                try
                {
                    // force to load MMStudioMainFrame class
                    ClassUtil.findClass(MMStudioMainFrame.class.getName());
                }
                catch (Throwable t)
                {
                    // cannot load class --> version mismatch probably
                    MessageDialog
                            .showDialog(
                                    "Cannot load Micro-Manager",
                                    "Your version of Micro-Manager seems to not be compatible !\n"
                                            + "This plugin is only compatible with version 1.4.16, 1.4.17 or 1.4.18\n"
                                            + "Also check that you are using the same architecture for Icy and Micro-Manager (32/64 bits)\n"
                                            + "You need to restart Icy to change the defined Micro-Manager folder.",
                                    MessageDialog.ERROR_MESSAGE);
                    // so user can change the defined MM folder
                    MMUtils.resetLibrayPath();
                    return;
                }

                final AtomicBoolean configLoaded = new AtomicBoolean(true);
                Preferences root = Preferences.userNodeForPackage(MMOptions.class);
                Preferences prefs = root.node(root.absolutePath() + "/" + "MMOptions");
                prefs.putBoolean("SkipSplashScreen", true);

                ThreadUtil.invokeNow(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        // config not loaded
                        configLoaded.set(false);

                        final LoadFrame f = new LoadFrame();
                        final int res = f.showDialog();

                        switch (res)
                        {
                        // no error ?
                            case 0:
                                System.setProperty("org.micromanager.default.config.file", f.getConfigFilePath());
                                Preferences.userNodeForPackage(MMStudioMainFrame.class).put("sysconfig_file",
                                        f.getConfigFilePath());
                                configLoaded.set(true);
                                break;

                            // cancel
                            case 1:
                                break;

                            // error
                            default:
                                new FailedAnnounceFrame(
                                        "Error while loading configuration file, please restart Micro-Manager.", 2);
                                break;
                        }
                    }
                });

                // cannot load config --> exit
                if (!configLoaded.get())
                    return;

                final LoadingFrame loadingFrame = new LoadingFrame(
                        "Please wait while loading Micro-Manager, Icy interface may not respond...");

                // show loading message
                loadingFrame.show();
                try
                {
                    try
                    {
                        // create main frame (here we are initialized)
                        instance = new MMMainFrame();
                    }
                    catch (Exception e)
                    {
                        IcyExceptionHandler.showErrorMessage(e, true, true);

                        MessageDialog
                                .showDialog(
                                        "Cannot load Micro-Manager",
                                        "Your version of Micro-Manager is probably not compatible !\n"
                                                + "This plugin is only compatible with version 1.4.16, 1.4.17 or 1.4.18\n"
                                                + "Also check that you are using the same architecture for Icy and Micro-Manager (32/64 bits).\n"
                                                + "You need to restart Icy to change the defined Micro-Manager folder.",
                                        MessageDialog.ERROR_MESSAGE);
                        // so user can change the defined MM folder
                        MMUtils.resetLibrayPath();
                        return;
                    }

                    // get the MM main frame
                    final MMStudioMainFrame mainFrame = instance.MMFRAME;
                    // get the MM core
                    final CMMCore core = mainFrame.getCore();

                    // load StageMover preferences
                    StageMover.loadPreferences(instance.pluginPreferences.node("StageMover"));
                    // set core for reporting
                    ReportingUtils.setCore(core);

                    try
                    {
                        // initialize circular buffer
                        core.initializeCircularBuffer();
                    }
                    catch (Exception e)
                    {
                        throw new Exception("Error while initializing circular buffer of Micro Manager", e);
                    }

                    try
                    {
                        options = (MMOptions) ReflectionUtil.getFieldObject(mainFrame, "options_", true);

                        // Ugly MMOption patch to hide imageJ display
                        if (options != null)
                            options.hideMDADisplay_ = true;
                    }
                    catch (Exception e)
                    {
                        options = null;
                        System.err
                                .println("Warning: cannot retrieve 'options_' field from the Micro Manager interface.");
                    }

                    ThreadUtil.invokeNow(new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            // In AWT Thread because it create a JComponent
                            mainFrame.getAcquisitionEngine().addImageProcessor(new ImageAnalyser());
                        }
                    });

                    // live event thread
                    liveManager = new LiveListenerThread();
                    liveManager.start();

                    // separate initialization which may require getInstance()
                    instance.init();
                    // this happen when micro manager cannot correctly load config file
                    if (!instance.isVisible())
                    {
                        new FailedAnnounceFrame(
                                "Could not initialize Micro-Manager, you can try to restart Icy to fix the issue.");
                        shutdown();
                        return;
                    }
                }
                catch (Exception e)
                {
                    IcyExceptionHandler.showErrorMessage(e, true, true);
                    new FailedAnnounceFrame(e.getMessage() + " You can try to restart Icy to fix the issue.");
                    shutdown();
                    return;
                }
                finally
                {
                    loadingFrame.hide();
                }
            }
            finally
            {
                initializing = false;
            }
        }

        // put it on front
        instance.toFront();
    }

    public static void shutdown()
    {
        if (liveListeners != null)
            liveListeners.clear();
        if (acqListeners != null)
            acqListeners.clear();
        StageMover.clearListener();
        if (metadatas != null)
            metadatas.clear();

        // stop live listener
        if (liveManager != null)
        {
            liveManager.interrupt();

            try
            {
                // wait until thread ended
                liveManager.join();
            }
            catch (InterruptedException e)
            {
                // ignore
            }

            liveManager = null;
        }

        if (instance != null)
            instance.close();

        instance = null;
        sequenceManager = null;
        options = null;
    }

    // custom TaggedImageAnalyzer so we have events for new image
    private static class ImageAnalyser extends TaggedImageAnalyzer
    {
        ImageAnalyser()
        {
            super();
        }

        @Override
        protected void analyze(TaggedImage image)
        {
            final List<AcquisitionListener> listeners = getAcquisitionListeners();

            // no more image or last one ?
            if ((image == null) || TaggedImageQueue.isPoison(image))
            {
                if (sequenceManager != null)
                    sequenceManager.done();

                // send acquisition ended event
                for (AcquisitionListener l : listeners)
                    l.acquisitionFinished(getAcquisitionResult());

                // done
                return;
            }

            final JSONObject tags = image.tags;

            try
            {
                boolean firstImage = (MDUtils.getPositionIndex(tags) == 0) && (MDUtils.getFrameIndex(tags) == 0)
                        && (MDUtils.getChannelIndex(tags) == 0) && (MDUtils.getSliceIndex(tags) == 0);
                boolean newAcquisition = (sequenceManager == null) || sequenceManager.isDone();

                // first acquisition image or new acquisition --> create the new acquisition
                if (firstImage || newAcquisition)
                {
                    final SequenceSettings settings = getAcquisitionSettings();
                    final JSONObject metadata = getAcquisitionMetaData();

                    // get the sequence manager
                    sequenceManager = new AcquisitionResult(settings, metadata, false);

                    // send acquisition started event
                    for (AcquisitionListener l : listeners)
                        l.acquisitionStarted(settings, metadata);
                }

                // image received
                sequenceManager.imageReceived(image);

                // send image received event
                for (AcquisitionListener l : listeners)
                    l.acqImgReveived(image);
            }
            catch (Exception e)
            {
                IcyExceptionHandler.showErrorMessage(e, true);
            }
        }
    }

    private static class LiveListenerThread extends Thread
    {
        public LiveListenerThread()
        {
            super("uManager - LiveListener");
        }

        @Override
        public void run()
        {
            while (!isInterrupted())
            {
                final CMMCore core = getCore();

                // running and we have a new image in the queue ?
                while (core.isSequenceRunning() && (core.getRemainingImageCount() > 0))
                {
                    try
                    {
                        final IcyBufferedImage image;

                        lock();
                        try
                        {
                            // retrieve the last image
                            image = getLastImage();
                            // not a poison image --> remove it from the queue
                            if (image != null)
                            {
                                try
                                {
                                    // acquisition may consume the image in the mean time
                                    if (!isAcquisitionRunning() && core.getRemainingImageCount() > 0)
                                        core.popNextImage();
                                }
                                catch (Exception e)
                                {
                                    // can happen with advanced acquisition set with a lower time interval than current
                                    // exposure time
                                    IcyExceptionHandler.showErrorMessage(e, true);
                                }
                            }
                        }
                        finally
                        {
                            unlock();
                        }

                        if (image != null)
                        {
                            // send image received event
                            for (LiveListener l : getLiveListeners())
                                l.liveImgReceived(image);
                        }
                    }
                    catch (Exception e)
                    {
                        // should not happen
                        IcyExceptionHandler.showErrorMessage(e, true);
                    }
                }

                // sleep a bit to free some CPU time
                ThreadUtil.sleep(1);
            }
        }
    }
}
