package plugins.tprovoost.Microscopy.MicroscopeAdvancedAcquisition;

import icy.gui.frame.IcyFrame;
import icy.gui.frame.IcyFrameAdapter;
import icy.gui.frame.IcyFrameEvent;
import icy.gui.frame.progress.AnnounceFrame;
import icy.main.Icy;
import icy.sequence.Sequence;
import icy.system.IcyExceptionHandler;
import icy.system.thread.ThreadUtil;
import icy.util.ReflectionUtil;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;
import java.util.prefs.Preferences;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.plaf.basic.BasicProgressBarUI;

import mmcorej.TaggedImage;

import org.json.JSONException;
import org.json.JSONObject;
import org.micromanager.AcqControlDlg;
import org.micromanager.MMOptions;
import org.micromanager.MMStudioMainFrame;
import org.micromanager.api.SequenceSettings;
import org.micromanager.utils.MDUtils;

import plugins.tprovoost.Microscopy.MicroManager.MicroManager;
import plugins.tprovoost.Microscopy.MicroManager.event.AcquisitionListener;
import plugins.tprovoost.Microscopy.MicroManager.tools.FrameUtils;

/**
 * Wrapper for org.micromanager.AcqControlDlg
 * 
 * @author Irsath Nguyen
 */
public class MicroscopeAdvancedAcquisition implements AcquisitionListener
{
    MicroscopeAdvancedAcquisitionPlugin plugin;
    IcyFrame frame;

    // extra GUI
    JButton startButton;
    JButton stopButton;
    JLabel label;
    JProgressBar progressBar;

    int index;
    boolean aborted;
    boolean liveModeWasRunning;

    public MicroscopeAdvancedAcquisition(MicroscopeAdvancedAcquisitionPlugin plugin)
    {
        super();

        this.plugin = plugin;
        aborted = false;
        liveModeWasRunning = false;

        init();

        MicroManager.addAcquisitionListener(MicroscopeAdvancedAcquisition.this);
    }

    private void init()
    {
        MMOptions options;

        try
        {
            options = (MMOptions) ReflectionUtil.getFieldObject(MicroManager.getMicroManagerMainInterface(),
                    "options_", true);
        }
        catch (Exception e)
        {
            options = null;
            System.err.println("Cannot retrieve 'options_' field from the Micro Manager interface.");
        }

        final AcqControlDlg dialog = new AcqControlDlg(MicroManager.getAcquisitionEngine(),
                Preferences.userNodeForPackage(MMStudioMainFrame.class), MicroManager.getMicroManagerMainInterface(),
                options);

        frame = FrameUtils.addMMWindowToDesktopPane(dialog);

        progressBar = new JProgressBar();
        progressBar.setMinimum(0);
        progressBar.setMaximum(100);
        progressBar.setUI(new BasicProgressBarUI()
        {
            @Override
            protected void paintDeterminate(Graphics g, JComponent c)
            {
                g.setColor(Color.DARK_GRAY);
                super.paintDeterminate(g, c);
            }
        });
        progressBar.setSize(450, 30);
        JPanel pan = new JPanel();
        label = new JLabel("No acquisition started : ");
        pan.add(label);
        pan.add(progressBar);
        frame.add(pan, BorderLayout.SOUTH);

        // We are forced to put manually the size because of the way the frame is
        // created by Micro-Manager, original size (521,690)
        frame.getIcyInternalFrame().setSize(524, 716);
        frame.getIcyExternalFrame().setSize(524, 716);

        frame.addFrameListener(new IcyFrameAdapter()
        {
            @Override
            public void icyFrameClosing(IcyFrameEvent e)
            {
                super.icyFrameClosing(e);

                plugin.shutdown();
            }
        });

        // find originals components (take from dialog as frame has a sub panel)
        startButton = findButtonComponents(dialog.getContentPane(), "Acquire!");
        stopButton = findButtonComponents(dialog.getContentPane(), "Stop");

        // found the original start button
        if (startButton != null)
        {
            // get original listeners
            final ActionListener[] listeners = startButton.getActionListeners();

            // remove them
            for (ActionListener l : listeners)
                startButton.removeActionListener(l);

            // add our own listener on start action
            startButton.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(final ActionEvent arg0)
                {
                    try
                    {
                        synchronized (MicroManager.getCore())
                        {
                            liveModeWasRunning = MicroManager.isLiveRunning();

                            if (liveModeWasRunning)
                            {
                                new AnnounceFrame("Micro-Manager live stopped by Advanced Acquisition", 10);
                                MicroManager.stopLiveMode();
                            }

                            // forward originals events
                            for (ActionListener l : listeners)
                                l.actionPerformed(arg0);
                        }
                    }
                    catch (Exception e)
                    {
                        IcyExceptionHandler.showErrorMessage(e, false);
                    }
                }
            });
        }

        if (stopButton != null)
        {
            // add our own listener on stop action
            stopButton.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(final ActionEvent arg0)
                {
                    aborted = true;
                }
            });
        }

    }

    static JButton findButtonComponents(Container container, String label)
    {
        for (Component c : container.getComponents())
        {
            // // panel component
            // if (c instanceof JPanel)
            // {
            // final JPanel panel = (JPanel) c;
            //
            // // we want to remove a specific panel
            // if (panel.getBorder() instanceof ComponentTitledBorder)
            // {
            // final Object obj = ReflectionUtil.getFieldObject(panel.getBorder(),
            // "comp", true);
            //
            // // "save image" panel ? --> remove it
            // if ((obj instanceof JCheckBox) && (((JCheckBox)
            // obj).getText().equals("Save images")))
            // dialog.getContentPane().remove(panel);
            // }
            // }

            if (c instanceof JButton)
            {
                final JButton button = (JButton) c;

                if (button.getText().equalsIgnoreCase(label))
                    return button;
            }
        }

        return null;
    }

    public void shutdown()
    {
        aborted = true;
        try
        {
            // stop the acquisition
            MicroManager.stopAcquisition();
        }
        catch (Exception e)
        {
            // may fail if MM core shutdown at same time
        }
        // remove listener
        MicroManager.removeAcquisitionListener(MicroscopeAdvancedAcquisition.this);

        frame.dispose();
    }

    public void toFront()
    {
        frame.toFront();
    }

    @Override
    public void acquisitionStarted(SequenceSettings settings, JSONObject metadata)
    {
        try
        {
            final int numImage = MDUtils.getNumPositions(metadata) * MDUtils.getNumFrames(metadata)
                    * MDUtils.getNumSlices(metadata) * MDUtils.getNumChannels(metadata);
            index = 0;

            ThreadUtil.invokeNow(new Runnable()
            {
                @Override
                public void run()
                {
                    label.setText("Acquisition running : ");

                    progressBar.setMinimum(0);
                    progressBar.setMaximum(numImage);

                    if (startButton != null)
                        startButton.setEnabled(false);
                }
            });
        }
        catch (JSONException e)
        {
            IcyExceptionHandler.showErrorMessage(e, true);
        }
    }

    @Override
    public void acqImgReveived(TaggedImage newImg)
    {
        ThreadUtil.invokeNow(new Runnable()
        {
            @Override
            public void run()
            {
                progressBar.setValue(index++);
            }
        });
    }

    @Override
    public void acquisitionFinished(List<Sequence> result)
    {
        if (!aborted)
        {
            for (Sequence s : result)
            {
                if (!Icy.getMainInterface().isOpened(s))
                    plugin.addSequence(s);
            }
        }

        aborted = false;

        ThreadUtil.invokeNow(new Runnable()
        {
            @Override
            public void run()
            {
                progressBar.setValue(0);
                label.setText("Acquisition completed : ");
                if (startButton != null)
                    startButton.setEnabled(true);
            }
        });

        // restore live mode if needed
        if (liveModeWasRunning)
        {
            try
            {
                MicroManager.startLiveMode();
            }
            catch (Exception e)
            {
                IcyExceptionHandler.showErrorMessage(e, true);
            }

            liveModeWasRunning = false;
        }
    }
}