package plugins.tprovoost.Microscopy.MicroManager.core;

import icy.image.IcyBufferedImage;
import icy.image.colormap.IcyColorMap;
import icy.main.Icy;
import icy.sequence.Sequence;
import icy.type.DataType;
import icy.type.point.Point3D;
import icy.util.OMEUtil;
import icy.util.StringUtil;

import java.awt.Color;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import loci.formats.ome.OMEXMLMetadataImpl;
import mmcorej.TaggedImage;
import ome.xml.model.primitives.NonNegativeInteger;

import org.json.JSONException;
import org.json.JSONObject;
import org.micromanager.api.SequenceSettings;
import org.micromanager.utils.ChannelSpec;
import org.micromanager.utils.MDUtils;
import org.micromanager.utils.MMScriptException;

import plugins.tprovoost.Microscopy.MicroManager.MicroManager;
import plugins.tprovoost.Microscopy.MicroManager.tools.StageMover;

/**
 * This class is to be used by AcquisitionListener.</br>
 * It initializes an empty icy sequence you need to feed with the {@link #imageReceived(TaggedImage)} method so the
 * sequence will be updated with those images.
 * 
 * @author Irsath Nguyen
 * @author Stephane Dallongeville
 */
public class AcquisitionResult
{
    private final Map<Integer, Sequence> sequences;
    private final Map<Sequence, Integer> planeIndexes;
    private final SequenceSettings settings;
    private final JSONObject summaryMetadata;
    private final boolean display;
    private boolean done;

    public AcquisitionResult(SequenceSettings settings, JSONObject summaryMetadata, boolean display)
    {
        this.settings = settings;
        this.summaryMetadata = summaryMetadata;
        this.display = display;
        sequences = new HashMap<Integer, Sequence>();
        planeIndexes = new HashMap<Sequence, Integer>();
        done = false;
    }

    public List<Sequence> getSequences()
    {
        return new ArrayList<Sequence>(sequences.values());
    }

    public void imageReceived(TaggedImage taggedImage) throws JSONException, MMScriptException
    {
        final JSONObject tags = taggedImage.tags;

        final Integer position = Integer.valueOf(MDUtils.getPositionIndex(tags));
        final int frame = MDUtils.getFrameIndex(tags);
        final int ch = MDUtils.getChannelIndex(tags);
        final int slice = MDUtils.getSliceIndex(tags);
        final int planeIndex;

        Sequence seq = sequences.get(position);

        if (seq == null)
        {
            seq = createSequence(tags);
            sequences.put(position, seq);

            if (display)
                Icy.getMainInterface().addSequence(seq);
        }

        IcyBufferedImage image = seq.getImage(frame, slice);

        if (image == null)
        {
            image = createImage();
            seq.setImage(frame, slice, image);
        }

        // set data
        image.setDataXY(ch, taggedImage.pix);

        // get plan index for this sequence
        final Integer value = planeIndexes.get(seq);
        if (value == null)
            planeIndex = 0;
        else
            planeIndex = value.intValue();

        final OMEXMLMetadataImpl metadata = seq.getMetadata();

        // fill metadata
        metadata.setPlaneTheT(NonNegativeInteger.valueOf(String.valueOf(frame)), 0, planeIndex);
        metadata.setPlaneTheZ(NonNegativeInteger.valueOf(String.valueOf(slice)), 0, planeIndex);
        metadata.setPlaneTheC(NonNegativeInteger.valueOf(String.valueOf(ch)), 0, planeIndex);

        Point3D.Double defaultPos;
        try
        {
            defaultPos = StageMover.getXYZ();
        }
        catch (Exception e1)
        {
            defaultPos = new Point3D.Double(0d, 0d, 0d);
        }
        final double xPos = MDUtils.hasXPositionUm(tags) ? MDUtils.getXPositionUm(tags) : defaultPos.x;
        final double yPos = MDUtils.hasYPositionUm(tags) ? MDUtils.getYPositionUm(tags) : defaultPos.y;
        final double zPos = MDUtils.hasZPositionUm(tags) ? MDUtils.getZPositionUm(tags) : defaultPos.z;
        double exposure = 0d;

        // exposure is in second for us
        if (MDUtils.hasExposureMs(tags))
            exposure = MDUtils.getExposureMs(tags) / 1000d;
        else if (ch < settings.channels.size())
            exposure = settings.channels.get(ch).exposure / 1000d;

        if (exposure == 0d)
        {
            try
            {
                exposure = MicroManager.getCore().getExposure() / 1000d;
            }
            catch (Exception e)
            {
                // ignore
            }
        }

        metadata.setPlanePositionX(OMEUtil.getLength(xPos), 0, planeIndex);
        metadata.setPlanePositionY(OMEUtil.getLength(yPos), 0, planeIndex);
        metadata.setPlanePositionZ(OMEUtil.getLength(zPos), 0, planeIndex);
        metadata.setPlaneExposureTime(OMEUtil.getTime(exposure), 0, planeIndex);
        if (MDUtils.hasElapsedTimeMs(tags))
            metadata.setPlaneDeltaT(OMEUtil.getTime(MDUtils.getElapsedTimeMs(tags) / 1000d), 0, planeIndex);

        // pass to next plane index
        planeIndexes.put(seq, Integer.valueOf(planeIndex + 1));
    }

    private IcyBufferedImage createImage() throws JSONException, MMScriptException
    {
        final int width = MDUtils.getWidth(summaryMetadata);
        final int height = MDUtils.getHeight(summaryMetadata);
        final int numChannels = MDUtils.getNumChannels(summaryMetadata);
        final int bpp = MDUtils.getBitDepth(summaryMetadata);

        switch (bpp)
        {
            case 8:
                return new IcyBufferedImage(width, height, numChannels, DataType.UBYTE);
            case 16:
                return new IcyBufferedImage(width, height, numChannels, DataType.USHORT);
            case 32:
                return new IcyBufferedImage(width, height, numChannels, DataType.UINT);
            default:
                throw new MMScriptException("Unknown bit depth");
        }
    }

    private Sequence createSequence(JSONObject metadata) throws JSONException, MMScriptException
    {
        final Calendar calendar = Calendar.getInstance();
        final DateFormat dateFormat = DateFormat.getDateInstance();
        final DateFormat timeFormat = DateFormat.getTimeInstance();

        final Sequence result = new Sequence();

        // set name
        result.setName("Acquisition - " + dateFormat.format(calendar.getTime()) + " "
                + timeFormat.format(calendar.getTime()));

        final OMEXMLMetadataImpl seqMeta = result.getMetadata();

        final double pixelSize = MDUtils.getPixelSizeUm(summaryMetadata);
        final double zStep = MDUtils.getZStepUm(summaryMetadata);
        final double intervalMs = MDUtils.getIntervalMs(summaryMetadata) / 1000d;

        if (pixelSize > 0d)
        {
            result.setPixelSizeX(pixelSize);
            result.setPixelSizeY(pixelSize);
        }
        if (zStep > 0d)
            result.setPixelSizeZ(zStep);
        if (intervalMs > 0d)
            result.setTimeInterval(intervalMs);

        // TODO: improve metadata fill
        seqMeta.setStageLabelX(OMEUtil.getLength(MDUtils.getXPositionUm(metadata)), 0);
        seqMeta.setStageLabelY(OMEUtil.getLength(MDUtils.getYPositionUm(metadata)), 0);
        try
        {
            seqMeta.setStageLabelZ(OMEUtil.getLength(MDUtils.getZPositionUm(metadata)), 0);
        }
        catch (JSONException e)
        {
            // we can ignore missing information here

        }

        // add 1 image to build colormodel
        result.addImage(createImage());

        // initialize colormap
        for (int c = 0; c < result.getSizeC(); c++)
        {
            final ChannelSpec chSpec = (c < settings.channels.size()) ? settings.channels.get(c) : null;
            String channelName = null;

            if (chSpec != null)
            {
                final IcyColorMap colormap = result.getColorModel().getColorMap(c);

                if (colormap != null)
                {
                    colormap.setARGBControlPoint(0, Color.BLACK);
                    colormap.setARGBControlPoint(255, chSpec.color);
                }

                channelName = chSpec.config;
            }

            if (StringUtil.isEmpty(channelName))
                channelName = MDUtils.getChannelName(metadata);

            seqMeta.setChannelName(channelName, 0, c);
        }

        return result;
    }

    public boolean isDone()
    {
        return done;
    }

    public void done()
    {
        done = true;
    }
}