/**
 * 
 */
package plugins.tprovoost.Microscopy.MicroscopeLive;

import icy.gui.component.NumberTextField;
import icy.gui.component.NumberTextField.ValueChangeListener;
import icy.gui.dialog.MessageDialog;
import icy.gui.frame.IcyFrame;
import icy.gui.frame.IcyFrameAdapter;
import icy.gui.frame.IcyFrameEvent;
import icy.gui.util.GuiUtil;
import icy.image.IcyBufferedImage;
import icy.main.Icy;
import icy.painter.Overlay;
import icy.sequence.Sequence;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceListener;
import icy.system.IcyExceptionHandler;

import java.awt.GridLayout;
import java.util.List;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;

import plugins.tprovoost.Microscopy.MicroManager.MicroManager;
import plugins.tprovoost.Microscopy.MicroManager.event.LiveListener;
import plugins.tprovoost.Microscopy.MicroManager.tools.FrameUtils;
import plugins.tprovoost.Microscopy.MicroManager.tools.StageMover;
import plugins.tprovoost.Microscopy.MicroManager.tools.StageMover.StageListener;

public class MicroscopeLive2D implements StageListener, LiveListener, SequenceListener
{
    MicroscopeLivePlugin plugin;
    IcyFrame frame;
    final Sequence liveSequence;
    final PainterCoordinates positionPainter;
    boolean startedHere;

    // GUI
    JButton btn_start;
    JButton btn_stop;
    NumberTextField tfExposure, tfFPS;
    boolean modifyingExposure;

    public MicroscopeLive2D(MicroscopeLivePlugin plugin)
    {
        super();

        this.plugin = plugin;
        frame = null;
        startedHere = false;
        modifyingExposure = false;

        initializeGui();

        // create live sequence
        final List<Sequence> sequences = Icy.getMainInterface().getSequences("Live Mode 2D");
        if (sequences.size() > 0)
            liveSequence = sequences.get(0);
        else
            liveSequence = new Sequence("Live Mode 2D");

        positionPainter = new PainterCoordinates();
        liveSequence.addOverlay(positionPainter);
        liveSequence.addOverlay(new PainterInfoConfig());
        liveSequence.addOverlay(new LiveOverlay(liveSequence));
        liveSequence.addListener(this);

        // add listeners
        StageMover.addListener(MicroscopeLive2D.this);
        MicroManager.addLiveListener(MicroscopeLive2D.this);

        updateState();
    }

    public void initializeGui()
    {
        frame = new IcyFrame("Live 2D Options", true, true, false, true);

        JPanel mainPanel = new JPanel();
        btn_start = FrameUtils.createUIButton("", "playback_play.png", new Runnable()
        {
            @Override
            public void run()
            {
                try
                {
                    startedHere = MicroManager.startLiveMode();
                    updateState();
                }
                catch (Exception e)
                {
                    MessageDialog.showDialog("Unable to start live, please check if an acquisition is running",
                            MessageDialog.ERROR_MESSAGE);
                }
            }
        });
        btn_stop = FrameUtils.createUIButton("", "playback_stop.png", new Runnable()
        {
            @Override
            public void run()
            {
                stop();
            }
        });

        // -------------------
        // OBSERVATION OPTIONS
        // -------------------
        final JPanel panel_observation = GuiUtil.generatePanel("Observation");
        panel_observation.setLayout(new GridLayout(2, 2));

        double exp;

        try
        {
            // retrieve current exposure
            exp = MicroManager.getCore().getExposure();
        }
        catch (Exception e)
        {
            exp = 10;
        }

        tfExposure = new NumberTextField();
        tfExposure.setNumericValue(exp);
        tfExposure.setHorizontalAlignment(SwingConstants.RIGHT);
        tfExposure.setToolTipText("Time of exposure. Zero means continuous acquisition.");
        tfExposure.addValueListener(new ValueChangeListener()
        {
            @Override
            public void valueChanged(double newValue, boolean validate)
            {
                if (modifyingExposure)
                    return;

                if (validate)
                {
                    try
                    {
                        MicroManager.setExposure(newValue);
                    }
                    catch (Exception e)
                    {
                        IcyExceptionHandler.showErrorMessage(e, true);
                    }
                }
            }
        });

        tfFPS = new NumberTextField();
        tfFPS.setNumericValue(1000d / exp);
        tfFPS.setHorizontalAlignment(SwingConstants.RIGHT);
        tfFPS.setToolTipText("Wanted frame per second calculated by (1000 / exposure)");
        tfFPS.addValueListener(new ValueChangeListener()
        {
            @Override
            public void valueChanged(double newValue, boolean validate)
            {
                if (modifyingExposure)
                    return;

                if (validate)
                {
                    try
                    {
                        MicroManager.setExposure(1000 / newValue);
                    }
                    catch (Exception e)
                    {
                        IcyExceptionHandler.showErrorMessage(e, true);
                    }
                }
            }
        });

        panel_observation.add(new JLabel("Exposure (ms):"));
        panel_observation.add(tfExposure);
        panel_observation.add(new JLabel("Frame Per Second :"));
        panel_observation.add(tfFPS);
        // ------------
        // RUN OPTIONS
        // ------------
        JPanel panel_buttons = GuiUtil.generatePanel("Run");
        panel_buttons.setLayout(new BoxLayout(panel_buttons, BoxLayout.X_AXIS));
        panel_buttons.add(Box.createHorizontalGlue());
        panel_buttons.add(btn_start);
        panel_buttons.add(Box.createHorizontalGlue());
        panel_buttons.add(btn_stop);
        panel_buttons.add(Box.createHorizontalGlue());

        // ----------------
        // DISPLAY
        // ----------------
        mainPanel.setLayout(new GridLayout(2, 1));
        mainPanel.add(panel_observation);
        mainPanel.add(panel_buttons);

        frame.addFrameListener(new IcyFrameAdapter()
        {
            @Override
            public void icyFrameClosing(IcyFrameEvent e)
            {
                super.icyFrameClosing(e);

                plugin.shutdown();
            }
        });

        frame.setLayout(new GridLayout(1, 1));
        frame.add(mainPanel);
        frame.pack();
        frame.setVisible(true);
        frame.addToDesktopPane();
        frame.requestFocus();
    }

    public void toFront()
    {
        frame.toFront();
    }

    void stop()
    {
        try
        {
            MicroManager.stopLiveMode();
        }
        catch (Exception ex)
        {
            IcyExceptionHandler.showErrorMessage(ex, true);
        }
    }

    public void shutdown()
    {
        // stop live if we started it
        if (startedHere)
            stop();

        // remove listener
        MicroManager.removeLiveListener(this);
        StageMover.removeListener(this);
        liveSequence.removeListener(this);

        // remove our overlays from live sequence
        for (Overlay overlay : liveSequence.getOverlays())
        {
            if ((overlay instanceof PainterCoordinates) || (overlay instanceof PainterInfoConfig)
                    || (overlay instanceof LiveOverlay))
                liveSequence.removeOverlay(overlay);
        }

        frame.dispose();
    }

    @Override
    public void sequenceChanged(SequenceEvent sequenceEvent)
    {

    }

    @Override
    public void sequenceClosed(Sequence sequence)
    {
        // stop the live mode
        if (startedHere)
            stop();

        // sequence may have lost its overlays
        liveSequence.addOverlay(positionPainter);
        liveSequence.addOverlay(new PainterInfoConfig());
        liveSequence.addOverlay(new LiveOverlay(liveSequence));
    }

    @Override
    public void onStagePositionChanged(String zStage, double z)
    {
        positionPainter.setZ(z);
    }

    @Override
    public void onXYStagePositionChanged(String XYStage, double x, double y)
    {
        positionPainter.setXY(x, y);
    }

    @Override
    public void liveImgReceived(IcyBufferedImage image)
    {
        if (liveSequence.isEmpty() || !liveSequence.isCompatible(image))
        {
            liveSequence.beginUpdate();
            try
            {
                liveSequence.removeAllImages();
                liveSequence.addImage(image);
            }
            finally
            {
                liveSequence.endUpdate();
            }
        }
        else
        {
            final IcyBufferedImage seqImage = liveSequence.getFirstImage();

            seqImage.beginUpdate();
            try
            {
                for (int c = 0; c < image.getSizeC(); c++)
                    seqImage.setDataXY(c, image.getDataXY(c));
            }
            finally
            {
                seqImage.endUpdate();
            }
        }
    }

    public void onExposureChanged(double newExposure)
    {
        modifyingExposure = true;
        try
        {
            tfExposure.setNumericValue(newExposure);
            tfFPS.setNumericValue(1000 / newExposure);
        }
        finally
        {
            modifyingExposure = false;
        }
    }

    @Override
    public void onStagePositionChangedRelative(String zStage, double z)
    {

    }

    @Override
    public void onXYStagePositionChangedRelative(String XYStage, double x, double y)
    {

    }

    @Override
    public void liveStarted()
    {
        updateState();
    }

    @Override
    public void liveStopped()
    {
        updateState();
        startedHere = false;
    }

    void updateState()
    {
        if (MicroManager.isLiveRunning())
        {
            btn_start.setEnabled(false);
            btn_stop.setEnabled(true);

            // make the sequence visible
            if (!Icy.getMainInterface().isOpened(liveSequence))
                Icy.getMainInterface().addSequence(liveSequence);
        }
        else
        {
            btn_start.setEnabled(true);
            btn_stop.setEnabled(false);
        }
    }
}
