package plugins.adufour.protocols;

import icy.file.FileUtil;
import icy.plugin.PluginLoader;
import icy.plugin.abstract_.PluginActionable;
import icy.preferences.PluginsPreferences;
import icy.preferences.XMLPreferences;
import icy.system.thread.ThreadUtil;

import java.awt.Toolkit;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactoryConfigurationError;

import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import plugins.adufour.blocks.util.BlocksML;
import plugins.adufour.blocks.util.BlocksReloadedException;
import plugins.adufour.protocols.gui.MainFrame;
import plugins.adufour.protocols.gui.ProtocolPanel;

public class Protocols extends PluginActionable
{
    // "command" on mac, "ctrl" on others
    public static final int                MENU_SHORTCUT_KEY        = Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();
    
    public static MainFrame                lastFrameInstance;
    
    private static final String            PREF_FOLDER              = "protocolFolder";
    
    public static final String             downloadedProtocolFolder = FileUtil.getApplicationDirectory() + File.separator + "protocols";
    
    private static volatile XMLPreferences preferences              = PluginsPreferences.getPreferences().node(Protocols.class.getName());
    
    public static String getDefaultProtocolFolder()
    {
        return preferences.get(PREF_FOLDER, System.getProperty("user.home"));
    }
    
    public static void setDefaultProtocolFolder(String path)
    {
        preferences.put(PREF_FOLDER, path);
    }
    
    public static MainFrame getLastActiveFrame()
    {
        return lastFrameInstance;
    }
    
    private MainFrame mainFrame;
    
    private boolean   reloading = false;
    
    public boolean isReloading()
    {
        return reloading;
    }
    
    public void setReloading(boolean reloading)
    {
        this.reloading = reloading;
    }
    
    /**
     * Saves the current state of the Protocols interface and restarts it.<br/>
     * This method is useful when plug-ins have been modified (via the plug-in loader) and requires
     * Protocols to restart to take into account the new changes
     * 
     * @param reloadingNode
     *            the node that caused the reload operation (should be reloaded last)
     * @throws TransformerFactoryConfigurationError
     * @throws TransformerException
     */
    public void reload(Document reloadingXML, String reloadingPath)
    {
        // 0) avoid silly situations...
        if (mainFrame == null) return;
        
        reloading = true;
        
        // 1) save the current state with all opened protocols into the preferences
        
        preferences.putInt("Window X", mainFrame.getX());
        preferences.putInt("Window Y", mainFrame.getY());
        
        int counter = 1;
        
        // save protocols to the preferences file one by one
        for (ProtocolPanel protocol : mainFrame.getProtocolPanels())
        {
            try
            {
                File attachedFile = protocol.getFile();
                String xmlProtocol;
                
                if (attachedFile != null && attachedFile.getAbsolutePath().equals(reloadingPath))
                {
                    xmlProtocol = BlocksML.getInstance().toString(reloadingXML);
                }
                else
                {
                    xmlProtocol = BlocksML.getInstance().toString(protocol.getWorkFlow());
                }
                
                XMLPreferences node = preferences.node("Protocol #" + counter++);
                node.putBoolean("dirty", protocol.isDirty());
                node.put("xml", xmlProtocol);
                if (attachedFile != null) node.put("fileName", attachedFile.getAbsolutePath());
            }
            catch (TransformerException e)
            {
                System.err.println("Warning: couldn't store protocol " + protocol.getName() + ":");
                e.printStackTrace();
                System.err.println("---");
            }
        }
        
        // 2) close the current Protocols instance
        
        mainFrame.close();
        
        ThreadUtil.invokeLater(new Runnable()
        {
            public void run()
            {
                // 3) launch a new instance of the Protocols plug-in
                try
                {
                    ((PluginActionable) PluginLoader.getPluginClass(Protocols.class.getName()).newInstance()).run();
                }
                catch (Exception e)
                {
                    e.printStackTrace();
                }
            }
        });
    }
    
    @Override
    public void run()
    {
        lastFrameInstance = mainFrame = new MainFrame(this);
        
        DocumentBuilder builder;
        try
        {
            builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
            
            // Reload potential temporary protocols from the preferences
            
            int x = preferences.getInt("Window X", mainFrame.getX());
            int y = preferences.getInt("Window Y", mainFrame.getY());
            mainFrame.setLocation(x, y);
            
            ArrayList<XMLPreferences> protocols = preferences.getChildren();
            
            if (protocols.size() == 0)
            {
                mainFrame.addProtocolPane(new ProtocolPanel(mainFrame));
            }
            else for (XMLPreferences node : preferences.getChildren())
            {
                ProtocolPanel panel = new ProtocolPanel(mainFrame);
                String fileName = node.get("fileName", null);
                if (fileName != null) panel.setFile(new File(fileName));
                mainFrame.addProtocolPane(panel);
                
                Document xml = builder.parse(new InputSource(new StringReader(node.get("xml", null))));
                
                try
                {
                    panel.loadWorkFlow(xml, node.getBoolean("dirty", false));
                    
                    // if the protocol loads correctly, remove it from the preferences
                    preferences.remove(node.name());
                }
                catch (BlocksReloadedException e)
                {
                    reload(xml, panel.getFile().getAbsolutePath());
                    return;
                }
            }
            
        }
        catch (ParserConfigurationException e)
        {
            e.printStackTrace();
        }
        catch (SAXException e)
        {
            e.printStackTrace();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
        
        mainFrame.addToDesktopPane();
        mainFrame.setVisible(true);
        
    }
}
